from __future__ import annotations

import traceback
from dataclasses import dataclass
from pathlib import Path
from typing import Generator, Union
from typing import TYPE_CHECKING

import axipy
from .observer import MergePolygonsObserver
from .utils import _run_in_gui_decor

if TYPE_CHECKING:
    from .__init__ import MergePolygons

PolygonLikeTypes = (axipy.Polygon, axipy.MultiPolygon)
PolygonLike = Union[axipy.Polygon, axipy.MultiPolygon]


class ProgressWithStepMixIn:
    @dataclass
    class ProgressState:
        progress_max: int = 1000
        one_step: int = 1
        step_needed: bool = False
        n: int = 0

    def __init__(self, task: axipy.Task) -> None:
        self.__progress_state = self.ProgressState()
        self.__task = task

    def init_progress_with_step(self, task: axipy.Task, count: int) -> int:
        ProgressWithStepMixIn.__init__(self, task)

        if count > self.__progress_state.progress_max:
            self.__progress_state.one_step = count // self.__progress_state.progress_max
            self.__progress_state.step_needed = True

        if self.__progress_state.step_needed:
            new_max = self.__progress_state.progress_max
        else:
            new_max = count

        return new_max

    def add_value_progress_with_step(self) -> None:
        if self.__progress_state.step_needed:
            self.__progress_state.n += 1
            if self.__progress_state.n % self.__progress_state.one_step == 0:
                self.__task.value += 1
        else:
            self.__task.value += 1


@dataclass
class FeaturePolygon:
    feature: axipy.Feature
    polygon: PolygonLike
    # was_merged: bool = False


class Worker(axipy.Task, ProgressWithStepMixIn):
    class NoGeomsToMerge(Exception):
        ...

    def __init__(
            self,
            plugin: 'MergePolygons',
            inp_table_name: str,
            field_name: str | None,
            out_table_path: Path,
    ) -> None:
        super().__init__(self.do_work)

        self._plugin: 'MergePolygons' = plugin

        self._inp_table_name: str = inp_table_name
        self._field_name: str | None = field_name
        self._out_table_path: Path = out_table_path

        self._new_features_count: int = 0

    def do_work(self) -> None:
        self.title = self._plugin.title
        self.message = self.title

        try:
            self._do_work_impl()
        except self.CanceledException:
            self._notify_canceled()
        except self.NoGeomsToMerge:
            self._notify_finished()
        except Exception as e:
            traceback.print_exc()
            self._notify_error(e)
        else:
            self._notify_finished()

    @_run_in_gui_decor
    def _notify_canceled(self) -> None:
        axipy.Notifications.push(self._plugin.title, "Отмена.", axipy.Notifications.Warning)

    @_run_in_gui_decor
    def _notify_error(self, e) -> None:
        axipy.Notifications.push(self._plugin.title, f"Ошибка: '{e}'.", axipy.Notifications.Critical)

    @_run_in_gui_decor
    def _notify_finished(self) -> None:
        if self._new_features_count > 0:
            message: str = (
                f"Операция успешно завершена. "
                f"Создано новых записей: {self._new_features_count}."
            )
            message_type = axipy.Notifications.Success
        else:
            message: str = (
                "Не найдено геометрий для объединения."
            )
            message_type = axipy.Notifications.Warning
        axipy.Notifications.push(
            self._plugin.title,
            message,
            message_type
        )

    def _do_work_impl(self) -> None:
        table: axipy.Table = axipy.data_manager.find(self._inp_table_name)
        file_path = self.process_one_table(table)
        self.add_on_active_map(file_path)

    @_run_in_gui_decor
    def add_on_active_map(self, file_name: Path) -> None:
        table = axipy.provider_manager.tab.open(str(file_name))
        layer = axipy.Layer.create(table)
        mv: axipy.MapView = MergePolygonsObserver.active_map_view()
        layers = mv.map.layers
        layers.add(layer)
        count = len(layers)
        layers.move(count - 1, 0)

    def _feature_polygon_filter(self, table: axipy.Table) -> Generator[FeaturePolygon]:
        for f in table.items():
            g = f.geometry
            if isinstance(g, PolygonLikeTypes):
                yield FeaturePolygon(f, g)

    def process_one_table(self, table: axipy.Table) -> Path:
        gen = self._feature_polygon_filter(table)

        result = self.merge_features_by_field_and_polygons(list(gen))
        self._new_features_count = len(result)
        if self._new_features_count == 0:
            raise self.NoGeomsToMerge

        features = self.compress_features(result)

        # New schema
        schema = table.schema
        if self._field_name:
            attrs = [schema[self._field_name]]
        else:
            attrs = []
        new_schema = axipy.Schema(*attrs, coordsystem=schema.coordsystem)

        tab_dest = axipy.provider_manager.tab.get_destination(str(self._out_table_path), new_schema)
        tab_dest.export(features)
        return self._out_table_path

    def compress_features(self, feature_polygon_list: list[FeaturePolygon]) -> Generator[axipy.Feature]:
        for fp in feature_polygon_list:
            if self._field_name:
                props = {self._field_name: fp.feature[self._field_name]}
            else:
                props = {}
            yield axipy.Feature(props, geometry=fp.polygon)

    def merge_features_by_field_and_polygons(
            self,
            feature_polygon_list: list[FeaturePolygon]
    ) -> list[FeaturePolygon]:
        was_merged: bool = False  # at least 1 polygon was merged
        count = len(feature_polygon_list)
        task_min, task_max = 0, (count ** 2 - count) / 2

        task_max = self.init_progress_with_step(self, task_max)

        self.range = axipy.Task.Range(task_min, task_max)

        for i in range(count):
            fp1 = feature_polygon_list[i]
            p1 = fp1.polygon
            f1 = fp1.feature
            if self._field_name:
                first_value = f1[self._field_name]
            else:
                first_value = None

            for j in range(i + 1, count):
                self.raise_if_canceled()
                fp2 = feature_polygon_list[j]
                f2 = fp2.feature
                p2 = fp2.polygon

                if self._field_name and f2[self._field_name] != first_value:
                    self.add_value_progress_with_step()
                    continue

                if p1.intersects(p2):
                    fp2.polygon = p1.union(p2)
                    if fp2.polygon is not None:
                        was_merged = True
                    # fp2.was_merged = True
                    # noinspection PyTypeChecker
                    feature_polygon_list[i] = None

                self.add_value_progress_with_step()

        if not was_merged:
            raise self.NoGeomsToMerge

        return [
            fp for fp in feature_polygon_list
            if (
                    fp is not None
                    # and fp.was_merged
            )
        ]
