from __future__ import annotations

from pathlib import Path
from typing import Final

import axipy
from PySide2.QtCore import QSize
from PySide2.QtGui import QIcon
from .dialog import Dialog
from .observer import MergePolygonsObserver

ICON_PATH: Final[Path] = Path(__file__).parent / "documentation" / "images"
ICON_NAME: Final[str] = "merge_polygons.png"

ICON_SIZES: Final[tuple[int, ...]] = (16, 24, 32)


class MergePolygons(axipy.Plugin):

    def __init__(self) -> None:
        self.title: str = self.tr("Объединение полигонов")
        tooltip: str = self.tr("Объединение полигонов с общей границей с возможностью фильтрации по полю.")

        self._observer = MergePolygonsObserver()

        self._button = axipy.ActionButton(
            title=self.title,
            tooltip=tooltip,
            on_click=self.on_click,
            enable_on=self._observer,
            icon=self._create_icon()
        )

        position = axipy.Position(self.tr("Основные"), self.tr("Команды"))
        position.add(self._button)

    def _create_icon(self) -> QIcon:
        icon = QIcon()
        for size in ICON_SIZES:
            icon_path: str = str(ICON_PATH / f"{size}px" / ICON_NAME)
            icon.addFile(icon_path, QSize(size, size))
        return icon

    def on_click(self) -> None:
        Dialog(self).open()

    def unload(self) -> None:
        self._observer.unload()
        self._button.remove()
