import os
import re
import traceback
from itertools import filterfalse
from pathlib import Path
from typing import TYPE_CHECKING, Dict, Iterable, List, Optional, Tuple, Union, cast

import axipy
from PySide2.QtCore import QDir, QObject, Qt, Signal, Slot
from PySide2.QtGui import QIcon, QKeyEvent, QShowEvent
from PySide2.QtWidgets import (
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QHeaderView,
    QMessageBox,
    QProgressBar,
    QPushButton,
    QTableWidgetItem,
    QWidget,
)

from .ui import Ui_Dialog

if TYPE_CHECKING:
    from . import ExportToDatabase

RE_EMPTY = re.compile(r"^\s*$")


class ExportToDbDialog(QDialog, Ui_Dialog):
    signal_pbar_set_max: Signal = Signal(QProgressBar, int)
    signal_pbar_add_value: Signal = Signal(QProgressBar)
    signal_table_pbar_set_text: Signal = Signal(str)

    def __init__(self, plugin: "ExportToDatabase") -> None:
        super().__init__(axipy.view_manager.global_parent)
        self.plugin = plugin
        self.setupUi(self)

        self._task: Optional[axipy.Task] = None

        self._ui_close_original_text = self._ui_close.text()

        self.init_ui()

        self.on_data_sources_updated()
        axipy.data_manager._shadow.dataSourcesUpdated.connect(self.on_data_sources_updated)

        self._postgre_id: str = ""
        self._oracle_id: str = ""
        self._mssql_id: str = ""

    def showEvent(self, event: QShowEvent) -> None:
        super().showEvent(event)
        self.on_data_sources_updated()

    @property
    def _ui_ok(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Ok)

    @property
    def _ui_close(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Close)

    def init_ui(self) -> None:
        self.tb_select_files.setIcon(QIcon.fromTheme("open"))
        self.tb_select_files.clicked.connect(self.on_open_files_clicked)

        self.tb_delete_selected.setIcon(QIcon.fromTheme("delete"))
        self.tb_delete_selected.clicked.connect(self.on_delete_selected_clicked)

        tb = self.table_widget_tables
        tb.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        tb.itemSelectionChanged.connect(self.on_selection_changed)

        self.le_error_file_name.setText(QDir.temp().filePath("export_error.sql"))
        self.tb_error_file_name.setIcon(QIcon.fromTheme("open"))
        self.tb_error_file_name.clicked.connect(self.on_choice_error_file_name_clicked)

        self.le_log_file_name.setText(QDir.temp().filePath("export_log.sql"))
        self.tb_log_file_name.setIcon(QIcon.fromTheme("open"))
        self.tb_log_file_name.clicked.connect(self.on_choice_log_file_name_clicked)

        self.cb_out_database.currentIndexChanged.connect(self.on_cb_out_database_index_changed)

        self.pbar_tables.hide()
        self.pbar_features.hide()

        self._ui_ok.setText(self.plugin.tr("Экспорт"))
        self._ui_ok.setEnabled(False)

        self.buttonBox.accepted.connect(self.accept)
        self.buttonBox.rejected.connect(self.reject)

    def toggle_ui(self, state: bool) -> None:
        widgets: Iterable[QObject] = self.children()
        widgets = filter(lambda x: isinstance(x, QWidget), widgets)
        for widget in widgets:
            if widget == self.buttonBox:
                continue
            getattr(widget, "setEnabled")(state)

        self._ui_ok.setEnabled(state)

    def accept(self) -> None:
        self.export_process()

    def reject(self) -> None:
        if self._task is None:
            super().reject()
        else:
            self.rejected.emit()

    def get_postgre_id(self) -> Optional[str]:
        if self._postgre_id == "":
            try:
                self._postgre_id = axipy.provider_manager.postgre.id
            except Exception:
                pass

        return self._postgre_id

    def get_mssql_id(self) -> Optional[str]:
        if self._mssql_id == "":
            try:
                self._mssql_id = axipy.provider_manager.mssql.id
            except Exception:
                pass

        return self._mssql_id

    def get_oracle_id(self) -> Optional[str]:
        if self._oracle_id == "":
            try:
                self._oracle_id = axipy.provider_manager.oracle.id
            except Exception:
                pass

        return self._oracle_id

    def on_data_sources_updated(self) -> None:
        try:
            prev = self.cb_out_database.currentText()
            self.cb_out_database.blockSignals(True)
            self.cb_out_database.clear()
            names: list = axipy.data_manager._shadow.getDbSourceDestNames()
            for name in names:
                params = axipy.data_manager._shadow.getDbSourceMap(name)
                self.cb_out_database.addItem(name, params)
            self.cb_out_database.blockSignals(False)
            if prev in names:
                self.cb_out_database.setCurrentIndex(names.index(prev))
            else:
                self.on_cb_out_database_index_changed(self.cb_out_database.currentIndex())
            self.check_enabled_export()
        except Exception as ex:
            print(ex)

    def on_cb_out_database_index_changed(self, index: int) -> None:
        prev = self.cb_out_database_schema.currentText()
        self.cb_out_database_schema.clear()
        if index == -1:
            return None

        i = self.cb_out_database.currentIndex()
        map_ = self.cb_out_database.itemData(i)

        list_owners = map_["listOwners"]
        current_schema = map_["currentSchema"]

        self.cb_out_database_schema.blockSignals(True)
        self.cb_out_database_schema.addItems(list_owners)
        if prev in list_owners:
            self.cb_out_database_schema.setCurrentIndex(list_owners.index(prev))
        else:
            self.cb_out_database_schema.setCurrentIndex(list_owners.index(current_schema))
        self.cb_out_database_schema.blockSignals(False)
        return None

    def keyPressEvent(self, event: QKeyEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self.on_delete_selected_clicked()
        else:
            super().keyPressEvent(event)

    def on_open_files_clicked(self) -> None:
        file_names = axipy.open_files_dialog(
            filter_arg="Файлы TAB (*.TAB *.tab *.Tab);;Все файлы (*.*)",
            title="Выбор файлов для экспорта",
        )

        if file_names:
            self.add_opened_files([str(name) for name in file_names])
            self.check_enabled_export()

    def add_opened_files(self, files: List[str]) -> None:
        tb = self.table_widget_tables
        row_count = tb.rowCount()

        def file_in_table(file_arg: str) -> bool:
            for i_arg in range(row_count):
                if tb.item(i_arg, 0).text() == file_arg:
                    return True

            return False

        files = list(filterfalse(file_in_table, files))

        n = len(files)

        tb.setRowCount(row_count + n)
        for i in range(n):
            text = files[i]
            tb.setItem(i + row_count, 0, QTableWidgetItem(text))
            tb.setItem(i + row_count, 1, QTableWidgetItem(Path(text).stem))

    def on_selection_changed(self) -> None:
        self.tb_delete_selected.setEnabled(bool(len(self.table_widget_tables.selectedItems())))

    def check_enabled_export(self) -> None:
        enabled = bool(self.table_widget_tables.rowCount() > 0 and self.cb_out_database.currentIndex() != -1)
        self._ui_ok.setEnabled(enabled)

    def on_delete_selected_clicked(self) -> None:
        tb = self.table_widget_tables
        for range_ in tb.selectedRanges():
            top = range_.topRow()
            count = range_.rowCount()
            for i in range(count):
                tb.removeRow(top)

        self.check_enabled_export()

    def on_choice_error_file_name_clicked(self) -> None:
        file_name = QFileDialog.getSaveFileName(
            self,
            "Сохранение файла ошибок",
            self.le_error_file_name.text(),
            "Файлы ошибок (*.sql)",
        )
        if file_name[0]:
            self.le_error_file_name.setText(file_name[0])

    def on_choice_log_file_name_clicked(self) -> None:
        file_name = QFileDialog.getSaveFileName(
            self, "Сохранение файла лога команд", self.le_log_file_name.text(), "Файлы команд (*.sql)"
        )
        if file_name[0]:
            self.le_log_file_name.setText(file_name[0])

    def export_process(self) -> None:
        self.toggle_ui(False)
        try:
            export_parameters = axipy.ExportParameters()
            export_parameters.geometryAsText = self.cb_geom_as_text.isChecked()
            export_parameters.dropTable = self.cb_drop_table.isChecked()
            export_parameters.createIndex = self.cb_create_index.isChecked()
            export_parameters.mapCatalog = self.cb_register_mapcatalog.isChecked()

            if self.gb_error_file_name.isChecked():
                export_parameters.errorFile = self.le_error_file_name.text()
            if self.gb_log_file_name.isChecked():
                export_parameters.logFile = self.le_log_file_name.text()

            if self.le_geom_attr.text():  # Наименование графического атрибута
                export_parameters.geometryColumnName = self.le_geom_attr.text()
            if self.le_style_attr.text():  # Наименование атрибута с оформлением
                export_parameters.renditonColumnName = self.le_style_attr.text()

            chosen_database_schema = self.cb_out_database_schema.currentText()
            map_: Dict[str, str] = self.cb_out_database.currentData()

            current_schema = map_["currentSchema"]
            if (
                map_["provider"] == self.get_oracle_id()
                and export_parameters.createIndex
                and chosen_database_schema != current_schema
            ):
                QMessageBox.critical(
                    self,
                    "Ошибка",
                    "При экспорте в схему БД oracle отличную от текущей, "
                    "создание пространственного индекса не поддерживается. "
                    f"(Текущая схема = {current_schema})",
                )
                raise Exception("Export canceled.")

            # Получение таблиц и alias
            tb = self.table_widget_tables
            row_count = tb.rowCount()
            tables = {Path(tb.item(row, 0).text()): tb.item(row, 1).text() for row in range(row_count)}

            self.pbar_tables.setValue(0)
            self.pbar_tables.setMaximum(0)
            self.pbar_features.setValue(0)
            self.pbar_features.setMaximum(0)
            self.pbar_tables.show()
            self.pbar_features.show()

            axipy.data_manager._shadow.dataSourcesUpdated.disconnect(self.on_data_sources_updated)

            ph = axipy.Task(
                self.export_task,
            )
            self._task = ph
            ph.finished.connect(self.on_task_finished)
            self.rejected.connect(ph.cancel)

            self.signal_pbar_set_max.connect(self.slot_pbar_set_max)
            self.signal_pbar_add_value.connect(self.slot_pbar_add_value)
            self.signal_table_pbar_set_text.connect(self.slot_pbar_tables_set_text)

        except Exception as ex:
            axipy.Notifications.push(
                self.plugin.title, f"Ошибка при старте экспорта: {ex}", axipy.Notifications.Critical
            )
            self.toggle_ui(True)
        else:
            ph.start(
                row_count,
                tables,
                export_parameters,
                map_,
                chosen_database_schema,
            )
            self._ui_close.setText(self.plugin.tr("Отмена"))

    @staticmethod
    def ensure_opened_table(table_path: Path) -> Optional[axipy.Table]:
        """Поиск уже открытой таблицы"""
        table: Optional[axipy.Table] = None
        all_tables: Iterable[axipy.Table] = filter(
            lambda do_: isinstance(do_, axipy.Table),  # type: ignore [arg-type]
            axipy.data_manager.all_objects,
        )
        for table_ in all_tables:
            prop = table_.properties
            if "tabFile" in prop and Path(prop["tabFile"]) == Path(table_path):
                table = table_
        return table

    def export_task(
        self,
        ph: axipy.Task,
        count: int,
        tables: Dict[Path, str],
        export_parameters: axipy.ExportParameters,
        map_: Dict[str, str],
        chosen_database_schema: str,
    ) -> None:
        original_log, tmp_log = self.init_log("logFile", export_parameters)
        original_log_errors, tmp_log_errors = self.init_log("errorFile", export_parameters)

        is_pbar_max_set = False
        success = 0

        for table_path, table_alias in tables.items():
            try:
                display_name_for_table = table_alias if table_alias else table_path.name
                self.signal_table_pbar_set_text.emit(display_name_for_table)
                self.export_one_table(
                    ph,
                    table_path,
                    table_alias,
                    map_,
                    chosen_database_schema,
                    export_parameters,
                    original_log,
                    original_log_errors,
                )
            except Exception as e:
                traceback.print_exc()
                axipy.Notifications.push(
                    self.plugin.title,
                    f"Файл '{table_path.name}' пропущен. Ошибка: '{e}'.",
                    axipy.Notifications.Warning,
                )
            else:
                success += 1
            finally:
                self.clean_tmp_log(export_parameters.logFile, tmp_log)
                self.clean_tmp_log(export_parameters.errorFile, tmp_log_errors)

                if not is_pbar_max_set:
                    self.signal_pbar_set_max.emit(self.pbar_tables, count)
                is_pbar_max_set = True

                self.signal_pbar_add_value.emit(self.pbar_tables)

                if ph.is_canceled:
                    self.notify_count(count, success)
                    return None

        self.notify_count(count, success)
        return None

    def export_one_table(
        self,
        ph: axipy.Task,
        table_path: Path,
        table_alias: str,
        map_: dict,
        chosen_database_schema: str,
        export_parameters: axipy.ExportParameters,
        original_log: Optional[str],
        original_log_errors: Optional[str],
    ) -> None:
        if RE_EMPTY.match(table_alias):
            table_alias = table_path.stem

        extension: str = table_path.suffix
        result = axipy.provider_manager.find_for_extension(extension)
        if result:
            data_provider: axipy.DataProvider = result[0]
        else:
            raise RuntimeError(f"Can't find provider to open file extension - {extension}")

        source: axipy.Source = data_provider.get_source(filepath=str(table_path))
        source["hidden"] = True
        table: Optional[axipy.Table] = None

        close_needed = False

        try:
            table = self.ensure_opened_table(table_path)
            if table is None:
                table = cast(axipy.Table, source.open())
                close_needed = True

            dest_kwargs = {
                "schema": table.schema,
                "db_name": map_["db_name"],
                "host": map_["host"],
                "user": map_["user"],
                "port": map_["port"],
                "password": "",
                "export_params": export_parameters,
            }

            provider = map_["provider"]

            if provider == self.get_postgre_id():
                dest_kwargs["dataobject"] = f"{chosen_database_schema}.{table_alias}"  # Схема.Имя выходной таблицы
                dest = axipy.provider_manager.postgre.get_destination(**dest_kwargs)
            elif provider == self.get_oracle_id():
                current_schema = map_["currentSchema"]
                if current_schema == chosen_database_schema:
                    dest_kwargs["dataobject"] = table_alias
                else:
                    dest_kwargs["dataobject"] = f"{chosen_database_schema}.{table_alias}"  # Схема.Имя выходной таблицы
                dest = axipy.provider_manager.oracle.get_destination(**dest_kwargs)
            elif provider == self.get_mssql_id():
                dest_kwargs["dataobject"] = f"{chosen_database_schema}.{table_alias}"  # Схема.Имя выходной таблицы
                dest = axipy.provider_manager.mssql.get_destination(**dest_kwargs)
            else:
                raise RuntimeError("Destination provider not found")

            features_count = table.count()
            self.signal_pbar_set_max.emit(self.pbar_features, features_count)

            def feature_callback(_f: axipy.Feature, _row_id: int) -> Union[None, bool]:
                self.signal_pbar_add_value.emit(self.pbar_features)
                if ph.is_canceled:
                    return False
                return None

            dest.export(table.items(), func_callback=feature_callback)
        except Exception as e:
            raise e
        finally:
            if table is not None and close_needed:
                axipy.run_in_gui(table.close)

            if export_parameters.logFile:
                ExportToDbDialog.append_log(original_log)
            if export_parameters.errorFile:
                ExportToDbDialog.append_log(original_log_errors)

    @staticmethod
    def init_log(
        attr: str,
        export_parameters: axipy.ExportParameters,
    ) -> Tuple[Optional[str], Optional[str]]:
        original_log_arg = getattr(export_parameters, attr)
        if original_log_arg:
            # create empty with overwrite
            open(original_log_arg, mode="w", encoding="utf-8").close()
            tmp_log_arg = str(Path(original_log_arg).with_suffix(".tmp"))
            setattr(export_parameters, attr, tmp_log_arg)
            return original_log_arg, tmp_log_arg
        else:
            return None, None

    @staticmethod
    def append_log(path: Optional[str]) -> None:
        if path is None:
            return None

        tmp_path = Path(path).with_suffix(".tmp")
        if not tmp_path.exists():
            return None

        with open(path, mode="a", encoding="utf-8") as x:
            with open(str(tmp_path), mode="r", encoding="utf-8") as y:
                x.writelines(y.readlines())

        return None

    @staticmethod
    def clean_tmp_log(param: Optional[str], file: Optional[str]) -> None:
        if param and file:
            if os.path.exists(file):
                os.remove(file)

    @Slot(QProgressBar, int)
    def slot_pbar_set_max(self, pbar: QProgressBar, max_: int) -> None:
        pbar.setMaximum(max_)

    @Slot(QProgressBar)
    def slot_pbar_add_value(self, pbar: QProgressBar) -> None:
        v = pbar.value()
        pbar.setValue(v + 1)

    @Slot(str)
    def slot_pbar_tables_set_text(self, text: str) -> None:
        self.pbar_tables.setFormat(text)

    @Slot(axipy.Task)
    def on_task_finished(self, ph: axipy.Task) -> None:
        # logging.debug("Task finished")

        ph.finished.disconnect(self.on_task_finished)
        self.rejected.disconnect(ph.cancel)
        self._task = None

        self.signal_pbar_set_max.disconnect(self.slot_pbar_set_max)
        self.signal_pbar_add_value.disconnect(self.slot_pbar_add_value)
        self.signal_table_pbar_set_text.disconnect(self.slot_pbar_tables_set_text)

        self.pbar_tables.hide()
        self.pbar_features.hide()

        axipy.data_manager._shadow.dataSourcesUpdated.connect(self.on_data_sources_updated)

        self._ui_close.setText(self._ui_close_original_text)

        self.toggle_ui(True)

    def notify_count(self, count: int, success: int) -> None:
        axipy.Notifications.push(self.plugin.title, self.plugin.tr(f"Экспортировано таблиц {success}/{count}."))
