from __future__ import annotations

from abc import abstractmethod
from typing import cast, Any, Generator, List

import axipy
from PySide2.QtCore import QAbstractItemModel, QObject, QModelIndex, Qt


# noinspection PyPep8Naming
class TreeItemABC:

    @abstractmethod
    def parent(self) -> TreeItemABC | None:
        ...

    @abstractmethod
    def child(self, number: int) -> TreeItemABC | None:
        ...

    @abstractmethod
    def childNumber(self) -> int:
        ...


# noinspection PyPep8Naming
class TreeItem(TreeItemABC):
    def __init__(self, data: str, parentItem: TreeItem | None = None, geometry: axipy.Geometry | None = None) -> None:
        # parentItem is optional because of the root item.
        self._data: str = data
        self.__geometry: axipy.Geometry | None = geometry

        self._parent_item: TreeItem | None = parentItem
        self.__child_items: list[TreeItem] = []
        if parentItem:
            parentItem.appendChild(self)
        self.__original_index: int = self.childNumber()

    # @staticmethod
    # def clone_child_items(items: list[TreeItem]) -> list[TreeItem]:
    #     ...
    #
    # def __copy__(self) -> TreeItem:
    #     new_data = self._data
    #     new_geometry = self.__geometry.clone() if self.__geometry else None
    #     # new_child_items = deepcopy(self.__child_items)
    #     inst = TreeItem(new_data, new_geometry)
    #     # inst._set_child_items(new_child_items)
    #     return inst

    # def _set_child_items(self, child_items: list[TreeItem]) -> None:
    #     self.__child_items = child_items

    def check_if_moved(self) -> bool:
        return self.__original_index != self.childNumber()

    # def __init_original_index(self) -> int:
    #     if self._parent_item is None:
    #         raise RuntimeError
    #
    #     return self.childNumber()

    def childCount(self) -> int:
        return len(self.__child_items)

    def data(self, _column: int = 0) -> str:
        return self._data

    def child(self, number: int) -> TreeItem | None:
        if number < 0 or number >= self.childCount():
            return None
        return self.__child_items[number]

    def parent(self) -> TreeItem | None:
        return self._parent_item

    def childItems(self) -> tuple[TreeItem, ...]:
        return tuple(self.__child_items)

    def childNumber(self) -> int:
        if self.parent():
            child_items = self.parent().childItems()
            if self in child_items:
                return child_items.index(self)
        return 0

    def removeChildren(self, position: int, count: int) -> bool:
        if position < 0 or position + count > len(self.__child_items):
            return False

        del self.__child_items[position:position + count]

        return True

    # def insertChildren(self, position: int, count: int, _columns: int) -> bool:
    #     if position < 0 or position > len(self.__child_items):
    #         return False
    #
    #     for row in range(count):
    #         item = TreeItem("", None, parentItem=self)
    #         self.__child_items.insert(position, item)
    #
    #     return True

    def moveChildren(self, take_index: int, count: int, insert_index: int) -> bool:
        if count != 1:
            return False

        items_list = self.__child_items

        try:

            if take_index > insert_index:
                items_list.insert(insert_index, items_list.pop(take_index))
            else:
                items_list.insert(insert_index, items_list[take_index])
                del items_list[take_index]

        except IndexError:
            return False

        return True

    def __str__(self) -> str:
        return f"{self._data=}"

    def __repr__(self) -> str:
        return f"{self._data=}"

    def geometry(self) -> axipy.Geometry | None:
        return self.__geometry

    def clearChildren(self) -> None:
        self.__child_items.clear()

    def appendChild(self, child: 'TreeItem') -> None:
        self.__child_items.append(child)

        if child.parent() != self:
            child.setParent(self)

    def setParent(self, parent: 'TreeItem') -> None:
        self._parent_item = parent

    def columnCount(self) -> int:
        return 1


# noinspection PyPep8Naming
class TreeModel(QAbstractItemModel):
    # TEXT_LIST_MIME: Final[str] = "application/vnd.text.list"
    # TEXT_LIST_MIME_SEP: Final[str] = ";"
    class UnsavedChangesFound(Exception):
        ...

    def __init__(self, root: TreeItem, parent: QObject = None) -> None:
        super().__init__(parent)

        self.__rootItem: TreeItem = root

    def __getItem(self, index: QModelIndex) -> TreeItem:
        if index.isValid():
            item = index.internalPointer()
            if isinstance(item, TreeItem):
                return item
        return self.__rootItem

    def __rebuild_holes(self, polygon: axipy.Polygon, holes: list[axipy.Polygon]) -> None:
        list_holes: axipy.ListHoles = cast(axipy.ListHoles, polygon.holes)
        for i in range(len(list_holes)):
            list_holes.remove(0)

        for hole in holes:
            list_holes.append(hole.points)

    def __geometry_generator(self, item: TreeItem) -> Generator[axipy.Geometry]:
        is_polygon: bool = False

        g = item.geometry()
        if isinstance(g, axipy.Polygon):
            if item.parent().geometry() is None:
                is_polygon: bool = True
            else:
                is_polygon: bool = False

        if is_polygon:
            holes_from_model = [item.geometry() for item in item.childItems() if item.geometry() is not None]
            self.__rebuild_holes(g, cast(List[axipy.Polygon], holes_from_model))

        if g:
            yield g

        if is_polygon:
            # Filter polygon children, which is holes
            return None

        for child in item.childItems():
            yield from self.__geometry_generator(child)

    def get_geometry_collection(self) -> axipy.GeometryCollection:
        g_col = axipy.GeometryCollection()
        for g in self.__geometry_generator(self.__rootItem):
            g_col.append(g)
        return g_col.try_to_simplified()

    def check_has_unsaved_changes(self) -> bool:
        try:
            self.__check_item_has_unsaved_changes(self.__rootItem)
        except self.UnsavedChangesFound:
            return True

        return False

    def __check_item_has_unsaved_changes(self, item: TreeItem) -> None:
        if item.check_if_moved():
            raise self.UnsavedChangesFound

        for child in item.childItems():
            self.__check_item_has_unsaved_changes(child)

    def updateAll(self) -> None:
        topLeft = self.createIndex(0, 0)
        bottomRight = self.createIndex(self.rowCount(), self.columnCount())
        self.layoutChanged.emit()
        self.dataChanged.emit(topLeft, bottomRight)

    def data(self, index: QModelIndex, role: int = ...) -> str | None:
        if not index.isValid():
            return None

        if role != Qt.DisplayRole:
            return None

        item = cast(TreeItem, index.internalPointer())
        return item.data(index.column())

    def flags(self, index: QModelIndex) -> Qt.ItemFlags:
        if not index.isValid():
            return Qt.ItemFlags(Qt.NoItemFlags)

        # if index.isValid():
        #     return Qt.ItemIsDragEnabled | Qt.ItemIsDropEnabled | super().flags(index)

        return super().flags(index)

    def headerData(self, section: int, orientation: Qt.Orientation, role: int = Qt.DisplayRole) -> Any:
        return None

    def index(self, row: int, column: int, parent: QModelIndex = None) -> QModelIndex:
        if parent is None:
            parent = QModelIndex()

        if parent.isValid() and (parent.column() != 0):
            return QModelIndex()

        parentItem: TreeItem = self.__getItem(parent)
        if not parentItem:
            return QModelIndex()

        childItem: TreeItem = parentItem.child(row)
        if childItem:
            return self.createIndex(row, column, childItem)
        return QModelIndex()

    def parent(self, index: QModelIndex) -> QModelIndex:
        if not index.isValid():
            return QModelIndex()

        childItem: TreeItem = self.__getItem(index)
        parentItem: TreeItem | None = childItem.parent() if childItem else None

        if parentItem == self.__rootItem or not parentItem:
            return QModelIndex()

        return self.createIndex(parentItem.childNumber(), 0, parentItem)

    def rowCount(self, parent: QModelIndex | None = None) -> int:
        if parent is None:
            parent = QModelIndex()

        if parent.isValid() and parent.column() > 0:
            return 0

        parentItem: TreeItem = self.__getItem(parent)

        return parentItem.childCount() if parentItem else 0

    def columnCount(self, parent: QModelIndex = None) -> int:
        return 1

    def removeRows(self, row: int, count: int, parent: QModelIndex = None) -> bool:
        if parent is None:
            parent = QModelIndex()

        parentItem: TreeItem = self.__getItem(parent)
        if not parentItem:
            return False

        self.beginRemoveRows(parent, row, row + count - 1)
        success = parentItem.removeChildren(row, count)
        self.endRemoveRows()

        return success

    # def insertRows(self, row: int, count: int, parent: QModelIndex = None) -> bool:
    #     if parent is None:
    #         parent = QModelIndex()
    #
    #     parentItem: TreeItem = self.__getItem(parent)
    #     if not parentItem:
    #         return False
    #
    #     self.beginInsertRows(parent, row, row + count - 1)
    #     success = parentItem.insertChildren(row, count, self.__rootItem.columnCount())
    #     self.endInsertRows()
    #     print(f"insertRows {success=}")
    #     return success

    def moveRows(
            self,
            sourceParent: QModelIndex,
            sourceRow: int,
            count: int,
            destinationParent: QModelIndex,
            destinationChild: int,
    ) -> bool:
        sourceParentItem: TreeItem = self.__getItem(sourceParent)
        sourceDestinationItem: TreeItem = self.__getItem(destinationParent)

        if not (sourceParentItem and sourceDestinationItem):
            return False

        if sourceParentItem != sourceDestinationItem:
            return False

        # Local move
        sourceLast: int = sourceRow + count - 1
        if destinationChild > sourceRow:
            destinationChild += 1

        self.beginMoveRows(
            sourceParent,
            sourceRow,
            sourceLast,
            destinationParent,
            destinationChild
        )
        # Local move
        result = sourceParentItem.moveChildren(sourceRow, count, destinationChild)
        self.endMoveRows()

        return result

    # def mimeTypes(self) -> list[str]:
    #     return [self.TEXT_LIST_MIME]
    #
    # def mimeData(self, indexes: list[QModelIndex]) -> QMimeData:
    #     data_list = []
    #     for index in indexes:
    #         if index.isValid():
    #             text = index.data(Qt.DisplayRole)
    #             data_list.append(text)
    #
    #     data_string = self.TEXT_LIST_MIME_SEP.join(data_list)
    #     mime_data = QMimeData()
    #     mime_data.setData(self.TEXT_LIST_MIME, data_string.encode())
    #     return mime_data
    #
    # def dropMimeData(self, data: QMimeData, action: Qt.DropAction, row: int, column: int, parent: QModelIndex) -> bool:
    #     print(f"dropMimeData({data.data(self.TEXT_LIST_MIME)=}, {action=}, {row=}, {column=}, {parent=})")
    #
    #     if action == Qt.IgnoreAction:
    #         return False
    #
    #     parentItem = self.__getItem(parent)
    #     if not parentItem:
    #         return False
    #
    #     data = data.data(self.TEXT_LIST_MIME)
    #     data_list = str(data).split(self.TEXT_LIST_MIME_SEP)
    #     for data in data_list:
    #         item = TreeItem(data, None)
    #         # parentItem
    #
    #     return True
    #
    # def supportedDragActions(self) -> Qt.DropActions | Qt.DropAction:
    #     return Qt.CopyAction | Qt.MoveAction
    #
    # def supportedDropActions(self) -> Qt.DropActions | Qt.DropAction:
    #     return Qt.CopyAction | Qt.MoveAction
