import json, os, logging
from typing import Union, Optional

from PySide2.QtCore import Qt, QSize
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import (
    QTreeWidget, QDockWidget, QAbstractItemView, QHeaderView, QTreeWidgetItem, 
    QWidget, QVBoxLayout, QToolBar, QAction
    )

from axipy import Position
from axipy.app import mainwindow
from axipy import (
    Plugin, Position, mainwindow, Layer, provider_manager, WebOpenData, show_message, DlgIcon
    )


class NspdTreeWidget(QTreeWidget):
    '''
    Класс, обеспечивающий древовидное представление доступных для открытия ресурсов
    '''

    def __init__(self, parent: QWidget = None) -> None:
        super().__init__(parent)
        self.__root = None
        self.setHeaderHidden(True)
        self.setRootIsDecorated(True)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.itemDoubleClicked.connect(self.__on_item_dbl_clicked)
        self.header().setStretchLastSection(False)
        self.header().setSectionResizeMode(QHeaderView.ResizeToContents)
        self.setIndentation(10)
        self.setMinimumHeight(1)

    def __on_item_dbl_clicked(self, item: QTreeWidgetItem, column: int) -> None:
        self.__process_open(item.data(0, Qt.UserRole))

    def __process_open(self, data: dict) -> None:
        '''
        Непосредственно открытие данных через соответсвующие провайдеры данных
        '''
        try:
            if data and 'layerType' in data:
                open_data = WebOpenData()
                if 'header' in data:
                    open_data.header = data['header']
                logging.debug(f'Type: {data["layerType"]} URL:{data["url"]}')
                if data['layerType'] == 'wms':
                    wms = provider_manager.wms.open(
                        data['url'], 
                        [str(data['layerId'])],
                        prj='EPSG:3857',
                        alias=data['title'], 
                        extra_data=open_data)
                    if wms:
                        layer = Layer.create(wms)
                        mainwindow.add_layer_interactive(layer)
                elif data['layerType'] == 'tms':
                    tms = provider_manager.tms.open(
                        data["url"],
                        size=(256, 256),
                        minLevel=3,
                        maxLevel=18,
                        alias=data["title"],
                        maxAttempts=3, 
                        extra_data=open_data)
                    if tms:
                        layer = Layer.create(tms)
                        mainwindow.add_layer_interactive(layer)
        except Exception as e:
            url = data['url'] if 'url' in data else None
            show_message(f'Ошибка открытия {url}', 'Заголовок', icon=DlgIcon.ERROR, details=str(e))
            raise


    def __process_folder_wmts(self, lr: dict, base: QTreeWidgetItem, header: dict) -> None:
        source=lr['source']
        if source == 'internal':
            url = r'https://nspd.gov.ru/api/aeggis/v2/' + str(lr['id']) + r'/wmts/{LEVEL}/{ROW}/{COL}.png'
            self.__create_layer_item_tms(lr, base, url, header)
        elif source == 'external':
            # URL доступа задан явно
            option = lr['options']
            ext = option['external']
            u = ext['url']
            # Приводим URL в поддерживаемй провайдером вид
            if '%(z)s' in u:
                u = u.replace('%(z)s', '{LEVEL}')
            if '%(x)s' in u:
                u = u.replace('%(x)s', '{ROW}')
            if '%(y)s' in u:
                u = u.replace('%(y)s', '{COL}')
            self.__create_layer_item_tms(lr, base, u, header)

    def load_file(self, file_name: str, params: dict) -> None:
        '''
        Загрузка в виде дерева доствпных для просмотра слоев
        '''
        header = params['header']
        with open(file_name, 'r', encoding='utf-8') as file:
            base = self.__create_folder_item(params['title'], self)
            data = json.load(file)
            # Данные TMS
            if params['type'] == 'tms':
                for lr in data:
                    self.__process_folder_wmts(lr, base, header)
            # Данные WMS
            elif params['type'] == 'wms':
                layers = {lr['layerId']: lr for lr in data['layers']}
                self.__process_folder_wms(data['tree'], layers, base, header)
   
    def __process_folder_wms(self, fld: dict, layers: dict, base: Union[QTreeWidgetItem, QTreeWidget], header: dict) -> None:
        if 'title' in fld:
            self.__root = self.__create_folder_item(fld['title'], base)
        if fld.get('layers'):
            for lr_id in fld['layers']:
                self.__create_layer_item_wms(layers[lr_id], header)
        if fld.get('folders'):
            for f in fld['folders']:
                self.__process_folder_wms(f, layers, base, header)

    def __create_folder_item(self, title: str, base: Union[QTreeWidgetItem, QTreeWidget]) -> QTreeWidgetItem:
        root = QTreeWidgetItem(base)
        root.setText(0, title)
        root.setExpanded(True)
        return root

    def __create_layer_item_tms(self, lr: dict, base: Union[QTreeWidgetItem, QTreeWidget], url: str, header: dict) -> None:
        '''
        Элемент в дереве, соотвествующий слою TMS (TmsDataProvider)
        '''
        child = QTreeWidgetItem(base)
        title = lr['name']
        child.setText(0, title)
        data = {'title':title, 'url':url, 'layerType':'tms', 'header':header}
        child.setData(0, Qt.UserRole, data)

    def __create_layer_item_wms(self, lr: dict, header: dict) -> None:
        '''
        Элемент в дереве, соотвествующий слою WMS (WmsDataProvider)
        '''
        child = QTreeWidgetItem(self.__root)
        title = lr['title']
        url = f'https://nspd.gov.ru/api/aeggis/v3/{lr["layerId"]}/wms?SERVICE=WMS'
        child.setText(0, title)
        child.setToolTip(0, url)
        data = {'layerId':lr['layerId'], 'title':title, 'url':url, 'layerType':'wms', 'header':header}
        child.setData(0, Qt.UserRole, data)

class RosreestrWms(Plugin):
    '''
    Модуль работы с данными НСПД
    '''
    def load(self) -> None:
        '''
        Загрузка модуля. Добавление кнопки вызова на панель и создание формы.
        '''
        self.__title = self.tr('Карты НСПД')
        self.__icon = QIcon(str(self.plugin_dir / 'favicon.png'))
        self.__button = self.create_action(
            self.__title,
            icon = self.__icon,
            on_click = self.show_widget,
            tooltip = self.tr('Каталог национальной системы пространственных данных'))
        position = Position(self.tr('Основные'), self.tr('Команды'))
        position.add(self.__button)
        self.__dock = None
        show_widget = self.settings.value('ShowWidget', False)
        if (isinstance(show_widget, bool) and show_widget or 
            isinstance(show_widget, str) and show_widget == 'true'):
            self.show_widget()

    def __remove_dock(self) -> None:
        if self.__dock is not None:
            self.__dock.close()
            self.__dock = None

    def unload(self) -> None:
        '''
        Выгрузка модуля. Удаление ресурсов.
        '''
        self.settings.setValue('ShowWidget', self.__dock is not None)
        self.__remove_dock()
        self.__button.remove()

    def __write_file(self, file_name: str, data: bytes) -> None:
        '''
        Сохранение json в файловой системе
        '''
        if os.path.isfile(file_name):
            bak_file_name = file_name + '.BAK'
            if os.path.exists(bak_file_name):
                os.remove(bak_file_name)
            os.rename(file_name, bak_file_name)
        f = open(file_name, "wb")
        f.write(data)
        f.close()

    def __retrieve_file(self, file_name: str, url: str) -> None:
        '''
        Загрузка перечня с сайта и сохранение в виде json файла 
        '''
        import urllib.request as r
        req = r.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
        req.add_header('referer', 'https://nspd.gov.ru/map/')
        try:
            logging.info(f"Получаем: {file_name} по адресу {url}")
            data = r.urlopen(req).read()
            if data:
                self.__write_file(file_name, data)
        except Exception as e:
            logging.critical(f"Ошибка: {e} {type(e).__name__} {e.args}")

    def _json_service_file(self, name: str, url: Optional[str] = None) -> str:
        fn = str( self.plugin_dir / 'services' / name )
        if not os.path.isfile(fn) and url:
            self.__retrieve_file(fn, url)
        return fn
    
    def _json_file(self, name: str) -> str:
        return str( self.plugin_dir / name )
    
    def __clear_dock(self) -> None:
        self.__dock = None

    def __create_tree_widget(self, parent: QWidget) -> NspdTreeWidget:
        w = NspdTreeWidget(parent)
        Services = []
        settings = self._json_file('rosreestr.json')
        logging.debug(f'Load settings file: {settings}')
        # Чтение из файла rosreestr.json списка доступных ресурсов
        with open(settings, 'r', encoding='utf-8') as file:
            Services = json.load(file)
            # Последовательная загрузка ресурсов в соотвествие с их внутренней структурой                
            for srv in Services['services']:
                fn = self._json_service_file(srv['file_name'], srv['url'])
                if os.path.isfile(fn):
                    w.load_file(fn, srv)
                else:
                    logging.critical(f"Ошибка: файл '{fn}' не найден")
        return w
    
    def show_widget(self) -> None:
        '''
        Создание, заполнение и показ древовидной структуры доступных ресурсов.
        '''
        if self.__dock is None:
            self.__dock = QDockWidget(self.__title)
            self.__dock.setAttribute(Qt.WA_DeleteOnClose)
            self.__dock.setObjectName("NspdDock")
            widget = QWidget(self.__dock)
            layout = QVBoxLayout()
            tw = self.__create_tree_widget(widget)
            tw.setWindowTitle(self.__title)
            layout.addWidget(tw)
            widget.setLayout(layout)
            self.__dock.setWidget(widget)
            self.__dock.setAllowedAreas(Qt.LeftDockWidgetArea | Qt.RightDockWidgetArea)
            self.__dock.destroyed.connect(self.__clear_dock)
            mainwindow.add_dock_widget(self.__dock, Qt.RightDockWidgetArea, self.__icon)
        else:
            self.__remove_dock()
