from __future__ import annotations

from dataclasses import dataclass
from pathlib import Path
from typing import Generator, List, Type, Union, cast

import axipy
from PySide2.QtCore import QPoint, Qt
from PySide2.QtGui import QKeyEvent, QMouseEvent, QPainter, QPaintEvent
from PySide2.QtUiTools import QUiLoader
from PySide2.QtWidgets import QMessageBox

ui_file: Path = Path(__file__).parents[1] / "ui" / "RightAngle.ui"

supported_types: tuple[Type[axipy.Geometry], ...] = (
    axipy.Line,
    axipy.LineString,
    axipy.Polygon,
    axipy.GeometryCollection,
)

GEOM_TYPES_SINGLE = Union[
    axipy.Line,
    axipy.LineString,
    axipy.Polygon,
]

supported_types_string: tuple[str, ...] = ("Линия", "Полилиния", "Полигон", "Коллекция")


def show_warning_no_supported_geometry() -> None:
    title = "Ошибка"
    message = "Геометрия не найдена или не поддерживается."
    mbox = QMessageBox(QMessageBox.Information, title, message, QMessageBox.Close)
    mbox.setDetailedText(f"Список поддерживаемых геометрий: '{', '.join(supported_types_string)}'.")
    mbox.exec()


def is_allowed(geometry: axipy.Geometry) -> bool:
    return isinstance(geometry, supported_types)


def to_points(geometry: axipy.Line | axipy.LineString) -> List[axipy.Pnt]:
    if isinstance(geometry, axipy.Line):
        return [geometry.begin, geometry.end]
    elif isinstance(geometry, axipy.LineString):
        return list(geometry.points)
    return []


@dataclass
class CollisionResult:
    segment: axipy.LineString
    segment_index: int
    part_index: int | None = None
    geom_index: int | None = None


@dataclass
class GeometryCollisionResult:
    feature: axipy.Feature
    geometry: GEOM_TYPES_SINGLE
    collision: CollisionResult

    def _add_point_line(self, geom: GEOM_TYPES_SINGLE, point: axipy.Pnt) -> GEOM_TYPES_SINGLE:
        if isinstance(geom, axipy.Line):
            return axipy.LineString([geom.begin, point, geom.end])
        geom.points.insert(self.collision.segment_index + 1, point)
        return geom

    def _add_point_single(self, geom: GEOM_TYPES_SINGLE, point: axipy.Pnt) -> GEOM_TYPES_SINGLE:
        if self.collision.part_index is not None:
            if self.collision.part_index == 0:
                geom.points.insert(self.collision.segment_index + 1, point)
            else:
                geom.holes[self.collision.part_index - 1].insert(self.collision.segment_index + 1, point)
            return geom

        return self._add_point_line(geom, point)

    def add_point(self, point: axipy.Pnt) -> GEOM_TYPES_SINGLE:
        if self.collision.geom_index is not None:
            if isinstance(self.geometry, axipy.GeometryCollection):
                geom = self.geometry[self.collision.geom_index]
                self.geometry[self.collision.geom_index] = self._add_point_single(geom, point)
            return self.geometry
        return self._add_point_single(self.geometry, point)


def find_segment_impl(
    points: list[axipy.Pnt] | axipy.ListPoint, geometry: axipy.Geometry, box: axipy.Polygon
) -> CollisionResult | None:
    for i in range(len(points) - 1):
        segment = axipy.LineString([points[i], points[i + 1]], cs=geometry.coordsystem)
        if box.intersects(segment):
            return CollisionResult(segment, i)
    return None


def find_in_polygon(polygon: axipy.Polygon, bbox: axipy.Polygon) -> CollisionResult | None:
    segment = find_segment_impl(polygon.points, polygon, bbox)
    if segment is not None:
        segment.part_index = 0
        return segment
    for index, hole in enumerate(polygon.holes):
        segment = find_segment_impl(hole, polygon, bbox)
        if segment is not None:
            segment.part_index = index + 1
            return segment
    return None


def find_in_single_geometry(geometry: GEOM_TYPES_SINGLE, bbox: axipy.Polygon) -> CollisionResult | None:
    if isinstance(geometry, axipy.Polygon):
        return find_in_polygon(geometry, bbox)
    return find_segment_impl(to_points(geometry), geometry, bbox)


def find_segment(geometry: GEOM_TYPES_SINGLE | axipy.GeometryCollection, bbox: axipy.Polygon) -> CollisionResult | None:
    if not isinstance(geometry, axipy.GeometryCollection):
        return find_in_single_geometry(geometry, bbox)

    for index, sub_geometry in enumerate(geometry):
        sub_geometry.coordsystem = geometry.coordsystem
        segment = find_in_single_geometry(sub_geometry, bbox)
        if segment is not None:
            segment.geom_index = index
            return segment
    return None


def project(segment: axipy.LineString, p: axipy.Pnt) -> axipy.Pnt:
    p0 = segment.points[0]
    p1 = segment.points[1]
    dx = p1.x - p0.x
    dy = p1.y - p0.y
    len2 = dx * dx + dy * dy
    r = ((p.x - p0.x) * dx + (p.y - p0.y) * dy) / len2

    return axipy.Pnt(p0.x + r * (p1.x - p0.x), p0.y + r * (p1.y - p0.y))


def find_in_table(table: axipy.Table, scene_box: axipy.Rect) -> GeometryCollisionResult | None:
    layer_box: axipy.Polygon = axipy.Polygon.from_rect(scene_box, table.coordsystem)
    features = table.items(bbox=scene_box)
    for feature in features:
        geometry = feature.geometry
        if not is_allowed(geometry):
            continue
        geometry = cast(Union[GEOM_TYPES_SINGLE, axipy.GeometryCollection], geometry)
        collision = find_segment(geometry, layer_box)
        if collision is None:
            continue
        return GeometryCollisionResult(feature, geometry, collision)

    return None


def __layers_generator(map_view: axipy.MapView) -> Generator[axipy.Layer]:
    yield map_view.map.cosmetic
    for layer in map_view.map.layers:
        yield layer


def find_in_map_view(map_view: axipy.MapView, bbox: axipy.Rect) -> GeometryCollisionResult | None:
    for layer in __layers_generator(map_view):
        layer_bbox = bbox.clone()

        data_object = layer.data_object
        if not isinstance(data_object, axipy.Table):
            continue

        if map_view.coordsystem != layer.coordsystem:
            try:
                layer_bbox = axipy.CoordTransformer(map_view.coordsystem, layer.coordsystem).transform(layer_bbox)
            except Exception:
                continue

        result = find_in_table(data_object, layer_bbox)
        if result is not None:
            return result

    return None


class RightAngleTool(axipy.MapTool):
    # enable_on = axipy.ObserverManager.Editable

    def __init__(self) -> None:
        super().__init__()

        self._start_scene_point: axipy.Pnt | None = None
        self._start_scene_box: axipy.Rect | None = None
        self._end_device_point: QPoint | None = None

        self._reset()

    def _reset(self) -> None:
        self._start_scene_point = None
        self._start_scene_box = None
        self._end_device_point = None

    def paintEvent(self, event: QPaintEvent, painter: QPainter) -> None:
        if not self.is_drawing() or self._end_device_point is None:
            return None

        painter.drawPolyline([self.to_device(self._start_scene_point), self.snap_device(self._end_device_point)])

    def mousePressEvent(self, event: QMouseEvent) -> bool:
        if event.button() != Qt.LeftButton:
            return self.PassEvent

        self._reset()
        self._start_scene_point = self.snap(self.to_scene(event.pos()))
        self._start_scene_box = self.get_select_rect(event.pos(), 30)
        return self.BlockEvent

    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        if event.button() != Qt.LeftButton:
            return self.PassEvent

        if not self.is_drawing():
            return self.PassEvent

        end_scene_point = self.snap(self.to_scene(event.pos()))
        view_cs = self.view.coordsystem
        collision = find_in_map_view(self.view, self._start_scene_box)
        if collision is None:
            self._reset()
            show_warning_no_supported_geometry()
            return self.PassEvent

        segment = collision.collision.segment
        if segment.coordsystem != view_cs:
            segment = segment.reproject(view_cs)
        projection_point = project(segment, self._start_scene_point)
        distance_m, dist_angle = axipy.Geometry.distance_by_points(projection_point, end_scene_point, view_cs)
        map_units = self.view.map.distanceUnit
        result = self.show_dialog(axipy.LinearUnits.m.to_unit(map_units, distance_m), map_units.name)
        if result is None:
            self._reset()
            return self.BlockEvent

        distance, segmenting = result
        distance_m = map_units.to_unit(axipy.LinearUnits.m, distance)
        _, angle = axipy.Geometry.distance_by_points(segment.points[0], segment.points[1], view_cs)
        clockwise: bool = ((dist_angle % 360) - (angle % 360)) % 360 < 180
        direction = angle + (90 if clockwise else -90)
        offset_point = axipy.Geometry.point_by_azimuth(projection_point, direction, distance_m, view_cs)

        if segmenting:
            # reproject segment point
            view_cs = self.view.coordsystem
            g_cs = collision.feature.geometry.coordsystem
            if g_cs and g_cs != view_cs:
                segment_point = axipy.CoordTransformer(view_cs, g_cs).transform(projection_point)
            else:
                segment_point = projection_point

            collision.feature.geometry = collision.add_point(segment_point)
            table = self.view.editable_layer.data_object
            if isinstance(table, axipy.Table):
                table.update(collision.feature)

        self.insert(projection_point, offset_point)

        self._reset()
        return self.BlockEvent

    # noinspection PyUnresolvedReferences
    @staticmethod
    def show_dialog(value: float = 1, units_name: str = "m") -> tuple[float, bool] | None:
        # TODO: convert ui file
        dialog = QUiLoader().load(str(ui_file))
        dialog.lblUnits.setText(units_name)
        dialog.leValue.setValue(value)
        dialog.leValue.selectAll()
        dialog.leValue.setFocus()
        if dialog.exec():
            return dialog.leValue.value(), dialog.cbSegmenting.isChecked()
        return None

    def mouseMoveEvent(self, event: QMouseEvent) -> bool:
        if not self.is_drawing():
            return self.PassEvent

        self._end_device_point = event.pos()
        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> bool:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return self.PassEvent

        return super().keyPressEvent(event)

    def insert(self, start: axipy.Pnt, end: axipy.Pnt) -> None:
        line = axipy.Line(start, end, self.view.coordsystem)
        feature = axipy.Feature(geometry=line, style=axipy.LineStyle())
        self.editable_table.insert(feature)

    def is_drawing(self) -> bool:
        return self._start_scene_point is not None

    @property
    def editable_table(self) -> axipy.Table:
        return self.view.editable_layer.data_object
