from typing import List, Optional

from axipy import CoordSystem, Notifications, Pnt, tr, view_manager
from axipy.concurrent import (
    AxipyProgressHandler,
    ProgressGuiFlags,
    ProgressSpecification,
    task_manager,
)
from axipy.da import Feature, Table, data_manager
from axipy.utl import Rect
from PySide2.QtCore import QEvent, QObject, QPoint, QPointF, QRectF, Qt, Signal
from PySide2.QtGui import QMouseEvent, QPainter, QPaintEvent
from PySide2.QtWidgets import (
    QCheckBox,
    QGridLayout,
    QGroupBox,
    QHBoxLayout,
    QRadioButton,
    QVBoxLayout,
    QWidget,
)

from .. import helper
from ..helper import print_
from ..map_tool_extended import PanelMapTool

paste_geometry_observer_key = "PasteGeometryStateManager"


class SetMeta(type):

    def __setattr__(cls: "GeometryStorage", key, value):
        if key == "features":
            super().__setattr__(key, value)
            super().__setattr__("size", len(value))
            cls.signals.state_changed.emit()
        else:
            super().__setattr__(key, value)


class GeometryStorage(metaclass=SetMeta):
    """
    Хранилище геометрии для инструментов Копировать/Вставить геометрию.
    """

    class GeometryStorageSignals(QObject):
        state_changed = Signal()

    signals = GeometryStorageSignals()

    features = None  # type: Optional[List[Feature]]
    size = None  # type: Optional[int]
    cs = None  # type: Optional[CoordSystem]

    @classmethod
    def has_features(cls) -> bool:
        return bool(cls.size)


class CopyGeometry:
    title = None  # type: Optional[str]

    # @time_f
    def on_triggered(self) -> None:

        selection_table = data_manager.selection
        n = selection_table.count()
        items = selection_table.items()
        cs = selection_table.coordsystem

        def copy_in_thread(ph: AxipyProgressHandler, items_arg, n_arg):
            list_ = []
            ph.set_max_progress(n_arg)
            for feature in items_arg:
                list_.append(feature)
                ph.add_progress(1)
                if ph.is_canceled():
                    break

            return list_

        spec = ProgressSpecification(
            window_title=self.title, description=tr("Копирование геометрии"), flags=ProgressGuiFlags.CANCELABLE
        )
        GeometryStorage.features = task_manager.run_and_get(spec, copy_in_thread, items, n)
        GeometryStorage.cs = cs

        self._notify_about_copied(GeometryStorage.size)

    def _notify_about_copied(self, size: int) -> None:
        if size == 0:
            n_type = Notifications.Warning
        else:
            n_type = Notifications.Success
        Notifications.push(self.title, tr(f"Было скопированно геометрий: {size}."), n_type)


class PasteWidget(QWidget):
    def __init__(self) -> None:
        super().__init__()
        vbox = QVBoxLayout()

        self.check_box_dont_copy = QCheckBox(tr("Не копировать атрибуты"))
        vbox.addWidget(self.check_box_dont_copy)

        hbox = QHBoxLayout()
        self.group_box = QGroupBox()
        grid = QGridLayout()
        counter = 1
        for i in range(3):
            for j in range(3):
                grid.addWidget(QRadioButton(f"{counter}"), i, j)
                counter += 1
        self.group_box.setLayout(grid)
        # Кнопка по центру (5)
        grid.itemAtPosition(1, 1).widget().setChecked(True)
        hbox.addWidget(self.group_box)
        hbox.addStretch()
        vbox.addLayout(hbox)

        vbox.addStretch()

        self.setLayout(vbox)


class PasteGeometry(PanelMapTool):

    def __init__(self) -> None:
        super().__init__(custom=True)
        self._envelope = None  # type: Optional[Rect]
        self._device_pnt = None  # type: Optional[QPoint]
        self._draw = False  # type: bool
        self._radio_pnt_scene = None  # type: Optional[Pnt]
        self._current_id = None  # type: Optional[int]
        self._is_active = False  # type: bool
        self._first_click = False  # type: bool
        self._redraw_on_exit = False  # type: bool
        self._paste_pnt = None  # type: Optional[Pnt]

    def load(self) -> None:
        super().load()

        self.connections.append(helper.Connection(view_manager.active_changed, self.active_changed_all))
        self.active_changed_all()
        self._first_click = False

        self.connections.append(helper.Connection(self.view.coordsystem_changed, self._update_envelope))

        tmp_list = self.widget.group_box.findChildren(QRadioButton)
        for i, elem in enumerate(tmp_list, 1):
            elem.clicked.connect(lambda x=i, _=None: setattr(self, "_current_id", x))
        self.widget.group_box.layout().itemAtPosition(1, 1).widget().click()

    def active_changed_all(self) -> None:
        self._is_active = bool(self.view.widget == view_manager.active.widget)

        if self.block():
            return None

        self._first_click = True

        if self._envelope is None:
            self.load_heavy()

    def load_once(self) -> None:
        PasteGeometry.widget = PasteWidget()
        super().load_once()

    def load_heavy(self) -> None:
        self._update_envelope()

    def handleEvent(self, event: QEvent) -> Optional[bool]:

        if event.type() == QEvent.Type.Leave:
            self._redraw_on_exit = True
            self.redraw()

        return self.PassEvent

    def _update_envelope(self) -> None:

        target_cs = self.view.coordsystem
        reproj = GeometryStorage.cs != target_cs

        if reproj:
            print_("reproj")
            spec = ProgressSpecification(tr(f'Подготовка инструмента для "{self}".'), self.title)
            self._envelope = task_manager.run_and_get(spec, self._update_envelope_reproj)
        else:
            print_("no reproj")
            geometries = map(lambda f: f.geometry, GeometryStorage.features)
            spec = ProgressSpecification(tr(f'Подготовка инструмента для "{self}".'), self.title)
            self._envelope = task_manager.run_and_get(spec, helper.quick_envelope, geometries, GeometryStorage.size)

    def _update_envelope_reproj(self, ph) -> Optional[Rect]:
        target_cs = self.view.coordsystem
        geometries = map(lambda f: f.geometry, GeometryStorage.features)
        geometries = map(lambda g: g.reproject(target_cs), geometries)

        return helper.quick_envelope(ph, geometries, GeometryStorage.size)

    def _is_drawable(self) -> bool:
        return all(
            (
                self._is_active,
                self._device_pnt is not None,
                self._envelope is not None,
            )
        )

    @staticmethod
    def _side_points_from_id(r: QRectF, i: int) -> QPointF:
        if i == 1:
            p = r.topLeft()
        elif i == 2:
            p = QPointF(r.center().x(), r.top())
        elif i == 3:
            p = r.topRight()
        elif i == 4:
            p = QPointF(r.left(), r.center().y())
        elif i == 5:
            p = r.center()
        elif i == 6:
            p = QPointF(r.right(), r.center().y())
        elif i == 7:
            p = r.bottomLeft()
        elif i == 8:
            p = QPointF(r.center().x(), r.bottom())
        elif i == 9:
            p = r.bottomRight()
        else:
            raise Exception(f"Unexpected id: '{i}'")
        return p

    @staticmethod
    def _side_points_from_id_inverted_y(r: QRectF, i: int) -> QPointF:
        """
        top и bottom инвертированы, т.к. y экрана (device) отсчет по 'y' идет сверху вниз
        """
        if i == 1:
            p = r.bottomLeft()
        elif i == 2:
            p = QPointF(r.center().x(), r.bottom())
        elif i == 3:
            p = r.bottomRight()
        elif i == 4:
            p = QPointF(r.left(), r.center().y())
        elif i == 5:
            p = r.center()
        elif i == 6:
            p = QPointF(r.right(), r.center().y())
        elif i == 7:
            p = r.topLeft()
        elif i == 8:
            p = QPointF(r.center().x(), r.top())
        elif i == 9:
            p = r.topRight()
        else:
            raise Exception("Unexpected id")
        return p

    def paintEvent(self, event: QPaintEvent, painter: QPainter):

        if self._redraw_on_exit:
            self._redraw_on_exit = False
            return self.PassEvent

        if not self._is_drawable():
            # Очистка карты
            return self.PassEvent

        # Здесь происходит основное изменение
        draw_rect = self.to_device(self._envelope)
        dpoint = self._side_points_from_id(draw_rect, self._current_id)

        if self._current_id != 5:
            p1 = draw_rect.center()
            p2 = dpoint
            dp = p2 - p1
            pnt = QPoint(self._device_pnt.x() - dp.x(), self._device_pnt.y() - dp.y())
            draw_rect.moveCenter(pnt)
        else:
            draw_rect.moveCenter(self._device_pnt)

        painter.setOpacity(1.0)
        painter.setPen(Qt.black)
        painter.drawRect(draw_rect)

        return self.PassEvent

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        self._device_pnt = self.snap_device(event.pos())
        self.redraw()

        return self.PassEvent

    def _displacement_vector(self) -> QPointF:
        """
        Вектор смещения для геометрии относительно центра ограничивающего прямоугольника.
        """
        # Получаем вектор смещения в текущих координатах карты
        draw_rect = self._envelope.to_qt()
        dpoint = self._side_points_from_id_inverted_y(draw_rect, self._current_id)
        p1 = dpoint  # type: QPoint
        p2 = self._paste_pnt.to_qt()
        return p2 - p1

    def mouseReleaseEvent(self, event: QMouseEvent):
        if event.button() == Qt.RightButton:
            return self.BlockEvent

        elif event.button() == Qt.LeftButton:

            if self._first_click:
                self._first_click = False
                return self.PassEvent

            self._paste_pnt = self.snap(self.to_scene(event.pos()))

            task_manager.run_and_get(
                self.spec(tr("Вставка геометрии")), self._paste_operation, helper.ensure_editable_table()
            )

            return self.BlockEvent

        else:
            return self.PassEvent

    def _paste_operation(self, handler: AxipyProgressHandler, editable_table: Table):
        dp = self._displacement_vector()
        copy_attr = not self.widget.check_box_dont_copy.isChecked()
        size = GeometryStorage.size
        handler.set_max_progress(size)
        handler.set_progress(-1)

        def prepare_features(f: Feature):
            handler.add_progress(1)
            handler.raise_if_canceled()
            g = f.geometry
            """Т.к. координатная система (КС) слоя с которого была скопирована геометрия
            может отличаться от КС карты, то для правильного смещения её нужно перепроицировать"""
            target_cs = self.view.coordsystem
            if g.coordsystem != target_cs:
                g = g.reproject(target_cs)
            # Копируем геометрию на новое место
            g = g.shift(dp.x(), dp.y())
            if copy_attr:
                f = Feature(dict(f.items()), geometry=g, style=f.style)
            else:
                f = Feature(geometry=g, style=f.style)
            return f

        features = map(prepare_features, GeometryStorage.features)
        n = helper.insert_to_table(handler, features, editable_table, self.title, default_notify=False)
        Notifications.push(self.title, tr(f"Было вставлено геометрий: {int(n)}."), Notifications.Information)
