from typing import List, Optional, Union, cast, overload

from axipy._internal._decorator import _deprecated_by
from axipy.cpp_gui import (
    BoundingRectWidget,
    PasswordWidget,
    ShadowChooseCoordSystemWidget,
    ShadowGui,
    ShadowStyleButton,
    ShadowStyleEditWidget,
    ShadowWidgetContainer,
)
from axipy.cs.coord_system import CoordSystem
from axipy.cs.unit import LinearUnit, LinearUnits
from axipy.da.geometry import Geometry
from axipy.da.style import Style, StyleGeometryType
from axipy.utl.rect import Rect
from PySide2.QtCore import Signal
from PySide2.QtGui import QPainter, QPaintEvent, QResizeEvent
from PySide2.QtWidgets import QDialog, QPushButton, QWidget

__all__: List[str] = [
    "ChooseCoordSystemDialog",
    "StyleButton",
    "PasswordDialog",
    "BoundingRectDialog",
    "StyleDialog",
    "select_style_dialog",
    "prompt_rect",
    "prompt_coordsystem",
]


class BoundingRectDialog(QDialog):
    """
    Диалог задания параметров прямоугольника. Например, охвата для таблицы или системы
    координат.

    Пример::

        dlg = BoundingRectDialog()
        dlg.unit = 'км'
        dlg.is_square = True
        dlg.rect = Rect(-100, -120, 200, 200)
        if dlg.exec() == QDialog.Accepted:
            print('>>>>', dlg.rect)

    See also:
        :class:`axipy.prompt_rect`
    """

    def __init__(self) -> None:
        """Конструктор класса."""
        super().__init__(ShadowGui.global_parent_static())
        self._shadow = BoundingRectWidget(self)
        self._shadow_container = ShadowWidgetContainer(self._shadow, self)

    @property  # type: ignore[override]
    def rect(self) -> Rect:
        """Устанавливает или возвращает параметры прямоугольника."""
        return Rect.from_qt(self._shadow.resultRect())

    @rect.setter
    def rect(self, r: Rect) -> None:
        self._shadow.setRect(Rect._rect_value_to_qt(r))

    @property
    def unit(self) -> str:
        """Устанавливает или возвращает наименование единиц измерения."""
        return self._shadow.unitName()

    @unit.setter
    def unit(self, v: str) -> None:
        self._shadow.setUnitName(v)

    @property
    def is_square(self) -> bool:
        """
        Устанавливает или возвращает контроль соблюдения равенства ширины и высоты.

        При этом, после ввода какого-либо значения, производится коррекция таким
        образом, чтобы в результате получился квадрат.
        """
        return self._shadow.isSquare()

    @is_square.setter
    def is_square(self, v: bool) -> None:
        self._shadow.setIsSquare(v)


class StyleDialog(QDialog):

    def __init__(self, value: Union[Style, Geometry, StyleGeometryType]) -> None:
        super().__init__(ShadowGui.global_parent_static())
        if isinstance(value, Style):
            self._shadow = ShadowStyleEditWidget.forStyle(value._shadow, self)
        elif isinstance(value, Geometry):
            self._shadow = ShadowStyleEditWidget.forGeometry(value._shadow, self)
        elif isinstance(value, StyleGeometryType):
            self._shadow = ShadowStyleEditWidget.forGeometryStyle(value, self)
        else:
            raise TypeError("Parameter is not supported.")

    def style(self) -> Style:  # type: ignore[override]
        return cast(Style, Style._wrap(ShadowStyleEditWidget.resultStyle(self._shadow)))


@overload
def select_style_dialog(value: StyleGeometryType) -> Optional[Style]: ...


@overload
def select_style_dialog(value: Style) -> Optional[Style]: ...


@overload
def select_style_dialog(value: Geometry) -> Optional[Style]: ...


def select_style_dialog(value: Union[StyleGeometryType, Style, Geometry]) -> Optional[Style]:
    """
    Диалог выбора стиля по геометрии в виде функции. В качестве входных параметров
    рассматривается или стиль или геометрия, для которого этот стиль выбирается.

    Args:
        value: Геометрия, ее тип или стиль. Если задана геометрия, то стиль по умолчанию подставляется как
            стиль нового объекта для данного типа геометрии.

    Return:
        Возвращает выбранный в диалоге стиль.

    По типу геометрического объекта::

        style = axipy.select_style_dialog(StyleGeometryType.Linear)

    Пример задания по геометрии::

        style = axipy.select_style_dialog(Point(3,3))

    Пример задания по стилю::

        style = select_style_dialog(axipy.Style.from_mapinfo('Symbol (35, 0, 12)'))
    """
    dlg = StyleDialog(value)
    if dlg.exec() == QDialog.Accepted:
        return dlg.style()
    return None


class ChooseCoordSystemDialog(QDialog):
    """
    Диалог выбора координатной системы.

        Пример::

            csMercator = CoordSystem.from_prj('10, 104, 7, 0')
            dialog = ChooseCoordSystemDialog(csMercator)
            if dialog.exec() == QDialog.Accepted:
                result_cs = dialog.chosenCoordSystem()
                print("Выбрано:", result_cs.title)

    See also:
        :class:`axipy.prompt_coord_system`
    """

    def __init__(self, coordsystem: Optional[CoordSystem] = None):
        """
        Args:
            coordsystem: Система координат по умолчанию.
                Если не указана, то задается как значение по умолчанию :meth:`axipy.CoordSystem.current`
        """
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        cs = coordsystem if coordsystem is not None else CoordSystem.current()
        self._shadow = ShadowChooseCoordSystemWidget(cs._shadow, self)

    # noinspection PyPep8Naming
    def chosenCoordSystem(self) -> CoordSystem:
        """Возвращает выбранную в диалоге систему координат."""
        # cast is ensured by the code logic.
        return cast(CoordSystem, CoordSystem._wrap(self._shadow.chosenCoordSystem()))


class StyleButton(QPushButton):
    """
    Кнопка, отображающая стиль и позволяющая его менять.

    Пример добавления кнопки на диалог::

        from PySide2.QtWidgets import QDialog
        from axipy import Style, StyleButton

        style = Style.from_mapinfo("Pen (1, 2, 8421504)  Brush (2, 255, 0)")


        class Dialog(QDialog):
            def __init__(self, parent=None):
                super().__init__(parent)
                self.pb = StyleButton(style, parent=self)
                self.pb.style_changed.connect(self.style_result)
                self.pb.setGeometry(100, 100, 100, 50)

            def style_result(self):
                print("Стиль изменен", self.pb.style)


        dialog = Dialog()
        dialog.open()
    """

    __styleChanged = Signal()

    def __init__(self, style: Optional[Style] = None, parent: Optional[QWidget] = None) -> None:
        """
        Args:
            style: Стиль по умолчанию.
            parent: Родительский виджет.

        """
        # Signature compatibility check for QPushButton.
        if isinstance(style, QWidget):
            style, parent = None, style
        super().__init__(parent if parent is not None else ShadowGui.global_parent_static())
        self.__shadow = ShadowStyleButton(self, style._shadow if style is not None else None)
        cast(Signal, self.__shadow.styleChanged).connect(self.__styleChanged)

    @property  # type: ignore[override]
    def style(self) -> Optional[Style]:
        """Устанавливает или возвращает стиль кнопки."""
        return Style._wrap(ShadowStyleButton.get_style(self.__shadow))

    @style.setter
    def style(self, v: Style) -> None:
        # TODO: Optional[Style]
        if v is not None:
            self.__shadow.set_style(v._shadow)

    @property
    def style_changed(self) -> Signal:
        """Сигнал смены стиля."""
        return self.__styleChanged

    def paintEvent(self, event: QPaintEvent) -> None:
        super().paintEvent(event)
        painter = QPainter(self)
        painter.drawPixmap(2, 2, self.__shadow.pixmap())

    def resizeEvent(self, event: QResizeEvent) -> None:
        super().resizeEvent(event)
        self.__shadow.reinitPixmap()


class PasswordDialog(QDialog):
    """
    Диалог задания или корректировки данных аутентификации пользователя.

    Пример::

        dialog = PasswordDialog()
        dialog.address = 'proxy.server'
        dialog.user_name = 'user'
        dialog.password = 'pass'
        if dialog.exec() == QDialog.Accepted:
            print(dialog.user_name, dialog.password)
    """

    def __init__(self) -> None:
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        self._shadow = PasswordWidget(self)

    @property
    def address(self) -> str:
        """
        Устанавливает или возвращает описание ресурса.

        Если значение указано, то в диалоговом окне оно будет присутствовать под именем
        'Адрес'.
        """
        return self._shadow.addressName()

    @address.setter
    def address(self, v: str) -> None:
        self._shadow.setAddressName(v)

    @property
    def user_name(self) -> str:
        """Устанавливает или возвращает имя пользователя."""
        return self._shadow.userName()

    @user_name.setter
    def user_name(self, v: str) -> None:
        self._shadow.setUserName(v)

    @property
    def password(self) -> str:
        """Устанавливает или возвращает пароль."""
        return self._shadow.password()

    @password.setter
    def password(self, v: str) -> None:
        self._shadow.setPassword(v)


def prompt_rect(
    rect: Rect = Rect(-10000000, -10000000, 10000000, 10000000),
    unit: Union[LinearUnit, str] = LinearUnits.m,
) -> Optional[Rect]:
    """
    Диалог выбора параметров прямоугольника.

    Args:
        rect: Значение по умолчанию.
        unit: Единицы измерения при отображении.

    See also:
        :class:`axipy.BoundingRectDialog`
    """
    if isinstance(unit, LinearUnit):
        unit = unit.localized_name

    dlg = BoundingRectDialog()
    dlg.rect = rect
    dlg.unit = unit
    if dlg.exec() == QDialog.Accepted:
        return dlg.rect
    return None


def prompt_coordsystem(cs: Optional[CoordSystem] = None) -> Optional[CoordSystem]:
    """
    Диалог выбора координатной системы.

    Args:
        cs: Система координат.

    See also:
        :class:`axipy.ChooseCoordSystemDialog`
    """
    dlg = ChooseCoordSystemDialog(cs)
    if dlg.exec() == QDialog.Accepted:
        return dlg.chosenCoordSystem()
    return None


def _apply_deprecated() -> None:
    # Using getattr to hide deprecated objects from axipy namespace on IDE inspections
    getattr(__all__, "extend")(("StyledButton",))

    class StyledButton(QPushButton):
        """
        Warning:
            .. deprecated::
        """

        styleChanged = Signal()

        @_deprecated_by("axipy.StyleButton")
        def __init__(self, style: Optional[Style] = None, parent: Optional[QWidget] = None) -> None:
            """
            Warning:
                .. deprecated::
            """
            super().__init__(parent if parent is not None else ShadowGui.global_parent_static())
            self._shadow = ShadowStyleButton(self, style._shadow if style is not None else None)
            cast(Signal, self._shadow.styleChanged).connect(self.styleChanged)

        def style(self) -> Optional[Style]:  # type: ignore[override]
            """
            Warning:
                .. deprecated::
            """
            return Style._wrap(ShadowStyleButton.get_style(self._shadow))

        def paintEvent(self, event: QPaintEvent) -> None:
            super().paintEvent(event)
            painter = QPainter(self)
            painter.drawPixmap(2, 2, self._shadow.pixmap())

        def resizeEvent(self, event: QResizeEvent) -> None:
            super().resizeEvent(event)
            self._shadow.reinitPixmap()

    globals().update(
        StyledButton=StyledButton,
    )


_apply_deprecated()
