import inspect
import sys
import threading
import traceback
from abc import abstractmethod
from enum import Enum, auto
from typing import TYPE_CHECKING, Any, Callable, Dict, List, NamedTuple, Optional, Tuple

import axipy
import PySide2.QtCore
import shiboken2
from axipy._internal._utils import _SignalBlocker
from axipy.cpp_core_core import ProgressHandler, ShadowProgressHandler, ShadowTask
from PySide2.QtCore import QEventLoop, QObject, Signal, Slot

if TYPE_CHECKING:
    from PySide2.QtWidgets import QDialog

__all__: List[str] = [
    "Task",
    "DialogTask",
]


class Task(QObject):
    """
    Пользовательская задача. Если выполнение функции в программе занимает длительное
    время, графический интерфейс Аксиомы будет неотзывчивым в течении этого времени.
    Чтобы длительная задача выполнялось в отдельном потоке и не нарушала работу
    интерфейса Аксиомы, такую функцию следует поместить в задачу, используя класс
    :class:`axipy.Task`.

    Задачи (объекты класса :class:`axipy.Task`) добавляются в менеджер :attr:`axipy.task_manager`
    при создании конструктором, и удаляются при
    завершении задачи (успешно или с ошибкой). Еще не запущенную задачу, можно удалить из менеджера, вызвав метод
    :attr:`axipy.Task.cancel`.
    Одну задачу можно запустить только один раз, завершенную задачу нельзя запустить повторно.

    See also:
        :meth:`axipy.run_in_gui`

    See also:
        :ref:`to_concurrent`
    """

    class Status(Enum):
        """Состояние задачи."""

        IDLE = auto()
        """В ожидании."""
        RUNNING = auto()
        """Запущена."""
        SUCCESS = auto()
        """Выполнена успешно."""
        CANCELED = auto()
        """Отменена."""
        ERROR = auto()
        """Завершилась с ошибкой."""

    class CanceledException(Exception):
        """Исключение при отмене задачи."""

        def __init__(self, message: Optional[str] = None) -> None:
            if message is None:
                message = axipy.tr("Задача была отменена")
            super().__init__(message)

    class Range(NamedTuple):
        """Диапазон прогресса задачи."""

        min: float
        """Минимальное значение прогресса задачи."""
        max: float
        """Максимальное значение прогресса задачи."""

    __FINISHED_STATES = (
        Status.SUCCESS,
        Status.CANCELED,
        Status.ERROR,
    )

    def __init__(
        self,
        function: Callable,
        name: str = "",
        description: str = "",
        title: str = "",
        message: str = "",
    ) -> None:
        """
        Создает экземпляр класса.

        Args:
            function: Функция.
            name: Имя.
            description: Описание.
            title: Отображаемое имя.
            message: Отображаемое описание.
        """
        super().__init__(parent=axipy.task_manager._task_manager_object)

        self.__function: Callable = function
        self.__self_needed: bool = self.__is_self_needed(self.__function)

        # Will be initialised on task start, to save memory
        self.__args: Tuple = None  # type: ignore[assignment]
        self.__kwargs: Dict = None  # type: ignore[assignment]

        self.__name: str = name
        self.__description: str = description
        self.__title: str = title
        self.__message: str = message

        self.__id = axipy.task_manager._current_id
        axipy.task_manager._current_id += 1

        self.__value: float = 0.0
        self.__min: float = 0.0
        self.__max: float = 100.0
        self.__range: "Task.Range" = self.Range(self.__min, self.__max)

        self.__is_canceled: bool = False
        self.__status: "Task.Status" = self.Status.IDLE
        self.__result: Any = None

        if not isinstance(self, DialogTask):
            self.range_changed.connect(self.__fix_value_on_range_changed)

        axipy.task_manager.added.emit(self)

    @staticmethod
    def __is_self_needed(function: Callable) -> bool:
        if inspect.ismethod(function) and hasattr(function, "__self__"):
            if isinstance(function.__self__, Task):
                return False
            else:
                return True
        else:
            return True

    def __new__(cls, *args: Any, **kwargs: Any) -> "Task":
        cls.__started: Signal = Signal()
        cls.__finished: Signal = Signal(Task)
        cls.__value_changed = Signal(float)

        if not issubclass(cls, DialogTask):
            cls.__range_changed = Signal(cls.Range)
            cls.__title_changed = Signal(str)
            cls.__message_changed = Signal(str)

        # noinspection PyTypeChecker
        # noinspection PyArgumentList
        return QObject.__new__(cls, *args, **kwargs)

    def __repr__(self) -> str:
        return f"<axipy.{self.__class__.__name__} name={self.name}>"

    def __str__(self) -> str:
        return self.__repr__()

    @property
    def name(self) -> str:
        """Возвращает имя задачи."""
        return self.__name

    @property
    def description(self) -> str:
        """Возвращает описание задачи."""
        return self.__description

    @property
    def title(self) -> str:
        """Устанавливает или возвращает отображаемое имя задачи."""
        return self.__title

    @title.setter
    def title(self, value: str) -> None:
        self.__title = value
        self.__title_changed.emit(value)

    @property
    def title_changed(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при изменении отображаемого имени задачи.

        :rtype: Signal[str]
        """
        return self.__title_changed

    @property
    def message(self) -> str:
        """Устанавливает или возвращает отображаемое описание задачи."""
        return self.__message

    @message.setter
    def message(self, value: str) -> None:
        self.__message = value
        self.__message_changed.emit(value)

    @property
    def message_changed(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при изменении отображаемого описании задачи.

        :rtype: Signal[str]
        """
        return self.__message_changed

    @property
    def value(self) -> float:
        """Устанавливает или возвращает текущее значение прогресса задачи."""
        return self.__value

    @value.setter
    def value(self, value: float) -> None:
        self.__value = value
        self.value_changed.emit(self.__value)

    @Slot(Range)
    def __fix_value_on_range_changed(self, range_: Range) -> None:
        actual_min = min(range_.min, range_.max)
        actual_max = max(range_.min, range_.max)

        if self.value < actual_min:
            self.value = actual_min
        elif self.value > range_.max:
            self.value = actual_max

    @property
    def value_changed(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при изменении текущего значения прогресса задачи.

        :rtype: Signal[str]
        """
        return self.__value_changed

    @property
    def min(self) -> float:
        """Устанавливает или возвращает минимальное значение диапазона прогресса
        задачи."""
        return self.__min

    @min.setter
    def min(self, value: float) -> None:
        self.__min = value

    @property
    def max(self) -> float:
        """Устанавливает или возвращает максимальное значение диапазона прогресса
        задачи."""
        return self.__max

    @max.setter
    def max(self, value: float) -> None:
        self.__max = value

    @property
    def range(self) -> "Range":
        """Устанавливает или возвращает диапазон прогресса задачи."""
        return self.__range

    @range.setter
    def range(self, value: "Range") -> None:
        self.__range = value
        self.__range_changed.emit(self.__range)

    @property
    def range_changed(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при изменении диапазона прогресса задачи.

        :rtype: Signal[axipy.Task.Range]
        """
        return self.__range_changed

    @property
    def thread_id(self) -> Optional[int]:
        """
        Возвращает идентификатор потока, в котором выполняется задача.

        Когда задача не выполняется, возвращает None.
        """
        if self.status == self.Status.RUNNING:
            return threading.get_native_id()
        else:
            return None

    @property
    def is_canceled(self) -> bool:
        """
        Возвращает True, если задача была отменена.

        Чтобы отменить задачу (назначить True для свойства :attr:`axipy.Task.is_canceled`),
        нужно вызвать :meth:`axipy.Task.cancel`.
        """
        return self.__is_canceled

    @property
    def status(self) -> "Status":
        """Возвращает текущее состояние задачи."""
        return self.__status

    @property
    def result(self) -> Any:
        """
        Возвращает результат задачи. (Возвращаемое значение пользовательской функции.)
        Если во время выполнения задачи, было вызвано исключение, то при получении
        результата, исключение будет вызвано повторно (За исключением пользовательской
        отмены :class:`axipy.Task.CanceledException`).

        Raises:
            Exception: Если во время выполнения задачи, было вызвано исключение.
        """

        if self.status == self.Status.ERROR:
            exc_inst = self.__result
            raise exc_inst

        return self.__result

    @property
    def id(self) -> int:
        """
        Возвращает идентификатор задачи.

        Идентификатором задачи является её порядковый номер в менеджере задач
        :attr:`axipy.task_manager`.
        """
        return self.__id

    @property
    def started(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при старте задачи.

        :rtype: Signal[]
        """
        return self.__started

    @property
    def finished(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при завершении задачи. Передает ссылку на
        экземпляр задачи, для получения результата, статуса и других атрибутов задачи.

        :rtype: Signal[:class:`axipy.Task`]
        """
        return self.__finished

    def __run_internally(self) -> None:
        # Начинает выполнение в отдельном потоке
        try:
            self.__status = self.Status.RUNNING
            self.started.emit()

            if self.__self_needed:
                result = self.__function(self, *self.__args, **self.__kwargs)
            else:
                result = self.__function(*self.__args, **self.__kwargs)

        except self.CanceledException:
            self.__status = self.Status.CANCELED
        except (Exception,):
            self.__result = sys.exc_info()[1]
            self.__status = self.Status.ERROR
        else:
            # noinspection PyUnusedLocal
            # locker = QWriteLocker(self.__lock)
            self.__result = result
            if self.is_canceled:
                self.__status = self.Status.CANCELED
            else:
                self.__status = self.Status.SUCCESS
            # locker.unlock()
        finally:
            self.finished.emit(self)
            self.deleteLater()

    def __init_task_args(self, *args: Any, **kwargs: Any) -> None:
        if self.status != self.Status.IDLE:
            raise RuntimeError("Can't start the same task more than once.")

        self.__args = args
        self.__kwargs = kwargs

    def start(self, *args: Any, **kwargs: Any) -> None:
        """
        Запускает задачу без ожидания результата. Чтобы получить результат, необходимо
        подписаться на сигнал :attr:`axipy.Task.finished` и обратиться к свойству
        :attr:`axipy.Task.result`.

        Args:
            *args: Параметры, передаваемые в пользовательскую функцию.
            **kwargs: Именованные параметры, передаваемые в пользовательскую функцию.
        """
        self.__init_task_args(*args, **kwargs)

        axipy.task_manager._start_runnable_task(self)

    def run_and_get(self, *args: Any, **kwargs: Any) -> Any:
        """
        Запускает задачу и ждет результат. Останавливается лишь выполнение кода python,
        цикл обработки событий Qt продолжает работу. (Сигналы и события продолжают
        обрабатываться.) Возвращает результат задачи. (Возвращаемое значение
        пользовательской функции.) Если во время выполнения задачи, было вызвано
        исключение, то при получении результата, исключение будет вызвано повторно (За
        исключением пользовательской отмены :class:`axipy.Task.CanceledException`).

        Args:
            *args: Параметры, передаваемые в пользовательскую функцию.
            **kwargs: Именованные параметры, передаваемые в пользовательскую функцию.

        Returns:
            Результат выполнения задачи. (Возвращаемое значение пользовательской функции.)

        Raises:
            Exception: Если во время выполнения задачи, было вызвано исключение.
        """
        self.__init_task_args(*args, **kwargs)

        loop = QEventLoop()
        self.finished.connect(loop.quit)
        axipy.task_manager._start_runnable_task(self)
        loop.exec_()

        return self.result

    @Slot()
    def cancel(self) -> None:
        """
        Уведомляет задачу об отмене.

        Устанавливает свойство :attr:`axipy.Task.is_canceled` на True и меняет
        состояние задачи :attr:`axipy.Task.status` на :attr:`axipy.Task.Status.CANCELED`.
        Вызов метода при статусе :attr:`axipy.Task.Status.IDLE` также удалит задачу из :attr:`axipy.task_manager`.
        Метод является слотом, Slot() (:class:`PySide2.QtCore.Slot`).
        """
        self.__is_canceled = True

        if self.status == self.Status.IDLE:
            self.__status = self.Status.CANCELED
            self.setParent(None)
            self.deleteLater()
        else:
            self.__status = self.Status.CANCELED

    def raise_if_canceled(self, message: Optional[str] = None) -> None:
        """
        Вызывает исключение об отмене задачи :class:`axipy.Task.CanceledException`.

        Args:
            message: Сообщение, передаваемое в исключение.
        """
        if self.is_canceled:
            raise self.CanceledException(message)


class DialogTask(Task):
    """
    Пользовательская задача с диалогом. Наследуется от класса :class:`axipy.Task`.
    Дополняет класс :class:`axipy.Task` диалогом, показывающим прогресс задачи.

    See also:
        :meth:`axipy.run_in_gui`

    See also:
        :ref:`to_concurrent`
    """

    if TYPE_CHECKING:
        # sphinx compatibility
        Range = Task.Range
        Status = Task.Status
        CanceledException = Task.CanceledException

    class __ShadowTaskWrapper(ShadowTask):

        def __init__(self, dt: "DialogTask") -> None:
            super().__init__()
            self.dt = dt

        def run_internally(self, *args: Any, **kwargs: Any) -> None:
            try:
                # Name mangling is necessary, to hide it and remove code duplication
                self.dt._Task__run_internally()  # type: ignore[attr-defined]
            except (Exception,):
                traceback.print_exc()
            finally:
                del self.dt

    def __init__(
        self,
        function: Callable,
        name: str = "",
        description: str = "",
        title: str = "",
        message: str = "",
        cancelable: bool = False,
        delayed: bool = True,
    ) -> None:
        """
        Создает экземпляр класса.

        Args:
            function: Функция.
            name: Имя.
            description: Описание.
            title: Отображаемое имя.
            message: Отображаемое описание.
            cancelable: Добавить кнопку отмены в диалог.
            delayed: Добавить задержку перед показом диалога.
        """
        super().__init__(function, name, description, title, message)

        self.__shadow_task = self.__ShadowTaskWrapper(self)

        self.__ph: ShadowProgressHandler = ShadowProgressHandler()
        self.__ph.setParent(self)
        self.__shadow_task.init_progress_handler(self.__ph)

        self.__ph_shadow.canceled.connect(self.cancel)

        self.__ph.range_changed.connect(self.__on_range_changed)

        self.__previous_range: Task.Range = self.Range(self.min, self.max)
        self.__infinite_progress: bool = False

        self.__cancelable: bool = cancelable
        self.__delayed: bool = delayed

    def __new__(cls, *args: Any, **kwargs: Any) -> "DialogTask":
        cls.__range_changed: Signal = Signal(cls.Range)
        return super().__new__(cls, *args, **kwargs)  # type: ignore[return-value]

    @property
    def __ph_shadow(self) -> ProgressHandler:
        return self.__ph.shadow()

    @Slot(float, float)
    def __on_range_changed(self, min_value: float, max_value: float) -> None:
        self.range_changed.emit(self.Range(min_value, max_value))

    @property
    def title(self) -> str:
        return super().title

    @title.setter
    def title(self, value: str) -> None:
        # noinspection PyAttributeOutsideInit
        self._Task__title = value
        self.__ph_shadow.setWindowTitle(value)

    @property
    def title_changed(self) -> Signal:
        return self.__ph_shadow.windowTitleChanged

    @property
    def message(self) -> str:
        return super().message

    @message.setter
    def message(self, value: str) -> None:
        # noinspection PyAttributeOutsideInit
        self._Task__message = value
        self.__ph_shadow.setMessage(value)

    @property
    def message_changed(self) -> Signal:
        return self.__ph_shadow.messageChanged

    @property
    def cancelable(self) -> bool:
        """Возвращает True, если в диалоге есть кнопка отмены."""
        return self.__cancelable

    @property
    def delayed(self) -> bool:
        """Возвращает True, если в диалоге есть задержка перед показом."""
        return self.__delayed

    def __prepare_dialog(self, *args: Any, **kwargs: Any) -> "QDialog":
        self._Task__init_task_args(*args, **kwargs)  # type: ignore[attr-defined]

        flags = axipy.ProgressGuiFlags.IDLE
        if self.cancelable:
            flags |= axipy.ProgressGuiFlags.CANCELABLE
        if not self.delayed:
            flags |= axipy.ProgressGuiFlags.NO_DELAY

        spec = axipy.ProgressSpecification(
            description=self.message,
            window_title=self.title,
            flags=flags,
        )
        d = axipy.task_manager._generate_dialog_for_task(self.__shadow_task, spec)
        d.setParent(axipy.mainwindow.widget)
        d.setAttribute(PySide2.QtCore.Qt.WA_DeleteOnClose, True)

        with _SignalBlocker(self.__ph_shadow):
            self.range = self.Range(self.min, self.max)

        axipy.task_manager._start_shadow_task(self, self.__shadow_task)

        return d

    def start(self, *args: Any, **kwargs: Any) -> None:
        d = self.__prepare_dialog(*args, **kwargs)

        # noinspection PyUnresolvedReferences
        if self.status not in self._Task__FINISHED_STATES:  # type: ignore[attr-defined]
            d.open()

    def run_and_get(self, *args: Any, **kwargs: Any) -> Any:
        d = self.__prepare_dialog(*args, **kwargs)
        # noinspection PyUnresolvedReferences
        if self.status not in self._Task__FINISHED_STATES:  # type: ignore[attr-defined]
            d.exec_()
        return self.result

    @property
    def value(self) -> float:
        return self.__ph_shadow.progress()

    @value.setter
    def value(self, value: float) -> None:
        self.__ph_shadow.setProgress(value)

    @property
    def value_changed(self) -> Signal:
        return self.__ph_shadow.progressValueChanged

    @property
    def min(self) -> float:
        return self.__ph.get_progress_min()

    @min.setter
    def min(self, value: float) -> None:
        self.__ph.set_progress_min(value)

    @property
    def max(self) -> float:
        return self.__ph.get_progress_max()

    @max.setter
    def max(self, value: float) -> None:
        self.__ph.set_progress_max(value)

    @property
    def range(self) -> Task.Range:
        list_float = self.__ph.get_progress_range()
        return self.Range(list_float[0], list_float[1])

    @range.setter
    def range(self, value: Task.Range) -> None:
        self.__ph.set_progress_range(value.min, value.max)

    @property
    def range_changed(self) -> Signal:
        """
        Возвращает сигнал, испускаемый при изменении диапазона прогресса задачи.

        :rtype: Signal[:class:`axipy.Task.Range`]
        """
        return self.__range_changed

    @property
    def infinite_progress(self) -> bool:
        """
        Возвращает или устанавливает бесконечную полоску прогресса для задачи.

        Возвращает :obj:`True`, если бесконечный прогресс установлен, иначе возвращает :obj:`False`.
        При изменении значения на :obj:`False`, восстанавливает предыдущее значения прогресса.
        При изменениях значения свойства, сигнал :attr:`axipy.Task.range_changed` не вызывается.
        """
        return self.__infinite_progress

    @infinite_progress.setter
    def infinite_progress(self, value: bool) -> None:
        if value:
            self.__previous_range = self.range
            range_ = self.Range(0, 0)
        else:
            range_ = self.__previous_range

        self.__ph_shadow.blockSignals(True)
        try:
            self.range = range_
        finally:
            self.__ph_shadow.blockSignals(False)

        self.__infinite_progress = value

    @property
    def is_canceled(self) -> bool:
        if shiboken2.isValid(self.__ph):
            return self.__ph_shadow.isCanceled()
        else:
            return super().is_canceled

    @Slot()
    def cancel(self) -> None:
        self.__ph_shadow.cancel()
        super().cancel()


def _apply_deprecated() -> None:
    # Using getattr to hide deprecated objects from axipy namespace on IDE inspections
    getattr(__all__, "extend")(
        (
            "AxipyTask",
            "AxipyAnyCallableTask",
        )
    )

    class AxipyTask(ShadowTask):
        """
        Warning:
            .. deprecated:: 6.0.0
                Используйте :class:`axipy.Task`.
        """

        # @_deprecated_by("axipy.Task")
        def __init__(self) -> None:
            super().__init__()
            self.__ph: Optional["axipy.AxipyProgressHandler"] = None
            self.set_progress_handler(axipy.AxipyProgressHandler())

        def _init_progress_handler(self, ph: "axipy.AxipyProgressHandler") -> None:
            if not isinstance(ph, axipy.AxipyProgressHandler):
                raise TypeError(f"Wrong type: {type(ph)}")
            super().init_progress_handler(ph._wrapper())

        # @_deprecated_by("axipy.Task")
        def progress_handler(self) -> "axipy.AxipyProgressHandler":
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            # deferred initialization of ph, won't be None
            return self.__ph  # type: ignore[return-value]

        # @_deprecated_by("axipy.Task")
        def set_progress_handler(self, ph: "axipy.AxipyProgressHandler") -> None:
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            self.__ph = ph
            self._init_progress_handler(self.__ph)

        def run_internally(self) -> None:
            try:
                # deferred initialization of ph, won't be None
                self.__ph.result = self.run()  # type: ignore[union-attr]
            except Exception:
                traceback.print_exc()
                exc_type, value = sys.exc_info()[:2]
                # deferred initialization of ph, won't be None
                self.__ph.error.emit((exc_type, value, traceback.format_exc()))  # type: ignore[union-attr]

        @abstractmethod
        # @_deprecated_by("axipy.Task")
        def run(self) -> Any:
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            pass

        # @_deprecated_by("axipy.Task")
        def on_finished(self, result: Any) -> None:
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            pass

    class AxipyAnyCallableTask(AxipyTask):
        """
        Warning:
            .. deprecated:: 6.0.0
                Используйте :class:`axipy.Task`.
        """

        # @_deprecated_by("axipy.Task")
        def __init__(self, fn: Callable, *args: Any, **kwargs: Any) -> None:
            super().__init__()
            self.__fn = fn
            self.__args = args
            self.__kwargs = kwargs
            self.__with_handler = True

        # @_deprecated_by("axipy.Task")
        def with_handler(self, value: bool) -> None:
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            self.__with_handler = value

        # @_deprecated_by("axipy.Task")
        def run(self) -> Any:
            """
            Warning:
                .. deprecated:: 6.0.0
                    Используйте :class:`axipy.Task`.
            """
            if self.__with_handler:
                return self.__fn(self.progress_handler(), *self.__args, **self.__kwargs)
            else:
                return self.__fn(*self.__args, **self.__kwargs)

    globals().update(
        AxipyTask=AxipyTask,
        AxipyAnyCallableTask=AxipyAnyCallableTask,
    )


_apply_deprecated()
