"""
Класс страницы wizard для получения прогресса и результатов конвертации.
"""
import re
import traceback
from copy import copy
from typing import List

from PySide2.QtCore import Signal, Slot, QObject, QTimer
from PySide2.QtWidgets import QProgressBar, QTextEdit, QVBoxLayout, QWizard, QWizardPage
from osgeo import gdal

from axipy import tr, task_manager, AxipyAnyCallableTask, CoordTransformer
from .convert_func import Convert
from .. import helper
from .. import wizardconverter
from ..helper import debug


class ResultsPage(QWizardPage):
    """
    Страница результатов конвертации. При инициализации страницы собирается пользовательский ввод из внутренних
    параметров parent(QWizard), выполняется обработка исключений. Собранные параметры передается в функцию
    gdal.VectorTranslate для конвертации. Конвертация выполняется в отдельном потоке. Информация о прогрессе
    конвертации и об ошибках и предупреждениях gdal выводится в окно вывода на странице и отображается
    в полосе прогресса.
    """

    def __init__(self, wizard) -> None:
        super().__init__()

        self.wzrd = wizard  # type: wizardconverter.WizardConverter

        # Вертикальная разметка
        convert_vbox = QVBoxLayout()
        self.setLayout(convert_vbox)
        # Окно вывода информации
        self.text_edit = QTextEdit()
        self.text_edit.setReadOnly(True)
        convert_vbox.addWidget(self.text_edit)
        # Полоса прогресса
        self.pbar = QProgressBar(self)
        self.pbar.hide()
        convert_vbox.addWidget(self.pbar)

        # Словарь для обработчика ошибок gdal
        self.err_msg_dict = {2: tr("Предупреждение"),
                             3: tr("Ошибка")}
        # Для подсчета одинаковых сообщений в обработчике gdal
        self.last_err_str = None  # Последнее сообщение
        self.Skipped = None  # Количество пропущенных сообщений
        self.err_str_dict = {}  # Словарь исключений для каждого файла

        self.allow_callback = False

        # Внутреннее значение progress bar для драйверов с callback
        self.internal_pbar_value = None

        # Разделительная линия консоли
        self.console_line = "_" * 40 + "\n"

        # Задача выполняемая в потоке
        self.task = None

        # Сигналы для связи со слотами из потока
        self.signals = None

    def initializePage(self) -> None:
        """
        Инициализаци страницы.
        Инициализация вспомогательных параметров для конвертации.
        Сбор польз. ввода и передача списка параметров в функцию для конвертирования.
        """
        try:
            """
            Инициализация гуи
            """
            # Инициализация text edit
            self.text_edit.clear()
            self.pbar.show()
            # Механизм режимов полоски прогресса
            self.internal_pbar_value = 100
            # Механизм пропуска одинаковых сообщений
            self.last_err_str = ""
            self.Skipped = 0
            # Инициализация словаря ошибок
            self.err_str_dict.clear()
            # Сброс текста кнопки отмена (для обработки <- назад -> далее)
            self.setButtonText(QWizard.CancelButton, self.wzrd.buttonText(QWizard.CancelButton))

            params = self.prepare_params()
            self.print_params(params)

            """
            Таймер использован т.к. после выполнения initializePage() состояние кнопок Wizard сбрасывается к исходному
            и после сброса срабатывает таймер, блокирующий кнопки.
            """
            QTimer.singleShot(0, lambda: self.wzrd.button(QWizard.BackButton).setEnabled(False))
            QTimer.singleShot(0, lambda: self.wzrd.button(QWizard.CancelButton).setEnabled(False))

            self.wzrd.starting_task()

            gdal.DontUseExceptions()  # Отключение исключений (Все ошибки отправляются в обработчик)

            self.signals = Signals()
            self.signals.text_edit_append.connect(self.text_edit_append)
            self.signals.change_pbar_cb.connect(self.change_pbar_cb)
            self.signals.init_pbar_range.connect(self.init_pbar_range)
            self.signals.change_pbar.connect(self.change_pbar)
            self.signals.set_allow_callback.connect(self.set_allow_callback)
            self.signals.set_pbar_max.connect(self.set_pbar_max)

            convert = Convert(self, params)
            self.task = AxipyAnyCallableTask(convert.run)
            self.task.with_handler(False)
            self.task.progress_handler().finished.connect(self.finished)
            task_manager.start_task(self.task)

        except (Exception,):
            if debug:
                traceback.print_exc()
            self.pbar.hide()
            self.text_edit.append(tr("Не удалось начать конвертацию."))

    def prepare_params(self) -> List:
        """
        Сбор и подготовка параметров для передачи в функцию конвертации.
        """

        # Thread GDAL callback
        def callback(*args) -> bool:
            """
            Функция обратной связи для получения прогресса конвертации.
            Если польз-ль отменил процесс конвертации, отправляет сигнал отмены в функцию VectorTranslate.
            *args в пармаетрах функции, потому что gdal при ошибке декодирования может отправить
            нечитаемый callback, что приводит к падению программы.

            arg0 Значение прогресса конвертации.
            arg1
            arg2
            """
            try:
                if self.allow_callback:
                    arg0 = args[0]
                    value = int(arg0 * 100)
                    self.signals.change_pbar_cb.emit(value)

                # False служит сигналом прерывания процесса конвертации для gdal
                if self.task.progress_handler().is_canceled():
                    return False
                else:
                    return True

            except Exception as e:
                self.signals.text_edit_append.emit(str(e))

        # Опции создания
        creation_options = {
            "format": self.wzrd.out_gdal_format,  # Выходной формат gdal
            "srcSRS": None,  # Входная СК
            "dstSRS": None,  # Выходная СК
            "reproject": self.wzrd.reproject,  # Параметр перепроецирования
            "datasetCreationOptions": self.wzrd.dataset,  # Параметры создания набора данных
            "layerCreationOptions": self.wzrd.layer,  # Параметры создания слоя
            "callback": callback,
            "skipFailures": self.wzrd.skip_failures,
        }

        # Инициализация и обработка декартовых СК
        if self.wzrd.inp_cs and self.wzrd.out_cs:  # inp_cs не имеет смысла без out_cs
            if self.wzrd.inp_cs.non_earth:  # На входе
                creation_options["srcSRS"] = self.wzrd.inp_cs.wkt
            else:
                creation_options["srcSRS"] = self.wzrd.inp_cs.proj

        if self.wzrd.out_cs:
            if self.wzrd.out_cs.non_earth:  # На выходе
                creation_options["dstSRS"] = self.wzrd.out_cs.wkt
            else:
                creation_options["dstSRS"] = self.wzrd.out_cs.proj

        """ Драйвера-исключения из общих правил """

        # Афинные преобразования и GML (EPSG)
        if self.wzrd.inp_cs and self.wzrd.out_cs:
            """
            Афинные преобраз. есть только у формата MapInfo.
            Получение pipeline для афинных преобраз. на входе или выходе.
            """
            p = re.compile(r'Affine\s+Units\s+".*",\s+(.*)')
            inp_affine = p.search(self.wzrd.inp_cs.prj)
            out_affine = p.search(self.wzrd.out_cs.prj)
            """
            GML может записать только EPSG (может не быть), преобразования с EPSG могут быть неточными,
            поэтому transformer.
            """
            out_gml_and_epsg = bool(self.wzrd.out_gdal_format == "GML" and self.wzrd.out_cs.epsg > 0)

            if any((inp_affine, out_affine, out_gml_and_epsg)):
                coord_operations = CoordTransformer.proj_transform_definition(
                    self.wzrd.inp_cs.proj, self.wzrd.out_cs.proj)
                if "pipeline" not in coord_operations:
                    coord_operations = "proj=pipeline step " + coord_operations
                if any((inp_affine, out_affine)):

                    def get_affine_pipeline(search1, search2, transform_def_affine) -> str:
                        """
                        Функция строит pipeline для трансформации координат с аффинными преобразованиями.
                        """
                        pipeline = transform_def_affine
                        p_pipeline = re.compile(r"pipeline")

                        # Прочитать афины из prj
                        if search1:
                            af1 = search1.group(1).replace(" ", "").split(",")
                            a, b, c, d, e, f = af1
                            affine = f"proj=affine s11={a} s12={b} xoff={c} s21={d} s22={e} yoff={f}"
                            n = p_pipeline.search(pipeline).end()
                            pipeline = pipeline[:n] + " step inv " + affine + pipeline[n:]
                        if search2:
                            af2 = search2.group(1).replace(" ", "").split(",")
                            a, b, c, d, e, f = af2
                            affine = f"proj=affine s11={a} s12={b} xoff={c} s21={d} s22={e} yoff={f}"
                            pipeline = pipeline + " step " + affine
                        return pipeline

                    coord_operations = get_affine_pipeline(inp_affine, out_affine, coord_operations)
                if out_gml_and_epsg:
                    if "pipeline" in coord_operations:
                        coord_operations += " step "
                    coord_operations += "proj=axisswap order=2,1"
                    creation_options["dstSRS"] = f"EPSG:{self.wzrd.out_cs.epsg}"
                creation_options["coordinateOperation"] = coord_operations
        elif self.wzrd.out_cs and self.wzrd.out_cs.epsg and self.wzrd.out_gdal_format == "GML":
            creation_options["dstSRS"] = f"EPSG:{self.wzrd.out_cs.epsg}"

        if "FORMAT=MIF" in creation_options["datasetCreationOptions"]:
            self.wzrd.out_ext = ".mif"

        # Параметры, передаваемые в функцию конвертирования
        return [
            self.wzrd.input_path_list,  # Список входных файлов
            [self.wzrd.out_path, self.wzrd.out_ext],  # Выходная директория, выходное расширение
            self.wzrd.open_options,  # Параметры открытия входных файлов
            creation_options,  # Параметры создания
        ]

    def print_params(self, params_arg: list) -> None:
        """
        Вывод отладочных данных, предназначенных для пользователя.
        """
        self.text_edit.append(f"{tr('Входной формат')}: {self.wzrd.inp_gdal_format}\n")
        self.text_edit.append(f"{tr('Выходной формат')}: {self.wzrd.out_gdal_format}\n")

        open_options = copy(self.wzrd.open_options)
        gdal_config_options = self.wzrd.gdal_config_options
        if gdal_config_options:
            open_options.append(gdal_config_options)
        if open_options:
            self.text_edit.append(f"{tr('Параметры открытия')}: {open_options}\n")
        # Параметры функции
        for k, v in params_arg[len(params_arg) - 1].items():
            if k in ("callback", "format"):
                continue
            txt = f"{k}={v}"
            self.text_edit.append(txt)
            if k == "srcSRS":
                if self.wzrd.inp_cs:
                    self.text_edit.append("{} = {}.".format(tr("Название входной СК"), self.wzrd.inp_cs.name))
                else:
                    self.text_edit.append("{} = {}.".format(tr("Название входной СК"),
                                                            tr("Входной формат не поддерживает систему координат")))
            elif k == "dstSRS":
                if self.wzrd.out_cs:
                    self.text_edit.append("{} = {}.".format(tr("Название выходной СК"), self.wzrd.out_cs.name))
                else:
                    self.text_edit.append("{} = {}.".format(tr("Название выходной СК"),
                                                            tr("Выходной формат не поддерживает систему координат")))
            self.text_edit.append("")
        self.text_edit.append(self.console_line + "\n")

    def cleanupPage(self) -> None:
        """
        Очистка страницы, при нажатии кнопки назад.
        Сбрасывает кнопки навигации внизу wizard.
        """
        layout = [QWizard.Stretch, QWizard.BackButton, QWizard.NextButton, QWizard.CancelButton]
        self.wzrd.setButtonLayout(layout)
        self.setButtonText(QWizard.NextButton, tr("Конвертировать"))

    @Slot()
    def finished(self) -> None:
        """
        Функция вызывается при завершении выполнения задачи в потоке.
        Изменяет клавиши навигации для выбора возможности повторного конвертирования.
        Скрывает полосу прогресса.
        """
        if self.task.progress_handler().is_canceled():
            helper.notify(tr("Конвертация файлов отменена"), 0)

        layout = [QWizard.Stretch, QWizard.BackButton, QWizard.NextButton, QWizard.CustomButton1, QWizard.CancelButton]
        self.setButtonText(QWizard.CancelButton, tr("Завершить"))
        self.wzrd.setButtonLayout(layout)

        self.wzrd.finishing_task()

    # Thread TextEdit Slot

    @Slot(str)
    def text_edit_append(self, txt: str) -> None:
        """
        Отправляет полученный сигнал со строкой в поле вывода пользовательского интерфейса.

        :param str txt: Строка, полученная в сигнале из потока.
        """
        msg_error = tr("Ошибка")
        msg_warning = tr("Предупреждение")
        color_dict = {
            msg_error: "orange",
            msg_warning: "yellow"
        }

        def colorize(msg_type: str) -> None:
            """
            Окрашивает цветом тип исключения в сообщении об исключении gdal, взависимости от типа исключения.
            Выводит окрашенное сообщение.

            :param str msg_type: Тип исключения gdal
            """
            color = color_dict.get(msg_type, "white")
            p = re.compile(msg_type + r"\s\d+")
            if p.match(txt):
                i = p.match(txt).end()
                self.text_edit.setTextBackgroundColor(color)
                self.text_edit.insertPlainText(f"{txt[:i]}")
                self.text_edit.setTextBackgroundColor("white")
                self.text_edit.insertPlainText(f"{txt[i:]}\n")
            else:
                self.text_edit.insertPlainText(f"{txt}\n")

        if txt.startswith(msg_error):
            colorize(msg_error)
        elif txt.startswith(msg_warning):
            colorize(msg_warning)
        else:
            self.text_edit.insertPlainText(f"{txt}\n")
        # Прокрутка страницы вниз
        self.text_edit.verticalScrollBar().setValue(self.text_edit.verticalScrollBar().maximum())

    # Thread Pbar Slots

    @Slot()
    def change_pbar(self) -> None:
        """
        Изменяет внутреннее значение полосы прогресса.
        """
        self.internal_pbar_value += 100
        self.pbar.setValue(self.internal_pbar_value)

    @Slot()
    def init_pbar_range(self) -> None:
        """
        Вычисляет длину списка файлов для конвертации.
        Задает разметку кратную 100 для полосы прогресса.
        """
        n = len(self.wzrd.input_path_list)

        self.pbar.setRange(0, n * 100)
        self.pbar.setValue(100)
        self.internal_pbar_value = 100

        self.allow_callback = True

    @Slot(int)
    def change_pbar_cb(self, value: int) -> None:
        """
        Движение полосы прогресса для драйверов c callback.
        :param value: Значение callback.
        """
        if value <= 100:
            i = self.internal_pbar_value + value
            self.pbar.setValue(i)

    @Slot(int)
    def set_pbar_max(self, value: int) -> None:
        """
        Альтернативная инициализация прогресса.
        """
        self.pbar.setMaximum(value)
        self.allow_callback = True
        self.internal_pbar_value = 0

    @Slot(bool)
    def set_allow_callback(self, value: bool) -> None:
        self.allow_callback = value


class Signals(QObject):
    """
    Сигналы для связи со слотами из потока.
    """
    text_edit_append = Signal(str)
    change_pbar_cb = Signal(int)
    set_pbar_max = Signal(int)
    init_pbar_range = Signal()
    change_pbar = Signal()
    set_allow_callback = Signal(bool)
