"""
Функция конвертации
"""
import re
import time
import traceback
import xml.etree.ElementTree as ElemTree
from datetime import datetime
from pathlib import Path
from types import TracebackType
from typing import List, Optional, Callable, Type

from PySide2.QtWidgets import QWizardPage
from axipy import tr, provider_manager
from osgeo import gdal, ogr
from osgeo.gdal import Dataset
from osgeo.ogr import Layer

from .. import helper
from .. import wizardconverter
from ..helper import debug, clear_data_dir


class GDALErrorHandlerContext:
    """
    Context manager for GDAL error handling.
    Automatically pushes error handler on entry and pops on exit.
    """

    def __init__(self, error_handler: Callable) -> None:
        self.error_handler = error_handler

    def __enter__(self) -> "GDALErrorHandlerContext":
        """Enter the context - push error handler"""
        gdal.PushErrorHandler(self.error_handler)
        return self

    def __exit__(
        self,
        exc_type: Optional[Type[BaseException]],
        exc_val: Optional[BaseException],
        exc_tb: Optional[TracebackType],
    ) -> bool:
        """Exit the context - pop error handler"""
        gdal.PopErrorHandler()
        return False  # Don't suppress exceptions


class Params:

    def __init__(self, params: List) -> None:
        self.inp_paths = params[0]  # type: Optional[List[Path]]
        self.dir_path = params[1][0]  # type: Optional[Path]
        self.out_ext = params[1][1]  # type: Optional[str]
        self.open_options = params[2]  # type: Optional[dict]
        self.creation_options = params[3]  # type: Optional[dict]


class Convert:

    def __init__(self, wzrd_page: QWizardPage, params: list) -> None:
        self.wzrd_page = wzrd_page
        self.wzrd = wzrd_page.wzrd  # type: wizardconverter.WizardConverter
        self.params = Params(params)  # type: Params

        self.input_names = []
        self.single_path = None
        self.many_to_one_gml = False

    def _handler(self, err_level=None, err_no=None, err_msg=None) -> None:
        """
        Обработчик ошибок gdal.
        Отправляет строковый сигнал на поле вывода в польз. интерфейс.

        err_level: Уровень исключения gdal.
        err_no: Идентификатор исключения gdal.
        err_msg: Описание исключения gdal.
        """
        try:

            err_level = self.wzrd_page.err_msg_dict.get(err_level, "Unexpected")

            # Если gdal выдает сообщение о том что драйвер не поддерживает callback
            if str(err_msg).startswith("Progress turned off"):
                if debug:
                    print(f"callback turned off. {self.wzrd.inp_gdal_format} -> {self.wzrd.out_gdal_format}")
                self.params.creation_options.pop("callback")  # Отключение callback
                return None

            # Удаление новой строки и пробелов в конце строки
            p = re.compile(r"(\n|\s)*\.?$")
            err_msg = p.sub("", err_msg)
            err_str = "{} {}: {}".format(err_level, err_no, err_msg)
            full_string = ""

            # Обработка частного сообщения gdal (MapInfo File to ESRI Shapefile)
            p = re.compile(r"(?P<err_level>{}) (?P<err_no>{}): {}".format(
                self.wzrd_page.err_msg_dict[2], 1,
                r"(?P<txt1>Value) (?P<value>.*) (?P<txt2>of field) (?P<field>.*) (?P<txt3>of feature) "
                r"(?P<feature>.*) (?P<txt4>not successfully written\."
                r"\sPossibly due to too larger number with respect to field width)"))
            if p.match(err_str):
                full_string = err_str
                err_str = p.sub("(?P=txt1)(?P=txt2)(?P=field)(?P=txt3)(?P=txt4)", err_str)

            if err_str not in self.wzrd_page.err_str_dict.keys():  # Если строки ошибки нет в ключах
                # Добавить ошибку в ключ со значением 1
                self.wzrd_page.err_str_dict[err_str] = {"full_string": full_string, "count": 1}
            else:  # Если строка ошибки есть в ключах
                self.wzrd_page.err_str_dict[err_str]["count"] += 1  # Значение по ключу увеличивается на 1
        except Exception as e:
            if debug:
                print("\n" * 2)
                traceback.print_exc()
                print("\n" * 4)
            self.wzrd_page.signals.text_edit_append.emit(str(e) + "\n")

    def _print_and_clear_err_str_dict(self) -> None:
        """
        Вывод ошибок и предупреждений после каждого файла.
        """
        try:
            for k, v in self.wzrd_page.err_str_dict.items():
                if v["full_string"]:
                    err_str = v["full_string"]
                else:
                    err_str = k
                # Точка в конце строки если строка не заканчивается на точку
                err_str = helper.add_dot(err_str)
                if v["count"] > 1:
                    self.wzrd_page.signals.text_edit_append.emit("{}".format(err_str))
                    self.wzrd_page.signals.text_edit_append.emit("{}: {}.".format(
                        tr("Пропущено похожих сообщений"), v["count"]))
                else:
                    self.wzrd_page.signals.text_edit_append.emit("{}".format(err_str))
                self.wzrd_page.signals.text_edit_append.emit("")
            self.wzrd_page.err_str_dict.clear()
        except Exception as e:
            self.wzrd_page.signals.text_edit_append.emit(str(e))

    def _change_cs_for_mif(self, out_path_arg: Path) -> None:
        """
        Замена proj на prj для MapInfo File формата MIF,т.к. строка СК, формируемая gdal не вполне
        корректная (например, 9999 для долготы широты).
        """
        p = re.compile(r'^(CoordSys\s+)Earth\s+Projection\s+(?:\s*(?:\S+|"[^"]*")\s*,)*\s*(?:\S+|"[^"]*")(.*)')

        if (self.wzrd.out_gdal_format == "MapInfo File" and
                "FORMAT=MIF" in self.params.creation_options["datasetCreationOptions"]):
            # Чтение из сконвертированного файла
            with open(str(out_path_arg), "r", encoding="cp1251") as f:
                lines = f.readlines()
            # Поиск и замена строчки системы координат
            for line_indx, line in enumerate(lines):
                # Проверка отмены конвертации
                if self.wzrd_page.task.progress_handler().is_canceled():
                    return None
                if p.match(line):
                    prj = self.wzrd.out_cs.prj
                    lines[line_indx] = p.sub(r"\1{}\2".format(prj), line)
                    break
            # Запись обратно в сконвертированный файл
            with open(str(out_path_arg), "w") as f:
                f.write("".join(lines))

    def _change_cs_for_tab(self, out_path_arg: Path) -> None:
        """
        Если на выходе tab;
        Если в выходной системе координат есть афинные преобразования;
        Система координат файла меняется вручную.
        """
        if (self.wzrd.out_gdal_format == "MapInfo File" and
                "FORMAT=TAB" in self.params.creation_options["datasetCreationOptions"]):
            p = re.compile(r'Affine\s+Units\s+".*",\s+(.*)')
            out_affine = p.search(self.wzrd.out_cs.prj)
            if out_affine:
                provider_manager.tab.change_coordsystem(str(out_path_arg), self.wzrd.out_cs)

    def _change_name_if_exists(self, path: Path) -> Path:
        """
        Проверка существования пути к файлу, если путь существует, возвращает путь к файлу с номером в скобках.
        Например: Файл(2), Файл(3) и т.д. Функция выполняется рекурсивно, пока не найдется несуществующий путь.
        """
        # TODO: Файлы с одинаковыми именами, но в разных выходных папках
        if path.name in self.input_names:
            p = re.compile(r"\((\d+)\)$")
            search = p.search(str(path.stem))
            if search:
                digit = int(search.group(1))
                new_stem = p.sub(f"({digit + 1})", str(path.stem))
            else:
                new_stem = path.stem + "(2)"
            new_path = (path.parent / new_stem).with_suffix(path.suffix)
            if new_path.name in self.input_names:
                return self._change_name_if_exists(new_path)
            else:
                self.input_names.append(new_path.name)
                return new_path
        else:
            self.input_names.append(path.name)
            return path

    def run(self) -> None:
        try:  # Основной процесс конвертации
            self._run_with_handler()
        except (Exception,):
            if debug:
                traceback.print_exc()
            self.wzrd_page.signals.text_edit_append.emit(tr("Ошибка в процессе конвертации.") + "\n")

    def _run_with_handler(self) -> None:
        with GDALErrorHandlerContext(self._handler):
            self._run()

    def _run(self) -> None:
        start_function = datetime.now()  # Время старта конвертации

        # Создать директорию, без исключения, если существует
        Path.mkdir(self.params.dir_path, exist_ok=True)

        if self.wzrd.out_gdal_format == "GML" and self.wzrd.file_creation_mode == "many_to_one":
            self._gml_many_to_one()
        else:
            # Основной цикл по файлам
            for inp_path_index, inp_path in enumerate(self.params.inp_paths):
                self._single_path(inp_path_index, inp_path)
                # Проверка отмены конвертации
                if self.wzrd_page.task.progress_handler().is_canceled():
                    break

        # Вывод длительности конвертации
        finish_function = datetime.now()
        self.wzrd_page.signals.text_edit_append.emit("{} ({}: {}).".format(
            tr("Конвертация завершена"),
            tr("длительность").capitalize(),
            get_time(start_function, finish_function)
        ))

    def _single_path(self, inp_path_index: int, inp_path: Path) -> None:

        # Настройка полоски прогресса для работы с/без callback, и для одного файла
        if inp_path_index == 1:
            if not self.many_to_one_gml:
                self.wzrd_page.signals.init_pbar_range.emit()  # Меняем с бесконечного на обычный range после 1ого элем.
        elif inp_path_index > 1:
            if not self.many_to_one_gml:
                self.wzrd_page.signals.change_pbar.emit()

        self.wzrd_page.signals.text_edit_append.emit("{}: {}\n".format(tr("Входной файл"), str(inp_path)))

        # Конструкция выходной строки
        out_path = self.params.dir_path.joinpath(inp_path.stem).with_suffix(self.params.out_ext)

        # gdal
        try:
            src_ds: Dataset = self.open_with_config(str(inp_path), open_options=self.params.open_options)
            mode = self.wzrd.file_creation_mode

            if mode == "many_to_many":  # Аналогично исходному
                self._many_to_many(src_ds, out_path)

            elif mode == "one_to_one":  # Каждую таблицу в отдельный файл
                self._one_to_one(inp_path, inp_path_index, src_ds, out_path)

            elif mode == "many_to_one":  # Собрать все таблицы в один файл
                self._many_to_one(inp_path_index, src_ds, out_path)

        except Exception as exception:
            raise Exception("Ошибка конвертации GDAL") from exception
        finally:
            src_ds = None

    def _many_to_many(self, src_ds: Dataset, out_path: Path) -> None:
        out_path = self._change_name_if_exists(out_path)
        self.wzrd_page.signals.text_edit_append.emit("{}: {}\n".format(tr("Выходной файл"), str(out_path)))
        self.translate_with_config(str(out_path), src_ds, **self.params.creation_options)
        self._print_and_clear_err_str_dict()
        self.wzrd_page.signals.text_edit_append.emit(self.wzrd_page.console_line)

    def _one_to_one(self, inp_path: Path, inp_path_index: int, src_ds: Dataset, out_path: Path) -> None:
        layer_count = src_ds.GetLayerCount()
        new_dir_path = None

        if layer_count > 1:  # Создание выходной папки для многослойного файла на входе
            new_dir_path = self.params.dir_path / inp_path.stem
            Path.mkdir(new_dir_path, exist_ok=True)
            self.wzrd_page.signals.text_edit_append.emit(
                "{}: {}\n".format(tr("Выходная папка"), str(new_dir_path)))
            self.wzrd_page.signals.text_edit_append.emit(self.wzrd_page.console_line)

            # Отключения callback на время конвертации по слоям
            self.wzrd_page.signals.set_allow_callback.emit(False)

        for l_index in range(layer_count):  # Основной цикл по слоям

            skip_layer = False
            layer = src_ds.GetLayer(l_index)  # type: Layer
            layer_name = layer.GetName()
            self.params.creation_options["layers"] = layer_name

            """
            Driver specific
            """
            if layer_count > 1:  # Имя слоя вместо имени файла для многослойных файлов на входе

                # У входных слоев DGN и DXF одинак. имена
                if new_dir_path is None:
                    raise Exception("new_dir_path is None")
                if self.wzrd.inp_gdal_format in ("DGN", "DXF"):
                    out_path = new_dir_path.joinpath(f"{layer_name}_{inp_path_index}").with_suffix(self.params.out_ext)
                else:
                    out_path = new_dir_path.joinpath(layer_name).with_suffix(self.params.out_ext)
            # Выбор поля геометрии для выходных форматов DGN, DXF (Не поддерживают другие поля)
            if self.wzrd.out_gdal_format in ("DGN", "DXF"):
                geom_column = layer.GetGeometryColumn()
                if geom_column == "":
                    geom_column = "_ogr_geometry_"
                sql_str = f'SELECT "{geom_column}" FROM "{layer_name}"'  # Для DXF
                if self.wzrd.out_gdal_format == "DGN":
                    sql_str = f'SELECT {240} AS "ColorIndex" FROM "{layer_name}"'  # Для DGN
                self.params.creation_options["SQLStatement"] = sql_str
                self.params.creation_options.pop("layers", None)
            # Selafin поддерживает только слои типа POINT
            elif self.wzrd.out_gdal_format == "Selafin":
                geometry_name = helper.geometry_type_to_name(layer.GetGeomType())
                if geometry_name != "POINT":
                    msg = f"" + \
                          f"{tr('Драйвер Selafin поддерживает геометрию слоя одного типа (POINT)')}. " + \
                          f"{tr('слой').capitalize()} {layer_name} " + \
                          f"{tr('пропущен')}."
                    self._handler(2, 1, msg)
                    skip_layer = True
            elif self.wzrd.out_gdal_format == "ESRI Shapefile":
                # Получение названий столбцов и их типов
                column_name_type_dict = {}
                ldfn = layer.GetLayerDefn()
                field_count = ldfn.GetFieldCount()
                for field_indx in range(field_count):
                    fdfn = ldfn.GetFieldDefn(field_indx)
                    column_name = fdfn.name
                    column_type = ogr.FieldDefn.GetFieldTypeName(fdfn, fdfn.GetType())
                    assert column_name not in column_name_type_dict
                    column_name_type_dict[column_name] = column_type

                def less_than_n_bytes(str_arg: str, n_bytes: int) -> str:
                    """Сокращение длины строчки в байтах до заданного значения. (Меньше либо равно)"""
                    while len(str_arg.encode("utf-8")) > n_bytes:
                        str_arg = str_arg[:-1]
                    return str_arg

                def if_in_list_add_number(list_arg: List, str_arg: str):
                    """
                    Обработка одинаковых названий столбцов. Строчка сокращается до 8 байт
                    и в конец строчки добавляется порядковый номер от 1 до 99.
                    """
                    p = re.compile(r"[_\d]\d$")
                    while str_arg in list_arg:
                        result = p.search(str_arg)
                        if result:
                            current_digit = int(result.group().replace("_", "0"))
                            new_digit = current_digit + 1
                            if 0 <= new_digit <= 9:
                                str_arg = less_than_n_bytes(str_arg, 8) + "_" + str(new_digit)
                            elif 10 <= new_digit <= 99:
                                str_arg = less_than_n_bytes(str_arg, 8) + str(new_digit)
                        else:
                            str_arg = less_than_n_bytes(str_arg, 8) + "_1"

                        str_arg = if_in_list_add_number(list_arg, str_arg)

                    return str_arg

                # Получение новых имен не превышающих 10 байтов
                new_names = []
                for name in column_name_type_dict.keys():
                    size = 0
                    new_name = ""
                    for char in name:
                        char_b = char.encode("utf-8")
                        size += len(char_b)
                        if size > 10:
                            break
                        else:
                            new_name += char
                    new_name = if_in_list_add_number(new_names, new_name)
                    new_names.append(new_name)

                # Создание SQL запроса для переименования новых полей и преобразования
                # Real -> float (По умолчанию Gdal обнуляет поля Real больше 4 знаков)
                new_column_names = []
                for arg1, arg2 in zip(column_name_type_dict.keys(), new_names):
                    if column_name_type_dict[arg1] == "Real":
                        new_column_names.append(f'CAST("{arg1}" as float(24)) as "{arg2}"')
                    elif arg1 != arg2:
                        new_column_names.append(f'"{arg1}" as "{arg2}"')
                    elif arg1 == arg2:
                        new_column_names.append(f'"{arg1}"')

                    if arg1 != arg2:
                        msg = tr("Формат ESRI Shapefile поддерживает названия столбцов длиной "
                                 "не больше 10 байт. ") + \
                              tr(f"Столбец '{arg1}' переименован в столбец '{arg2}'.")
                        self._handler(2, 6, msg)

                column_names_str = ", ".join(new_column_names)
                sql_str = f'SELECT {column_names_str} FROM "{layer_name}"'

                self.params.creation_options["SQLStatement"] = sql_str
                self.params.creation_options.pop("layers", None)

            """
            Конвертация
            """
            if layer_count > 1:
                self.wzrd_page.signals.text_edit_append.emit(
                    "{}: {}\n".format(tr("Входной слой"), layer_name))
            # Обработка пропуска слоев
            if skip_layer:
                self.wzrd_page.signals.text_edit_append.emit(
                    "{}\n".format(tr("Выходной файл не создан.")))
            else:  # Конвертация
                out_path = self._change_name_if_exists(out_path)
                self.wzrd_page.signals.text_edit_append.emit(
                    "{}: {}\n".format(tr("Выходной файл"), str(out_path)))
                self.translate_with_config(str(out_path), src_ds, **self.params.creation_options)
                self._change_cs_for_mif(out_path)
                self._change_cs_for_tab(out_path)
            # Завершающее оформление
            self._print_and_clear_err_str_dict()
            self.wzrd_page.signals.text_edit_append.emit(self.wzrd_page.console_line)

        self.wzrd_page.signals.set_allow_callback.emit(True)

    def _many_to_one(self, inp_path_index: int, src_ds: Dataset, out_path: Path) -> None:

        if inp_path_index == 0:  # На первом входном файле, задаем имя для одного собираемого файла
            if self.wzrd.file_name_for_append:
                self.single_path = out_path.with_name(
                    self.wzrd.file_name_for_append).with_suffix(self.params.out_ext)
            else:
                self.single_path = out_path

        elif inp_path_index == 1:  # На втором входном файле, меняем режим на "append"
            self.params.creation_options["accessMode"] = "append"
            # Datasource creation options ignored since an existing datasource being updated.
            self.params.creation_options.pop("datasetCreationOptions")

        if self.single_path is None:
            raise Exception("single_path is None")

        self.translate_with_config(str(self.single_path), src_ds, **self.params.creation_options)
        self.wzrd_page.signals.text_edit_append.emit("{}: {}".format(tr("Выходной файл"), str(self.single_path)))
        self._print_and_clear_err_str_dict()
        self.wzrd_page.signals.text_edit_append.emit(self.wzrd_page.console_line)

    def _gml_many_to_one(self) -> None:

        self.wzrd_page.signals.set_pbar_max.emit(100)

        self.gml_dict = {}.fromkeys(self.params.inp_paths)
        for gml_dict_key in self.gml_dict.keys():
            self.gml_dict[gml_dict_key] = list()

        for inp_path in self.params.inp_paths:

            try:
                src_ds = gdal.OpenEx(str(inp_path), open_options=self.params.open_options)
                for layer_index in range(src_ds.GetLayerCount()):
                    self.gml_dict[inp_path].append(src_ds.GetLayer(layer_index).GetName())
            except Exception as exception:
                if debug:
                    print(inp_path)
                raise Exception("Ошибка конвертации GDAL") from exception
            finally:
                src_ds = None

        out_path = self.params.dir_path.joinpath(self.params.inp_paths[0].stem).with_suffix(self.params.out_ext)

        if self.wzrd.file_name_for_append:
            single_path = out_path.with_name(self.wzrd.file_name_for_append).with_suffix(self.params.out_ext)
        else:
            single_path = out_path

        root_elem = ElemTree.Element("OGRVRTDataSource")
        for ds_path, layer_names in self.gml_dict.items():
            for layer_name in layer_names:
                layer_elem = ElemTree.SubElement(root_elem, "OGRVRTLayer", name=layer_name)
                ElemTree.SubElement(layer_elem, "SrcDataSource").text = str(ds_path)
                if self.params.open_options:
                    open_options_elem = ElemTree.SubElement(layer_elem, "OpenOptions")
                    for open_option in self.params.open_options:
                        key, val = open_option.split("=")
                        ElemTree.SubElement(open_options_elem, "OOI", key=key).text = val
        tree = ElemTree.ElementTree(root_elem)

        self.wzrd.data_dir.mkdir(parents=True, exist_ok=True)
        vrt_path = self.wzrd.data_dir / "vrt.xml"  # type: Path
        vrt_path_str = str(vrt_path)
        tree.write(vrt_path_str, encoding="utf-8")

        # Конвертация
        try:
            vrt_ds = gdal.OpenEx(vrt_path_str)
            self.wzrd_page.signals.text_edit_append.emit(
                "{}: {}\n".format(tr("Выходной файл"), str(single_path)))
            self.translate_with_config(str(single_path), vrt_ds, **self.params.creation_options)
        except Exception as exception:
            raise Exception("Ошибка виртуального формата") from exception
        finally:
            vrt_ds = None

            clear_data_dir(self.wzrd.data_dir.parent)

        # Завершающие оформление
        self._print_and_clear_err_str_dict()
        self.wzrd_page.signals.text_edit_append.emit(self.wzrd_page.console_line)

    def open_with_config(self, path: str, open_options) -> Dataset:
        gdal_config_options = self.wzrd.gdal_config_options
        original_config_options = {k: gdal.GetConfigOption(k) for k, v in gdal_config_options.items()}
        for k, v in gdal_config_options.items():
            gdal.SetConfigOption(k, v)
        try:
            dataset: Dataset = gdal.OpenEx(path, open_options=open_options)
        except Exception as e:
            raise e
        finally:
            for k, v in original_config_options.items():
                gdal.SetConfigOption(k, v)
        return dataset

    def translate_with_config(self, *args, **kwargs) -> None:
        gdal_config_options = self.wzrd.gdal_config_options
        original_config_options = {k: gdal.GetConfigOption(k) for k, v in gdal_config_options.items()}
        for k, v in gdal_config_options.items():
            gdal.SetConfigOption(k, v)
        try:
            gdal.VectorTranslate(*args, **kwargs)
        except Exception as e:
            raise e
        finally:
            for k, v in original_config_options.items():
                gdal.SetConfigOption(k, v)


def get_time(start: datetime, end: datetime) -> str:
    """
    Получение времени между двумя промежутками типа datetime.
    На выходе строка в формате %H:%M:%S либо в долях секунды.
    """
    time_delta = end - start
    seconds = time_delta.total_seconds()
    if seconds < 1:
        return "{:.3f}s".format(seconds)
    else:
        return time.strftime("%H:%M:%S", time.gmtime(seconds))
