from __future__ import annotations

from collections.abc import Iterable, Generator
from typing import cast

import axipy
from PySide2.QtCore import Signal
from axipy._internal._observer_abc import _ObserverABC

from .utils import SUPPORTED_TYPES, _init_geom_generator


class SpatialFilterObserver(_ObserverABC):
    @property
    def name(self) -> str:
        return "ru_axioma_gis_spatial_filter_SpatialFilterObserver"

    @property
    def signals(self) -> Signal | Iterable[Signal]:
        return (
            axipy.ObserverManager.Selection.changed,
            axipy.ObserverManager.ActiveMapView.changed,
            axipy.selection_manager.changed,
        )

    def test_value(self) -> bool:
        return (
                axipy.ObserverManager.Selection.value and
                axipy.ObserverManager.ActiveMapView.value and
                self._test_value()
        )

    @staticmethod
    def selection_table() -> axipy.SelectionTable:
        """Ensured by observer."""
        return axipy.data_manager.selection

    @staticmethod
    def active_map_view() -> axipy.MapView:
        """Ensured by observer."""
        return axipy.view_manager.active

    def _test_value(self) -> bool:
        # Can't be None, because of Observer.
        selection_table: axipy.SelectionTable = axipy.data_manager.selection

        count: int = selection_table.count()
        if count == 1:
            f = next(selection_table.items())
            g = f.geometry
            if g is None:
                return False
            return self._check_geom(g)

        elif count > 1:
            return True

        return False

    @classmethod
    def get_tables_from_active_map_view(cls) -> Generator[axipy.Table]:
        active_view = axipy.view_manager.active
        # Must be MapView, because of Observer.
        map_view = cast(axipy.MapView, active_view)
        list_layers = map_view.map.layers
        yield from cls._get_tables_from_list_layers(list_layers)

    @classmethod
    def _get_tables_from_list_layers(cls, list_layers: axipy.ListLayers) -> Generator[axipy.Table]:
        for elem in list_layers:
            if isinstance(elem, axipy.VectorLayer):
                yield elem.data_object
            elif isinstance(elem, axipy.ListLayers):
                yield from cls._get_tables_from_list_layers(elem)

    @classmethod
    def check_if_only_supported_types_selected(cls, task: axipy.DialogTask) -> bool:
        task.title = "Проверка типов геометрии"
        task.message = "Проверка наличия поддерживаемых типов геометрии в выборке."

        # Can't be None, because of Observer.
        selection_table = axipy.data_manager.selection

        return all(map(cls._check_geom, _init_geom_generator(selection_table.items(), selection_table.count(), task)))

    @staticmethod
    def _check_single_geom(g: axipy.Geometry) -> bool:
        if isinstance(g, SUPPORTED_TYPES):
            return True

        return False

    @classmethod
    def _check_geom(cls, g: axipy.Geometry) -> bool:
        if isinstance(g, axipy.GeometryCollection):
            g = cast(Iterable, g)
            return any(map(lambda elem: cls._check_single_geom(elem), g))

        return cls._check_single_geom(g)
