import math
from pathlib import Path
from typing import Optional, Iterator, Tuple, TYPE_CHECKING, cast

import axipy
from PySide2.QtCore import Qt, Slot, Signal, QItemSelection
from PySide2.QtGui import QIcon, QKeyEvent
from PySide2.QtWidgets import (QFileDialog, QDialog, QDialogButtonBox, QWidget, QProgressBar,
                               QStackedWidget, QTableWidgetItem, QPushButton)
from axipy import (tr, CurrentSettings, provider_manager,
                   CoordSystem, Notifications, AxipyAnyCallableTask, DefaultSettings,
                   data_manager, Table, Schema, Feature, view_manager, Map, Layer, MapView, gui_instance,
                   View, Point,
                   selection_manager, task_manager)
from .constants import Col, POINT_STYLE_MI, SOURCE_RANGE, TARGET_RANGE
from .exporter import Exporter
from .model import Model
from .qtransformcalculator import QTransformCalculator
from .ui import Ui_Dialog
from .utils import DictPnt, Pnt, try_except_silent, try_except, filter_one_input_table, filter_one_map_view, \
    TEMPORARY_MAP_VIEW, ATTRS

if TYPE_CHECKING:
    from .__init__ import RegisterVector


class RegisterVectorDialog(QDialog, Ui_Dialog):
    signal_notify = Signal(str, int)

    signal_pbar_features_set_max = Signal(int)
    signal_pbar_features_add_value = Signal()

    def __init__(self, plugin: 'RegisterVector', parent=None) -> None:
        super().__init__(parent)
        self.setupUi(self)
        # self.setAttribute(Qt.WA_DeleteOnClose, True)

        self._plugin: RegisterVector = plugin
        self._title: str = self._plugin.tr("Привязка вектора")

        # ui file

        self._pbar_range_init: bool = False

        self._task: Optional[AxipyAnyCallableTask] = None

        self._model: Optional[Model] = None

        self._current_coordinate_tables: Optional[Pnt] = None
        self._current_coordinate_maps: Optional[Pnt] = None

        self._current_tmp_table_tables: Optional[Table] = None
        self._current_tmp_table_maps: Optional[Table] = None

        self._current_map_view_tables: Optional[MapView] = None
        self._current_map_view_maps: Optional[MapView] = None

        self.signal_notify.connect(self.slot_notify)

        self.signal_pbar_features_set_max.connect(self.slot_pbar_features_set_max)
        self.signal_pbar_features_add_value.connect(self.slot_pbar_features_add_value)

        self.init_ui()

    def init_ui(self) -> None:
        self.pbar_features.hide()

        self.setWindowTitle(self._title)

        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"
        self.setWindowFlags(self.windowFlags() ^ Qt.WindowMaximizeButtonHint)  # Добавление "Раскрыть"

        self.tb_select_save_file.setIcon(QIcon.fromTheme("save"))
        self.tb_add_row.setIcon(QIcon.fromTheme("add"))
        self.tb_add_row.clicked.connect(self.slot_add_row)
        self.tb_delete_row.setIcon(QIcon.fromTheme("delete"))
        self.tb_delete_row.clicked.connect(self.slot_delete_row)

        self.model = Model()
        self.model.dataChanged.connect(self.slot_model_data_changed)
        self.model.signal_row_count_changed.connect(self.slot_model_row_count_changed)
        self.table_view_widget_coordinates.setModel(self.model)

        self.tb_select_save_file.clicked.connect(self.slot_save_file_name)

        self.cb_inp_tables.currentTextChanged.connect(self.slot_current_text_changed_cb_inp_tables)
        self.cb_out_maps.currentTextChanged.connect(self.slot_current_text_changed_cb_out_maps)

        self._ui_ok.setText(self._plugin.tr("Трансформировать"))
        self._ui_ok.setEnabled(False)

        self._ui_cancel.setText(self._plugin.tr("Закрыть"))

    @property
    def _ui_ok(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Ok)

    @property
    def _ui_cancel(self) -> QPushButton:
        return self.buttonBox.button(QDialogButtonBox.Cancel)

    @property
    def model(self) -> Model:
        return self._model

    @model.setter
    def model(self, value: Model) -> None:
        self._model = value

    @property
    def current_coordinate_tables(self) -> Pnt:
        return self._current_coordinate_tables

    @current_coordinate_tables.setter
    def current_coordinate_tables(self, value: Pnt):
        self._current_coordinate_tables = value

    @property
    def current_coordinate_maps(self) -> Pnt:
        return self._current_coordinate_maps

    @current_coordinate_maps.setter
    def current_coordinate_maps(self, value: Pnt):
        self._current_coordinate_maps = value

    @property
    def current_tmp_table_tables(self) -> Table:
        return self._current_tmp_table_tables

    @current_tmp_table_tables.setter
    def current_tmp_table_tables(self, value: Table):
        table = self._current_tmp_table_tables
        if table is not None:
            table.rollback()
            table.close()
        self._current_tmp_table_tables = value

    @current_tmp_table_tables.deleter
    def current_tmp_table_tables(self) -> None:
        table = self._current_tmp_table_tables
        if table is not None:
            table.rollback()
            table.close()
            self._current_tmp_table_tables = None

    @property
    def current_tmp_table_maps(self) -> Table:
        return self._current_tmp_table_maps

    @current_tmp_table_maps.setter
    def current_tmp_table_maps(self, value: Table):
        table = self._current_tmp_table_maps
        if table is not None:
            table.rollback()
            table.close()
        self._current_tmp_table_maps = value

    @current_tmp_table_maps.deleter
    def current_tmp_table_maps(self) -> None:
        table = self._current_tmp_table_maps
        if table is not None:
            table.rollback()
            table.close()
            self._current_tmp_table_maps = None

    @property
    def current_map_view_tables(self) -> MapView:
        return self._current_map_view_tables

    @current_map_view_tables.setter
    def current_map_view_tables(self, value: MapView):
        self._current_map_view_tables = value

    @current_map_view_tables.deleter
    def current_map_view_tables(self) -> None:
        self._current_map_view_tables.widget.deleteLater()
        self._current_map_view_tables = None

    @property
    def current_map_view_maps(self) -> MapView:
        return self._current_map_view_maps

    @current_map_view_maps.setter
    def current_map_view_maps(self, value: MapView):
        self._current_map_view_maps = value

    @current_map_view_maps.deleter
    def current_map_view_maps(self) -> None:
        self._current_map_view_maps.widget.deleteLater()
        self._current_map_view_maps = None

    def refresh_ui(self) -> None:
        # reverse order, first tmp map added to maps list
        self.update_cb_out_maps()
        self.update_cb_inp_tables()

    def keyPressEvent(self, event: QKeyEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self.slot_delete_row()
        super().keyPressEvent(event)

    def accept(self) -> None:
        try:
            self.export_process_prepare()
        except (Exception,):
            Notifications.push(self._title, "Не удалось запустить процесс конвертации", Notifications.Critical)
            self.slot_on_finished()

    def open(self) -> None:
        selection_manager.changed.connect(self.slot_selection_changed)

        self.refresh_ui()
        super().open()

    def reject(self) -> None:
        selection_manager.changed.disconnect(self.slot_selection_changed)

        self._reject()

        if (
                self._task is not None and
                self._task.progress_handler().is_running() and
                not self._task.progress_handler().is_canceled()
        ):
            self._task.progress_handler().cancel()
            return None

        if self._task is None or not self._task.progress_handler().is_running():
            super().reject()

    @try_except_silent()
    def _reject(self) -> None:
        for stack in (self.stacked_widget_tables, self.stacked_widget_maps):
            self.clear_stack_widget(stack)

        del self.current_tmp_table_tables
        del self.current_tmp_table_maps

    @Slot()
    def slot_save_file_name(self) -> None:
        name, _ = QFileDialog.getSaveFileName(filter="Файлы TAB (*.tab)", )
        if name:
            self.le_save_file_name.setText(name)

        self.check_enabled_export()

    @staticmethod
    def clear_stack_widget(stack: QStackedWidget) -> None:
        while stack.count():
            w = stack.currentWidget()
            stack.removeWidget(w)
            w.deleteLater()

    def update_cb_inp_tables(self) -> None:
        cb = self.cb_inp_tables
        cb.clear()
        cb.addItems(
            table.name for table in data_manager.tables
            if filter_one_input_table(table)
        )

    def update_cb_out_maps(self) -> None:
        cb = self.cb_out_maps
        cb.clear()
        cb.addItems(
            v.title for v in view_manager.mapviews
            if filter_one_map_view(v)
        )

    @Slot()
    def slot_model_data_changed(self) -> None:
        self.rebuild_tables_from_table_widget()
        self.rebuild_errors()

    @Slot()
    def slot_model_row_count_changed(self) -> None:
        self.rebuild_tables_from_table_widget()
        self.rebuild_errors()

    def rebuild_errors(self) -> None:
        points_table: DictPnt = self.read_points_from_range(SOURCE_RANGE)
        points_map: DictPnt = self.read_points_from_range(TARGET_RANGE)

        transformer = QTransformCalculator(points_table, points_map)
        result: QTransformCalculator.Result = transformer.transform()

        if result is None:
            for row in range(self.model.rowCount()):
                self.model.grid_set(row, Col.Error, None)
            return None
        else:
            for k, v in result.errors_dict.items():
                self.model.grid_set(k, Col.Error, v)

    @staticmethod
    def _internal_create_map_view_from_map(map_: Map) -> MapView:
        # Method uses internal undocumented function and should not be used in user plugins
        shadow_map = map_._shadow
        mv = View._wrap(gui_instance._shadow.createMapView(shadow_map, False))
        mv.title = TEMPORARY_MAP_VIEW
        return mv

    @Slot(str)
    def slot_current_text_changed_cb_inp_tables(self, str_arg: str) -> None:
        stack = self.stacked_widget_tables
        self.clear_stack_widget(stack)

        self.add_to_stack_tables(str_arg, stack)

        self.rebuild_table_from_table_widget(self.current_tmp_table_tables, SOURCE_RANGE)

    @Slot(str)
    def slot_current_text_changed_cb_out_maps(self, str_arg: str) -> None:
        stack = self.stacked_widget_maps
        self.clear_stack_widget(stack)

        self.add_to_stack_maps(str_arg, stack)

        self.rebuild_table_from_table_widget(self.current_tmp_table_maps, TARGET_RANGE)

    @Slot()
    def slot_selection_changed(self) -> None:
        self._selection()

    @try_except_silent()
    def _selection(self) -> None:
        selection_model = self.table_view_widget_coordinates.selectionModel()
        selection_model.clearSelection()

        table = selection_manager.table
        if table is None:
            return None

        ids = selection_manager.ids
        for f in table.items(ids=ids):
            row = f[ATTRS.row.name]
            x_col = f[ATTRS.x_col.name]
            y_col = f[ATTRS.y_col.name]

            indx_x = self.model.createIndex(row, x_col)
            indx_y = self.model.createIndex(row, y_col)
            s = QItemSelection(indx_x, indx_y)
            selection_model.select(s, selection_model.Select)

    @staticmethod
    def create_temporary_table_as_cosmetic_layer(cs: CoordSystem) -> Table:
        destination = provider_manager.shp.get_destination('', Schema(*ATTRS, coordsystem=cs))
        destination["hidden"] = True
        table = destination.create_open()
        return table

    @staticmethod
    def add_tmp_table_to_map_as_upper_layer(tmp_table: Table, map_: Map) -> None:
        tmp_layer = Layer.create(tmp_table)
        layers = map_.layers
        layers.append(tmp_layer)
        n = len(layers)
        layers.move(n - 1, 0)

    @try_except_silent()
    def add_to_stack_tables(self, str_arg: str, stack: QStackedWidget) -> None:
        data_object: Table = data_manager.find(str_arg)
        if data_object is None:
            return None

        layer = Layer.create(data_object)
        try:
            layer.selectable = False
        except (Exception,):
            pass
        map_ = Map([layer])
        mv = self._internal_create_map_view_from_map(map_)

        cloned_map = mv.map
        tmp_table = self.create_temporary_table_as_cosmetic_layer(mv.coordsystem)
        self.current_tmp_table_tables = tmp_table
        self.add_tmp_table_to_map_as_upper_layer(tmp_table, cloned_map)

        self.current_map_view_tables = mv
        cast(Signal, mv.mouse_moved).connect(self.slot_mouse_moved_tables)
        stack.addWidget(mv.widget)

    @try_except_silent()
    def add_to_stack_maps(self, str_arg: str, stack: QStackedWidget) -> None:
        for mv in view_manager.mapviews:
            if mv.title == str_arg:
                new_mv = self._internal_create_map_view_from_map(mv.map)

                cloned_map = new_mv.map

                for layer in cloned_map.layers:
                    try:
                        layer.selectable = False
                    except (Exception,):
                        pass

                tmp_table = self.create_temporary_table_as_cosmetic_layer(new_mv.coordsystem)
                self.current_tmp_table_maps = tmp_table
                self.add_tmp_table_to_map_as_upper_layer(tmp_table, cloned_map)

                self.current_map_view_maps = new_mv
                cast(Signal, new_mv.mouse_moved).connect(self.slot_mouse_moved_maps)
                stack.addWidget(new_mv.widget)

    @Slot(float, float)
    def slot_mouse_moved_tables(self, x: float, y: float):
        self.current_coordinate_tables = Pnt(x, y)

    @Slot(float, float)
    def slot_mouse_moved_maps(self, x: float, y: float):
        self.current_coordinate_maps = Pnt(x, y)

    def get_point_from_position(self, row, col_range: Tuple[int, int]) -> Optional[Pnt]:
        model = self.model
        coords = []
        for col in col_range:
            value = model.grid_get(row, col)
            if value is None:
                return None
            coords.append(value)
        pnt = Pnt(coords[0], coords[1])
        return pnt

    def get_float_from_table_widget_item(self, item: QTableWidgetItem) -> Optional[float]:
        if item is None:
            return None

        data = item.data(Qt.UserRole)
        if data is None:
            data = self.convert_string_to_float(item.text())

        return data

    @try_except("Не удалось преобразовать строку в вещественное число")
    def convert_string_to_float(self, str_arg: str) -> Optional[float]:
        return float(str_arg)

    @staticmethod
    def find_existing_feature(row: int, col_range: Tuple[int, int], table: Table) -> Optional[Feature]:
        for f in table.items():
            x_col = col_range[0]
            y_col = col_range[1]
            if row == f[ATTRS.row.name] and f[ATTRS.x_col.name] == x_col and f[ATTRS.y_col.name] == y_col:
                return f

    @staticmethod
    def create_feature(range_: Tuple[int, int], pnt: Pnt, row: int) -> Feature:
        f_dict = dict(zip((attr.name for attr in ATTRS), (row, *range_)))
        point = Point(*pnt)
        f = Feature(f_dict, geometry=point, style=POINT_STYLE_MI)
        return f

    def add_coordinates(self, range_: Tuple[int, int], pnt: Pnt) -> None:
        self.model.grid_add_point(range_, pnt)

    def __get_precision_from_map_view(self, mw: axipy.MapView) -> int:
        w_scene = mw.scene_rect.width
        w_device = mw.device_rect.width
        pixel_size = w_device / w_scene
        n_digits = int(math.log10(pixel_size * 1e16)) - 16
        return n_digits

    def __round_to_precision(self, value: float, precision: int) -> float:
        if precision == 0:
            precision = None
        return round(value, precision)

    def __correct_pnt_precision(self, pnt: axipy.Pnt, mv: axipy.MapView) -> axipy.Pnt:
        p = self.__get_precision_from_map_view(mv)
        return axipy.Pnt(self.__round_to_precision(pnt.x, p), self.__round_to_precision(pnt.y, p))

    def add_coordinates_tables(self) -> None:
        pnt = self.current_coordinate_tables
        mv = self.current_map_view_tables
        pnt = self.__correct_pnt_precision(pnt, mv)
        self.add_coordinates((Col.x1, Col.y1), pnt)

    def add_coordinates_maps(self) -> None:
        pnt = self.current_coordinate_maps
        mv = self.current_map_view_maps
        pnt = self.__correct_pnt_precision(pnt, mv)
        self.add_coordinates((Col.x2, Col.y2), pnt)

    @Slot()
    def slot_add_row(self) -> int:
        tb = self.model

        row = tb.rowCount()
        tb.insertRow(row)

        self.check_enabled_export()
        return row

    @Slot()
    def slot_delete_row(self) -> None:
        view = self.table_view_widget_coordinates

        selected_rows = []
        for index in view.selectionModel().selectedRows():
            row = index.row()
            if row not in selected_rows:
                selected_rows.append(row)

        for row in sorted(selected_rows, reverse=True):
            self.model.removeRow(row)

        self.check_enabled_export()

    def rebuild_tables_from_table_widget(self) -> None:
        self.rebuild_table_from_table_widget(self.current_tmp_table_tables, SOURCE_RANGE)
        self.rebuild_table_from_table_widget(self.current_tmp_table_maps, TARGET_RANGE)

    def rebuild_table_from_table_widget(self, table: Table, columns_range) -> None:
        # todo: table is None
        if table is None:
            return None

        table.rollback()
        features = []
        for row in range(self.model.rowCount()):
            pnt = self.get_point_from_position(row, columns_range)
            if pnt is None:
                continue
            f = self.create_feature(columns_range, pnt, row)
            features.append(f)
        table.insert(features)

    def read_points_from_range(self, columns_range: Tuple[int, int]) -> DictPnt:
        points: DictPnt = {}

        for row in range(self.model.rowCount()):
            pnt = self.get_point_from_position(row, columns_range)
            if pnt is None:
                continue
            points[row] = pnt

        return points

    @staticmethod
    def process_table_widget_item(item: QTableWidgetItem) -> str:
        if item is None:
            return ""
        else:
            return item.text()

    def check_enabled_export(self) -> None:
        # todo check points length
        if (
                self.le_save_file_name.text()
        ):
            enabled = True
        else:
            enabled = False

        self._ui_ok.setEnabled(enabled)

    @staticmethod
    def get_existing_last_save_path() -> Path:
        last_save_path = CurrentSettings.LastSavePath
        if not Path(last_save_path).exists():
            last_save_path = DefaultSettings.LastSavePath
        return last_save_path

    def toggle_ui(self, state: bool) -> None:
        objects = self.children()
        widgets: Iterator[QWidget] = filter(lambda x: isinstance(x, QWidget), objects)
        for widget in widgets:
            if widget == self.buttonBox:
                continue
            widget.setEnabled(state)

        self._ui_ok.setEnabled(state)

    @staticmethod
    def prepare_pbar(pbar: QProgressBar):
        """Делает прогресс бесконечным, для первого файла."""
        pbar.setValue(0)
        pbar.setMaximum(0)
        pbar.show()

    @Slot(int)
    def slot_pbar_features_set_max(self, max_: int) -> None:
        self.pbar_features.setMaximum(max_)

    @Slot()
    def slot_pbar_features_add_value(self) -> None:
        pbar = self.pbar_features
        v = pbar.value()
        pbar.setValue(v + 1)

    @Slot(str, int)
    def slot_notify(self, msg: str, msg_type: axipy.Notifications) -> None:
        Notifications.push(self._title, msg, msg_type)

    def export_process_prepare(self) -> None:
        self.toggle_ui(False)
        self._ui_ok.setEnabled(False)
        self._ui_cancel.setText(tr("Отмена"))

        for pbar in (self.pbar_features,):
            self.prepare_pbar(pbar)

        points_table: DictPnt = self.read_points_from_range(SOURCE_RANGE)
        points_map: DictPnt = self.read_points_from_range(TARGET_RANGE)

        transformer = QTransformCalculator(points_table, points_map)

        q_transform = transformer.calculate()
        if q_transform is None:
            self.slot_on_finished()

        data_object = data_manager.find(self.cb_inp_tables.currentText())
        target_cs = self.current_map_view_maps.coordsystem

        exporter = Exporter(
            inp_table=data_object,
            target_cs=target_cs,
            q_transform=q_transform,
            out_file_path=Path(self.le_save_file_name.text()),
            widget_dialog=self,
        )

        if self._task is None:
            self._task = AxipyAnyCallableTask(exporter.export)
            self._task.progress_handler().finished.connect(self.slot_on_finished)
            task_manager.start_task(self._task)

    @Slot()
    def slot_on_finished(self) -> None:
        self.pbar_features.hide()
        self._ui_cancel.setText(tr("Закрыть"))

        self.toggle_ui(True)
        self.check_enabled_export()

        if self._task is not None:
            task_result: Exporter.Result = self._task.progress_handler().result
            if task_result is not None:
                Notifications.push(
                    self._title,
                    self._plugin.tr(f"Конвертация завершена. "
                                    f"Сконвертировано записей: "
                                    f"{task_result.successfully_exported_elements} из {task_result.all_elements}."),
                )
            self._task.progress_handler().finished.disconnect(self.slot_on_finished)
            self._task = None
