from __future__ import annotations

import traceback
from collections.abc import Iterator, Iterable, Generator
from dataclasses import dataclass
from enum import Enum, auto
from pathlib import Path
from typing import TYPE_CHECKING

import axipy
from axipy._internal._decorator import _run_in_gui_decor
from .utils import NotifyResultsMixIn, ProgressWithStepMixIn

if TYPE_CHECKING:
    from .__init__ import ExportToFilePlugin


class ExportToFileWorker(axipy.Task, NotifyResultsMixIn, ProgressWithStepMixIn):
    @dataclass
    class TableMeta:
        table: axipy.Table
        file_name_path: Path
        close_later: bool = False

    class TargetFileExistsError(FileExistsError):
        ...

    class ExportMode(Enum):
        DEFAULT = auto()
        """Обычный экспорт, со сменой проекции и пересчетом координат."""
        PRESERVE_GEOMETRY = auto()
        """Не пересчитывать координаты при смене проекции."""
        OVERRIDE_INP_CS = auto()
        """Переопределить входную проекцию."""

    def __init__(
            self,
            plugin: 'ExportToFilePlugin',
            *,
            paths: list[Path],
            out_path: Path,
            preserve_geometry: bool,
            inp_cs_override: axipy.CoordSystem | None,
            out_cs: axipy.CoordSystem,
    ) -> None:
        super().__init__(self._do_work)
        self.plugin: 'ExportToFilePlugin' = plugin

        self._paths: list[Path] = paths
        self._out_folder: Path = out_path
        self._preserve_geometry: bool = preserve_geometry
        self._inp_cs_override: axipy.CoordSystem | None = inp_cs_override
        self._override_inp_cs: bool = inp_cs_override is not None
        self._out_cs: axipy.CoordSystem = out_cs

        self._inp_meta_tables: list[ExportToFileWorker.TableMeta] = []
        self._export_mode: ExportToFileWorker.ExportMode | None = None

        self._successful_count: int = 0

    def _do_work(self) -> None:
        self.title = self.plugin.title
        self.message = self.title

        try:
            self._do_work_impl()
        except self.CanceledException:
            self._notify_canceled()
        except Exception as e:
            traceback.print_exc()
            self._notify_error(e)
        finally:
            self._close_tmp_tables()

    def _close_tmp_tables(self) -> None:
        for meta in self._inp_meta_tables:
            if meta.close_later:
                meta.table.close()

    @_run_in_gui_decor
    def _notify_canceled(self) -> None:
        axipy.Notifications.push(self.plugin.title, "Отмена.", axipy.Notifications.Warning)

    @_run_in_gui_decor
    def _notify_error(self, e) -> None:
        axipy.Notifications.push(self.plugin.title, f"Ошибка: '{e}'.", axipy.Notifications.Critical)

    def _do_work_impl(self) -> None:
        self.title = self.plugin.title
        self.message = self.title

        self.range = self.Range(0, 0)
        self._export_mode = self._determine_mode()

        # Save, to close tmp later.
        self._inp_meta_tables = list(self._get_table_meta_gen())

        tables: list[axipy.Table] = [meta.table for meta in self._inp_meta_tables]
        all_count: int = len(tables)

        if self._export_mode == self.ExportMode.PRESERVE_GEOMETRY:
            range_max = all_count
        else:
            range_max = sum(table.count() for table in tables)
        # Fix range
        range_max = self.init_progress_with_step(self, range_max)
        self.range = self.Range(0, range_max)

        for meta_table in self._inp_meta_tables:
            self._export_one_meta_table(meta_table)

        self.notify_skipped(self.plugin.title)
        self.notify_exceptions(self.plugin.title)
        self.notify_results(self.plugin.title, self._successful_count, all_count)

    def _determine_mode(self) -> ExportMode:
        if self._preserve_geometry:
            return self.ExportMode.PRESERVE_GEOMETRY
        elif self._override_inp_cs:
            return self.ExportMode.OVERRIDE_INP_CS
        else:
            return self.ExportMode.DEFAULT

    def _override_inp_cs_gen(self, features: Iterable[axipy.Feature]) -> Generator[axipy.Feature]:
        for f in features:
            g = f.geometry
            if g is not None:
                g.coordsystem = self._inp_cs_override
                f.geometry = g
            yield f

    def _check_table_opened(self, file_name_target: Path) -> axipy.Table | None:
        """Поиск уже открытой таблицы."""
        all_tables: Iterator[axipy.Table] = filter(
            lambda obj: isinstance(obj, axipy.Table), axipy.data_manager.all_objects)

        found: axipy.Table | None = None
        for table in all_tables:
            file_name = table.file_name
            if file_name and file_name == file_name_target:
                found = table
                break
        return found

    def _get_table_meta_gen(self) -> Generator[TableMeta]:
        for path in self._paths:
            try:
                table_meta = self._get_table_meta(path)
            except Exception as e:
                traceback.print_exc()
                self.check_exception(e)
            else:
                yield table_meta

    def _get_table_meta(self, path: Path) -> TableMeta:
        # Поиск уже открытой таблицы
        table = self._check_table_opened(path)

        # Открытие временной таблицы
        if table is None:
            table = self._get_source_tab_from_path(path, hidden=True).open()
            close_later = True
        else:
            close_later = False

        if table is None:
            raise RuntimeError

        return self.TableMeta(table, path, close_later)

    def _export_one_meta_table(
            self,
            meta_table: TableMeta
    ) -> None:
        try:
            self._export_one_meta_table_impl(meta_table)
        except self.TargetFileExistsError:
            pass
        except self.CanceledException:
            raise self.CanceledException
        except Exception as e:
            print(f"Ошибка конвертации таблицы '{meta_table.table.name}': {e}")
            self.check_exception(e)
        else:
            self._successful_count += 1

    def _export_one_meta_table_impl(
            self,
            meta_table: TableMeta
    ) -> None:
        file_name = self._construct_out_file_name(meta_table.file_name_path)

        if self.check_skipped(file_name):
            raise self.TargetFileExistsError

        export_mode = self._export_mode
        if export_mode == self.ExportMode.PRESERVE_GEOMETRY:
            self._export_preserve_geometry_single(meta_table)
        elif export_mode == self.ExportMode.OVERRIDE_INP_CS:
            self._export_override_inp_cs_single(meta_table)
        elif export_mode == self.ExportMode.DEFAULT:
            self._export_default_single(meta_table)
        else:
            raise RuntimeError

    def _export_one_table_copy_and_change(self, inp_file_name: Path, out_cs_rect: axipy.Rect) -> None:
        if not self._preserve_geometry:  # Guard
            raise RuntimeError
        out_file_name = self._construct_out_file_name(inp_file_name)
        axipy.provider_manager.tab.copy_table_files(inp_file_name, out_file_name)
        self._out_cs.rect = out_cs_rect
        axipy.provider_manager.tab.change_coordsystem(out_file_name, self._out_cs)

    def _get_source_tab_from_path(self, path: Path, hidden: bool = True) -> axipy.Source:
        source = axipy.provider_manager.tab.get_source(str(path))
        source["hidden"] = hidden
        return source

    def _get_destination_tab_from_path(self, path: Path, schema: axipy.Schema = None) -> axipy.Destination:
        if schema is None:
            schema = axipy.Schema()
        destination = axipy.provider_manager.tab.get_destination(str(path), schema)
        return destination

    def _construct_out_file_name(self, inp_file_name: Path) -> Path:
        return self._out_folder / inp_file_name.name

    def _export_preserve_geometry_single(self, meta_table: TableMeta) -> None:
        self.raise_if_canceled()
        self._export_one_table_copy_and_change(meta_table.file_name_path, meta_table.table.coordsystem.rect)
        self.add_value_progress_with_step()

    def _init_feature_gen_with_progress(self, table: axipy.Table) -> Generator[axipy.Feature]:
        for f in table.items():
            self.raise_if_canceled()
            self.add_value_progress_with_step()
            yield f

    def _export_override_inp_cs_single(self, meta_table: TableMeta) -> None:
        features = self._init_feature_gen_with_progress(meta_table.table)
        features = self._override_inp_cs_gen(features)
        self._export_features_common(features, meta_table.table.schema, meta_table.file_name_path)

    def _export_default_single(self, meta_table: TableMeta) -> None:
        features = self._init_feature_gen_with_progress(meta_table.table)
        self._export_features_common(features, meta_table.table.schema, meta_table.file_name_path)

    def _copy_schema_change_cs(self, schema: axipy.Schema, cs: axipy.CoordSystem) -> axipy.Schema:
        # todo: simplify.
        attrs = [schema.by_name(name) for name in schema.attribute_names]
        new_schema = axipy.Schema(*attrs, coordsystem=cs)
        return new_schema

    def _export_features_common(
            self,
            features: Iterable[axipy.Feature],
            out_schema: axipy.Schema,
            inp_file_name: Path,
    ) -> None:
        out_schema = self._copy_schema_change_cs(out_schema, self._out_cs)

        out_file_name = self._construct_out_file_name(inp_file_name)
        destination = self._get_destination_tab_from_path(out_file_name, out_schema)
        destination.export(features)
