from typing import List

import axipy
from axipy.da import Feature, GeometryType

from .data_types import SearchGeometries

geom_translator = {
    GeometryType.Point: "Точка",
    GeometryType.Line: "Линия",
    GeometryType.LineString: "Полилиния",
    GeometryType.Polygon: "Полигон",
    GeometryType.MultiPoint: "Коллекция точек",
    GeometryType.MultiLineString: "Коллекция полилиний",
    GeometryType.MultiPolygon: "Коллекция полигонов",
    GeometryType.GeometryCollection: "Смешанная коллекция",
    GeometryType.Arc: "Дуга",
    GeometryType.Ellipse: "Эллипс",
    GeometryType.Rectangle: "Прямоугольник",
    GeometryType.RoundedRectangle: "Скругленный прямоугольник",
    GeometryType.Text: "Текст",
}

geom_type_to_sql_geom_name = {
    GeometryType.Point: "Point",
    GeometryType.Line: "Line",
    GeometryType.LineString: "Polyline",
    GeometryType.Polygon: "Region",
    GeometryType.MultiPoint: "Multipoint",
    GeometryType.MultiLineString: "Polyline",
    GeometryType.MultiPolygon: "Region",
    GeometryType.GeometryCollection: "Collection",
    GeometryType.Arc: "Arc",
    GeometryType.Ellipse: "Ellipse",
    GeometryType.Rectangle: "Rectangle",
    GeometryType.RoundedRectangle: "Rounded Rectangle",
    GeometryType.Text: "Text",
}


def geom_type_to_str(gtype: GeometryType) -> str:
    return geom_translator[gtype]


def str_to_geom_type(gtype: str) -> GeometryType:
    for key, value in geom_translator.items():
        if gtype == value:
            return key
    raise RuntimeError(axipy.tr("Неподдерживаемый тип геометрии"))


def to_sql_style_str(f: Feature) -> str:
    """
    Замененяет Pen на Line для площадных объектов

    В коллеркциях чтобы отличать стиль линии от стиля обводки полигона
    название стиля обводки задается как Line.  В то время как str(feature.style)
    возвращает Pen. Пока решено исправить эту проблему заменой для площадных
    объектов
    """
    assert f.geometry
    if f.geometry.type in [
        GeometryType.Polygon,
        GeometryType.MultiPolygon,
        GeometryType.Ellipse,
        GeometryType.Rectangle,
        GeometryType.RoundedRectangle,
    ]:
        return str(f.style).replace("Pen", "Line")
    return str(f.style)


def make_geometries_str(search_geometries: SearchGeometries) -> str:
    geometries = ""
    for gtype in geometry_types_from_settings(search_geometries):
        if len(geometries) != 0:
            geometries += ", "
        geometries += f"'{geom_type_to_sql_geom_name[gtype]}'"
    return geometries


def geometry_types_from_settings(search_geometries: SearchGeometries) -> List[GeometryType]:
    """
    Возвращает список геометрий для которые были выбраны пользователем для поиска
    """
    return [gtype for gtype, ok in search_geometries.items() if ok]
