"""
Корневой модуль.
"""

from typing import Any

from axipy.cpp_core_core import DefaultSettingsList, ShadowSettings

__all__ = ["Settings"]


class Settings(DefaultSettingsList):
    """Настройки ГИС Аксиома.

    Класс не требует создания объекта. Используйте методы класса.

    .. literalinclude:: /../../tests/doc_examples/test_example_settings.py
        :caption: Пример использования
        :pyobject: example_settings
        :lines: 2-
        :dedent: 4


    .. csv-table:: Атрибуты
        :header: Значение, Тип, Наименование

        SilentCloseWidget, :class:`bool`, Подтверждать закрытие несохраненных данных
        SnapSensitiveRadius, :class:`int`, Привязка узлов - размер
        EditNodeColor, :class:`~PySide2.QtGui.QColor`, Узлы при редактировании - цвет
        EditNodeSize, :class:`int`, Узлы при редактировании - размер
        NearlyGeometriesTopology, :class:`bool`, Перемещать узлы соседних объектов при редактировании
        NodesUpdateMode, :class:`bool`, Объединять историю изменения узлов в режиме Форма
        ShowDrawingToolTip, :class:`bool`, Показывать данные при рисовании
        CreateTabAfterOpen, :class:`bool`, Создавать TAB при открытии
        RenameDataObjectFromTab, :class:`bool`, Переименовывать открытый объект по имени TAB файла
        LastSavePath, :class:`str`, Последний пусть сохранения
        UseLastSelectedFilter, :class:`bool`, Запоминать последний фильтр в диалоге открытия файлов
        SelectByInformationTool, :class:`bool`, Инструмент «Информация» выбирает объект
        SaveAsToOriginalFileFolder, :class:`bool`, Сохранять копию в каталог с исходным файлом
        LastNameFilter, :class:`str`, Последний использованный фильтр файлов
        SensitiveMouse, :class:`int`, Чувствительность мыши в пикселях
        ShowSplashScreen, :class:`bool`, Отображать экран загрузки
        RulerColorLine, :class:`QColor`, Линейка - цвет линии
        UseAntialiasing, :class:`bool`, Использовать сглаживание при отрисовке
        ShowDegreeTypeNumeric, :class:`bool`, Отображать градусы в формате Десятичное значение
        DrawCoordSysBounds, :class:`bool`, Отображать границы мира
        PreserveScaleMap, :class:`bool`, Сохранять масштаб при изменении размеров окна
        ShowMapScaleBar, :class:`bool`, Показывать масштабную линейку
        ShowScrollOnMapView, :class:`bool`, Показывать полосы прокрутки
        LoadLastWorkspace, :class:`bool`, Загружать при старте последнее рабочее пространство
        ShowMeshLayout, :class:`bool`, Отображать сетку привязки
        MeshSizeLayout, :class:`float`, Размер ячейки
        SnapToMeshLayout, :class:`bool`, Привязывать элементы отчета к сетке
        ShowMeshLegend, :class:`bool`, Отображать сетку привязки
        MeshSizeLegend, :class:`float`, Размер ячейки
        SnapToMeshLegend, :class:`bool`, Привязывать к сетке
        LastOpenPath, :class:`str`, Последний каталог откуда открывались данные
        LastPathWorkspace, :class:`str`, Последний каталог к рабочему набору
        DefaultPathCache, :class:`str`, Каталог с кэшированными данными
        UserDataPaths, :class:`list[str]`, Список пользовательских каталогов с названиями
        EnableSmartTabs, :class:`bool`, Умное переключение вкладок
        DistancePrecision, :class:`int`, Точность по умолчанию для расстояний и площадей

    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def value(cls, key) -> Any:
        """Читает значение параметра.

        Args:
            key: Параметр.

        Returns:
            Значение.

        Например::

            val = Settings.value(Settings.LastOpenPath)
        """
        return ShadowSettings.value(key)

    @classmethod
    def setValue(cls, key, value):
        """Устанавливает значение параметра.

        Args:
            key: Параметр.
            value: Значение.

        Raises:
            TypeError: Если значение неправильного типа.

        Например::

            Settings.setValue(Settings.LastOpenPath, 'C:/mydir')
        """
        passedType = type(value)
        expectedType = type(cls.defaultValue(key))
        if passedType != expectedType:
            raise TypeError(
                f'Key type mismatch; expected {expectedType} but got {passedType} instead.')
        return ShadowSettings.set_value(key, value)

    @classmethod
    def defaultValue(cls, key) -> Any:
        return ShadowSettings.default_value(key)

    @classmethod
    def reset(cls, key):
        """Устанавливает значение по умолчанию.

        Args:
            key: Параметр.
        """
        ShadowSettings.set_value(key, ShadowSettings.default_value(key))
