import gc
import runpy
from pathlib import Path
from typing import Optional, Union

import axipy.gui.gui_class as gui_class
from axipy.app import mainwindow
from axipy.axioma_plugin import tr
from axipy.settings import CurrentSettings

from PySide2.QtWidgets import QFileDialog

__all__ = [
    "open_file_dialog",
    "execfile",
    "get_dependencies_folder",
]


def open_file_dialog(filter_arg: str = None) -> Optional[Path]:
    """
    Открывает диалог выбора файла. Возвращает путь к выбранному файлу.
    Если нет главного окна Аксиомы, спрашивает путь к файлу в консоли.

    Args:
        filter_arg: Типы файлов. Например: ``'MapInfo Tab (*.tab);;Таблицы Excel (*.xls *.xlsx)'``.

    """
    if mainwindow.is_valid:
        params = {
            "parent": mainwindow.qt_object(),
            "dir": str(CurrentSettings.LastOpenPath)
        }
        if filter_arg is not None:
            params["filter"] = filter_arg
        if not CurrentSettings.UseNativeFileDialog:
            params["options"] = QFileDialog.DontUseNativeDialog
        result = QFileDialog.getOpenFileName(**params)[0]
    else:
        result = input(tr("Введите путь к файлу: "))

    if result:
        result_str = result
        result = Path(result)
        if not (result.is_file() or result.exists()):
            raise FileNotFoundError(f"File not found: \n{result_str}")
        else:
            return result
    else:
        return None


def execfile(path: Union[str, Path]):
    """
    Выполняет скрипт на языке python из файла.

    Args:
        path: Путь к исполняемому файлу.

    """

    def run_path(path_arg: str):
        try:
            runpy.run_path(path_arg)
        except Exception as e:
            raise e
        finally:
            gc.collect()

    if isinstance(path, str):
        run_path(path)
    elif isinstance(path, Path):
        run_path(str(path))
    elif isinstance(path, list) and len(path) > 0:
        run_path(path[0])
    else:
        raise TypeError('Parameter is not supported.')


def get_dependencies_folder() -> Path:
    """
    Возвращает папку, для установки зависимых пакетов.
    """
    return Path(
        gui_class.gui_instance._shadow.installedPluginsPath()
    ) / "installed_modules" / "dependencies" / "site-packages"
