from typing import Union, List, Iterator, Optional

from PySide2.QtCore import Signal
from axipy.cpp_gui import ShadowSelectionManager

from axipy._internal._decorator import InitOnce
from axipy.da import Table, Feature
from .gui_class import gui_instance

__all__ = ["SelectionManager", "selection_manager"]


class SelectionManager:
    """
    Класс доступа к выделенным объектам.

    Note:
        Получить экземпляр сервиса можно в атрибуте :attr:`axipy.selection_manager`.
    """

    @property
    def table(self) -> Optional[Table]:
        """Таблица, являющаяся источником текущего выделения. Если ничего не выделено, то None"""
        return Table._wrap(self._service().table())

    @property
    def ids(self) -> List[int]:
        """Список идентификаторов выделенных записей."""
        return self._service().ids()

    @property
    def count(self) -> int:
        """Размер выделения, то есть количество выделенных записей
        (количество элементов в списке идентификаторов)."""
        return self._service().count()

    def __len__(self):
        return self.count

    def clear(self):
        """Очищает выборку."""
        self._service().clear()

    def set(self, table: Table, ids: Union[List[int], int]):
        """Создает выборку из записей таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.
        """
        if isinstance(ids, int):
            ids = [ids]
        self._service().set(table._shadow, ids)

    def add(self, table: Table, ids: Union[List[int], int]):
        """Добавляет выборку записи таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.
        """
        if self.count == 0:
            return self.set(table, ids)
        if isinstance(ids, int):
            ids = [ids]
        self._service().add(table._shadow, ids)

    def remove(self, table: Table, ids: Union[List[int], int]):
        """Удаляет из выборки записи таблицы по их идентификаторам.

        Args:
            table: Таблица.
            ids: Идентификаторы записей.
        """
        if isinstance(ids, int):
            ids = [ids]
        self._service().remove(table._shadow, ids)

    def get_as_cursor(self) -> Iterator[Feature]:
        """Возвращает выборку в виде итератора по записям.

        Пример::

            for f in selection_manager.get_as_cursor():
                print('Feature id={}. Страна={}'.format(f.id, f['Страна']))

        Warning:

            .. deprecated:: 3.5
                Используйте :attr:`axipy.DataManager.selection`.
        """
        if self.table is None:
            return iter(())
        return self.table.itemsByIds(self.ids)

    def get_as_table(self) -> Optional[Table]:
        """Возвращает выборку в виде таблицы.

        Returns:
            Таблица или :data:`None`, если выборки нет.
        
        Содержимое таблицы основывается на текущей выборке на момент вызова данного метода.
        При последующем изменении или сбросе выборки контент данной таблицы не меняется.
        Результирующей таблице присваивается наименование в формате `data*`, которое в последствии можно изменить.
        При закрытии базовой таблицы данная таблицы так-же закрывается.

        Пример::

            # Получаем таблицу из выборки.
            tbl = selection_manager.get_as_table()
            # Задаем желаемое имя таблицы (необязательно)
            tbl.name = 'my_table'
            # Регистрация в каталоге (необязательно)
            app.mainwindow.catalog().add(tbl)
            for f in tbl.items():
                print('Feature id={}. Страна={}'.format(f.id, f['Страна']))

        Warning:

            .. deprecated:: 3.5
                Используйте :attr:`axipy.DataManager.selection`.
        """
        return Table._wrap(self._service().as_table())

    @InitOnce
    def _service(self):
        return ShadowSelectionManager(gui_instance._shadow)

    @property
    def _shadow(self):
        return self._service()

    @property
    def changed(self) -> Signal:
        """
        Выделение было изменено.

        :rtype: Signal[]
        """
        return self._service().selectionChanged


selection_manager = SelectionManager()
