from .schema import Attribute, Schema


__all__ = [
    "AttributeFactory",
    "attr"
]


class AttributeFactory:
    """
    Фабричный класс создания атрибутов и схем таблиц.

    Attention:
        Устарел в версии 3.0.0. Используйте :class:`axipy.Schema` и
        статические методы класса :class:`axipy.Attribute`.

    Attributes:
        DEFAULT_STRING_LENGTH (int): Длина символьных атрибутов по умолчанию.
        DEFAULT_DECIMAL_LENGTH (int): Длина десятичных атрибутов по умолчанию.
        DEFAULT_DECIMAL_PRECISION (int): Точность десятичных атрибутов по умолчанию.

    Пример создания атрибута::

        string_attr = attr.string('attr_name', 80)

    Пример создания схемы::

        schema = attr.schema(
            attr.string('column_1'),
            attr.integer('column_2'),
            coordsystem='prj:Earth Projection 12, 62, "m", 0'
        )

    Note:
        Используйте более короткий псевдоним этого класса :attr:`axipy.attr`.
    """

    DEFAULT_STRING_LENGTH = 80
    DEFAULT_DECIMAL_LENGTH = 10
    DEFAULT_DECIMAL_PRECISION = 5

    def __init__(self):
        raise NotImplementedError

    @staticmethod
    def string(name: str, length: int = DEFAULT_STRING_LENGTH) -> 'Attribute':
        """Создает атрибут строкового типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута.
        """
        return Attribute(name, f'string:{length}')

    @staticmethod
    def decimal(name: str, length: int = DEFAULT_DECIMAL_LENGTH,
                precision: int = DEFAULT_DECIMAL_PRECISION) -> 'Attribute':
        """Создает атрибут десятичного типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута. Количество символов, включая запятую.
            precision: Число знаков после запятой.
        """
        return Attribute(name, f'decimal:{length}.{precision}')

    @staticmethod
    def integer(name: str) -> 'Attribute':
        """Создает атрибут целого типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'int')

    @staticmethod
    def float(name: str) -> 'Attribute':
        """Создает атрибут вещественного типа.

        То же, что и :meth:`double`

        Args:
            name: Имя атрибута.
        """
        return Attribute.double(name)

    @staticmethod
    def double(name: str) -> 'Attribute':
        """Создает атрибут вещественного типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'double')

    @staticmethod
    def bool(name: str) -> 'Attribute':
        """Создает атрибут логического типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'bool')

    @staticmethod
    def date(name: str) -> 'Attribute':
        """Создает атрибут типа дата.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'date')

    @staticmethod
    def time(name: str) -> 'Attribute':
        """Создает атрибут типа время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'time')

    @staticmethod
    def datetime(name: str) -> 'Attribute':
        """Создает атрибут типа дата и время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'datetime')

    @staticmethod
    def schema(*attributes: Attribute, coordsystem: str = None) -> 'Schema':
        """Создает схему из списка атрибутов.

        Args:
            *attributes: Список атрибутов.
            coordsystem: Система координат в виде строки.

        Returns:
            Новая схема.

        See also:
            :meth:`axipy.cs.CoordSystem.from_string`.
        """
        return Schema(*attributes, coordsystem=coordsystem)


attr = AttributeFactory
