import re
import xml.etree.ElementTree as ElemTree
from PySide2.QtCore import Qt, QDir, QFileInfo, QSettings
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import (QDialog, QDialogButtonBox, QLineEdit, QPushButton,
                               QListWidgetItem, QFileDialog, QInputDialog, QMessageBox)
from axipy import Plugin, DefaultSettings, CurrentSettings, Notifications, gui_instance
from pathlib import Path
from typing import Any, List
from .ui.style_catalog import Ui_StyleCatalog

RE_EMPTY = re.compile(r"^\s*$")


class StyleCatalogsDialog(QDialog):
    NAME = "name"
    PEN = "pen"
    BRUSH = "brush"
    SYMBOL = "symbol"

    KEYS_ORDER_LIST = "keys_order_list"
    ITEMS_DICT = "items_dict"

    # backwards compatibility
    XML_FILE_NAME = "StyleCatalog.xml"

    def __init__(self, plugin: Plugin, parent=None) -> None:
        super().__init__(parent)

        self._plugin: Plugin = plugin
        self._settings: QSettings = plugin.settings

        self._title: str = self._plugin.tr("Каталоги со стилями")

        self._ui = Ui_StyleCatalog()
        self._ui.setupUi(self)

        self._ui_ok: QPushButton = self._ui.buttonBox.button(QDialogButtonBox.Ok)
        self._ui_cancel: QPushButton = self._ui.buttonBox.button(QDialogButtonBox.Cancel)

        self._init_ui()

        default_item_text = self._plugin.tr("По умолчанию")
        data = {
            self.PEN: str(DefaultSettings.PenCatalog),
            self.BRUSH: str(DefaultSettings.BrushCatalog),
            self.SYMBOL: str(DefaultSettings.SymbolCatalog),
        }
        self.add_item(default_item_text, data)

        self.read_settings()

        self._ui.list_widget.setCurrentRow(0)

        # backwards compatibility
        resource_dir = QDir(gui_instance._shadow.system_resource_catalog())
        self._system_catalog_list: List[QFileInfo] = self._find_system_catalogs(resource_dir)
        # logging.debug(f"{self._system_catalog_list=}")
        app_local_data_location = QDir(gui_instance._shadow.installedPluginsPath())
        self._xml_file_info = QFileInfo(app_local_data_location.absoluteFilePath(self.XML_FILE_NAME))
        # logging.debug(f"{self._xml_file_info=}")

        self.init_sets()

    def _init_ui(self) -> None:
        self._ui.tb_pen.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_pen.clicked.connect(lambda: self.catalog_common_clicked(self.PEN))

        self._ui.tb_brush.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_brush.clicked.connect(lambda: self.catalog_common_clicked(self.BRUSH))

        self._ui.tb_symbol.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_symbol.clicked.connect(lambda: self.catalog_common_clicked(self.SYMBOL))

        self._ui.list_widget.currentItemChanged.connect(self.on_current_item_changed)

        self._ui.tb_up.setIcon(QIcon.fromTheme("arrow_up"))
        self._ui.tb_bottom.setIcon(QIcon.fromTheme("arrow_down"))
        self._ui.tb_add.setIcon(QIcon.fromTheme("add"))
        self._ui.tb_remove.setIcon(QIcon.fromTheme("delete"))
        self._ui.tb_up.clicked.connect(self.on_up)
        self._ui.tb_bottom.clicked.connect(self.on_bottom)
        self._ui.tb_add.clicked.connect(self.on_add)
        self._ui.tb_remove.clicked.connect(self.on_remove)

        self._ui.pb_save_to_xml.clicked.connect(self.save_to_xml)
        self._ui.pb_open_from_xml.clicked.connect(self.open_from_xml)

        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)

        self._ui_ok.setText(self._plugin.tr("Применить"))

    def read_settings(self) -> None:
        # logging.debug("read_settings()")
        if self._settings.contains(self.KEYS_ORDER_LIST) and self._settings.contains(self.ITEMS_DICT):
            keys_order_list = self._settings.value(self.KEYS_ORDER_LIST)
            items_dict = self._settings.value(self.ITEMS_DICT)
            # logging.debug(f"{items_dict=}")

            for key in keys_order_list:
                self.add_item(key, items_dict[key])

    def save_settings(self) -> None:
        # logging.debug("save_settings()")
        keys_order_list = []
        items_dict = {}

        list_widget = self._ui.list_widget
        count = list_widget.count()
        for row in range(1, count):
            item = list_widget.item(row)
            item_text = item.text()
            item_data = item.data(Qt.UserRole)
            keys_order_list.append(item_text)
            items_dict[item_text] = item_data

        self._settings.setValue(self.KEYS_ORDER_LIST, keys_order_list)
        self._settings.setValue(self.ITEMS_DICT, items_dict)
        # logging.debug(f"{items_dict=}")

    def add_item(self, text: str, data: Any) -> None:
        item = QListWidgetItem(text)
        list_widget = self._ui.list_widget
        count = list_widget.count()
        for row in range(count):
            existing_item = list_widget.item(row)
            existing_item_text = existing_item.text()
            if text == existing_item_text:
                item = existing_item
        item.setData(Qt.UserRole, data)
        self._ui.list_widget.addItem(item)

    def accept(self) -> None:
        CurrentSettings.PenCatalog = Path(self.le_pen_text)
        CurrentSettings.BrushCatalog = Path(self.le_brush_text)
        CurrentSettings.SymbolCatalog = Path(self.le_symbol_text)
        self.save_settings()
        super().accept()

    def reject(self) -> None:
        self.save_settings()
        super().reject()

    # xml

    def save_to_xml(self) -> None:

        file_name, _selected_filter = QFileDialog.getSaveFileName(
            parent=self,
            caption="Выбор имени файла xml",
            dir=str(CurrentSettings.LastSavePath),
            filter="Файлы xml (*.xml)",
        )
        if not file_name:
            return None

        CurrentSettings.LastSavePath = Path(file_name).parent

        root_elem = ElemTree.Element("catalogs")

        list_widget = self._ui.list_widget
        count = list_widget.count()
        for row in range(1, count):
            item = list_widget.item(row)
            item_text = item.text()
            item_data = item.data(Qt.UserRole)
            attrs = {
                self.NAME: item_text,
                self.PEN: item_data[self.PEN],
                self.BRUSH: item_data[self.BRUSH],
                self.SYMBOL: item_data[self.SYMBOL],
            }
            ElemTree.SubElement(root_elem, "catalog", **attrs)

        element_tree = ElemTree.ElementTree(root_elem)
        element_tree.write(file_name, encoding="UTF-8")

        Notifications.push(self._title, f"Файл с наборами '{file_name}' сохранен.", Notifications.Success)

    def open_from_xml(self) -> None:

        file_name, _selected_filter = QFileDialog.getOpenFileName(
            parent=self,
            caption="Открытие файла xml",
            dir=str(CurrentSettings.LastOpenPath),
            filter="Файлы xml (*.xml)",
        )
        if not file_name:
            return None

        CurrentSettings.LastOpenPath = Path(file_name).parent

        success = self.load_xml_file(Path(file_name))
        if success:
            Notifications.push(self._title, f"Файл с наборами '{file_name}' загружен.", Notifications.Success)

    def load_xml_file(self, file_name: Path) -> bool:
        try:
            if not file_name.exists():
                return False

            with open(str(file_name), mode='r', encoding="UTF-8") as stream:
                element_tree = ElemTree.parse(stream)
                root_element = element_tree.getroot()

                for element in root_element:
                    data = {
                        self.PEN: element.attrib[self.PEN],
                        self.BRUSH: element.attrib[self.BRUSH],
                        self.SYMBOL: element.attrib[self.SYMBOL],
                    }
                    self.add_item(element.attrib[self.NAME], data)
        except (Exception,):
            return False
        else:
            return True

    # ~xml

    def catalog_common_clicked(self, key: str) -> None:
        if key == self.PEN:
            caption = self._plugin.tr("Выбор каталога со стилями линий")
        elif key == self.BRUSH:
            caption = self._plugin.tr("Выбор каталога со стилями заливки")
        elif key == self.SYMBOL:
            caption = self._plugin.tr("Выбор каталога с растровыми символами")
        else:
            raise RuntimeError()

        dir_name = QFileDialog.getExistingDirectory(
            self,  # parent
            caption,  # caption
            str(CurrentSettings.LastOpenPath),  # dir
            QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks,  # options
        )
        if not dir_name:
            return None

        CurrentSettings.LastOpenPath = Path(dir_name)

        if key == self.PEN:
            self.le_pen_text = dir_name
        elif key == self.BRUSH:
            self.le_brush_text = dir_name
        elif key == self.SYMBOL:
            self.le_symbol_text = dir_name

        data = {
            self.PEN: self.le_pen_text,
            self.BRUSH: self.le_brush_text,
            self.SYMBOL: self.le_symbol_text,
        }
        self._ui.list_widget.currentItem().setData(Qt.UserRole, data)

    def toggle_controls_on_default_current(self, state: bool) -> None:
        controls = [
            # self._ui.tb_add,  always allowed
            self._ui.tb_remove,
            self._ui.tb_up,
            self._ui.tb_bottom,

            self._ui.tb_pen,
            self._ui.tb_brush,
            self._ui.tb_symbol,
        ]
        if state:
            count = self._ui.list_widget.count()
            # logging.debug(f"{count=}")
            last_row = count - 1
            current_row = self._ui.list_widget.currentRow()
            # logging.debug(f"{current_row=}, {last_row=}")
            if current_row == 1:
                controls.remove(self._ui.tb_up)
                self._ui.tb_up.setEnabled(False)

            if current_row == last_row:
                controls.remove(self._ui.tb_bottom)
                self._ui.tb_bottom.setEnabled(False)

        for control in controls:
            control.setEnabled(state)

    def on_current_item_changed(self, current: QListWidgetItem, _previous: QListWidgetItem):
        # logging.debug("on_current_item_changed()")
        if self._ui.list_widget.row(current) == 0:
            self.toggle_controls_on_default_current(False)
        else:
            self.toggle_controls_on_default_current(True)

        item_data = current.data(Qt.UserRole)
        self.le_pen_text = item_data[self.PEN]
        self.le_brush_text = item_data[self.BRUSH]
        self.le_symbol_text = item_data[self.SYMBOL]

    # line edits

    @property
    def le_pen_text(self) -> str:
        return self._ui.le_pen.text()

    @le_pen_text.setter
    def le_pen_text(self, value: str) -> None:
        self._ui.le_pen.setText(str(Path(value)))

    @property
    def le_brush_text(self) -> str:
        return self._ui.le_brush.text()

    @le_brush_text.setter
    def le_brush_text(self, value: str) -> None:
        self._ui.le_brush.setText(str(Path(value)))

    @property
    def le_symbol_text(self) -> str:
        return self._ui.le_symbol.text()

    @le_symbol_text.setter
    def le_symbol_text(self, value: str) -> None:
        self._ui.le_symbol.setText(str(Path(value)))

    def on_up(self) -> None:
        current_item = self._ui.list_widget.currentItem()
        if not current_item:
            return None

        current_row = self._ui.list_widget.currentRow()

        if current_row > 1:
            taken_item = self._ui.list_widget.takeItem(current_row)
            self._ui.list_widget.insertItem(current_row - 1, taken_item)
            self._ui.list_widget.setCurrentItem(taken_item)

    def on_bottom(self) -> None:
        current_item = self._ui.list_widget.currentItem()
        if not current_item:
            return None

        current_row = self._ui.list_widget.currentRow()
        last_row = self._ui.list_widget.count() - 1
        if current_row != last_row:
            taken_item = self._ui.list_widget.takeItem(current_row)
            self._ui.list_widget.insertItem(current_row + 1, taken_item)
            self._ui.list_widget.setCurrentItem(taken_item)

    def on_add(self) -> None:
        try:
            text, success = QInputDialog.getText(
                self,  # parent
                self._plugin.tr("Название набора"),  # title
                self._plugin.tr("Набор стилей"),  # label
                QLineEdit.Normal,  # echo
                '',  # text
                self.windowFlags() & ~Qt.WindowContextHelpButtonHint  # flags
            )
            if not success:
                return None

            error_title = self._plugin.tr("Ошибка ввода названия набора")

            if RE_EMPTY.match(text):
                QMessageBox.critical(
                    self,
                    error_title,
                    self._plugin.tr(f"Пустое название набора.")
                )
                return None

            list_widget = self._ui.list_widget
            count = list_widget.count()
            for row in range(count):
                item = list_widget.item(row)
                item_text = item.text()
                if text == item_text:
                    QMessageBox.critical(
                        self,
                        error_title,
                        self._plugin.tr(f"Название '{text}' уже существует.")
                    )
                    return None

            data = {
                self.PEN: str(DefaultSettings.PenCatalog),
                self.BRUSH: str(DefaultSettings.BrushCatalog),
                self.SYMBOL: str(DefaultSettings.SymbolCatalog),
            }
            self.add_item(text, data)
            self._ui.list_widget.setCurrentRow(self._ui.list_widget.count() - 1)

        except Exception as ex:
            print(ex)

    def on_remove(self) -> None:
        current_row = self._ui.list_widget.currentRow()
        if current_row < 0:
            return None
        # change to first after delete, because count doesn't update after take
        self._ui.list_widget.setCurrentRow(0)
        self._ui.list_widget.takeItem(current_row)

    # backwards compatibility

    def _find_system_catalogs(self, resource_dir: QDir) -> List[QFileInfo]:
        """
        Ищем каталоги в стандартных местах
        \\resource\\ExtendedStyles
        """
        # logging.debug(f"_find_system_catalogs({resource_dir=})")
        result = []
        if not resource_dir or not isinstance(resource_dir, QDir):
            return result

        extended_styles_dir = QDir(resource_dir.filePath("ExtendedStyles"))
        # logging.debug(f"{extended_styles_dir=}")
        if extended_styles_dir.exists():
            catalog_list = extended_styles_dir.entryInfoList(QDir.AllDirs | QDir.NoDotAndDotDot)
            # logging.debug(f"{catalog_list=}")
            for catalog in catalog_list:
                # logging.debug(f"{catalog=}")
                xml_file = QFileInfo(QDir(catalog.absoluteFilePath()), self.XML_FILE_NAME)
                # logging.debug(f"{xml_file=}")
                if xml_file.exists():
                    result.append(xml_file)
        return result

    def init_sets(self) -> None:
        for system_catalog in self._system_catalog_list:
            self.load_xml_file(Path(system_catalog.filePath()))

        if self._xml_file_info.exists():
            self.load_xml_file(Path(self._xml_file_info.filePath()))
