from typing import List, Tuple, Optional, Iterator, Any

from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QAction
from axipy._internal._metaclass import _MappingMetaExtended, _MappingMetaDocumentation

from .gui_class import gui_instance

__all__ = ["ActionManager"]


class _CustomMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator[str]:
        keys = map(lambda action: action.objectName(), gui_instance._shadow.all_actions())
        return iter(keys)

    def __len__(cls) -> int:
        return len(gui_instance._shadow.all_actions())

    def __getitem__(cls, key: str) -> QAction:
        result = gui_instance._shadow.find_action_by_id(key)
        if result is None:
            raise KeyError(f"{key}")
        else:
            return result


class ActionManager(_MappingMetaDocumentation, metaclass=_CustomMappingMetaExtended):
    """
    Менеджер системных действий и инструментов.
    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    class _Desc:
        def __get__(self, obj, objtype=None) -> List[str]:
            return gui_instance._shadow.all_icon_names()

    all_icon_names: List[str] = _Desc()
    """
    Перечень доступных идентификаторов иконок, присутствующих в ресурсах Аксиомы.

    На основании данного идентификатора можно создать иконку для действия :class:`PySide2.QtWidgets.QAction`.
    Использовать выбранный идентификатор предлагается с помощью функции :meth:`icon_by_name`.

    Для создания иконки для :class:`PySide2.QtWidgets.QAction` также доступен стандартный путь указания имени файла.
    """

    @classmethod
    def items(cls) -> List[Tuple[str, QAction]]:
        """
        Возвращает список кортежей ключ-значение, где ключи это идентификаторы действий, a значения это объекты класса
        :class:`PySide2.QtWidgets.QAction`.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это идентификаторы действий."""
        return super().keys()

    @classmethod
    def values(cls) -> List[QAction]:
        """
        Возвращает список значений, где значения это это объекты класса
        :class:`PySide2.QtWidgets.QAction`.
        """
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Any = None) -> Optional[QAction]:
        """Возвращает значение по ключу."""
        return super().get(key, default_value)

    @staticmethod
    def activate(name: str):
        """
        Делает активным инструмент по его идентификатору если это возможно.
        Если действие не найдено, генерируется исключение.
        Если действие недоступно в настоящий момент (неактивно), установка игнорируется.
        
        Args:
            name: Идентификатор действия

        Активация инструмента 'Сдвиг'::

            ActionManager.activate('Pan')

        Вызов диалога 'Стиль символа'::
        
            ActionManager.activate('SymbolStyle')
        """
        act = ActionManager.get(name)
        if act:
            if act.isEnabled():
                act.toggle()
                act.trigger()
        else:
            raise ValueError(f'Действие {name} не найдено.')

    @staticmethod
    def icon_by_name(name: str) -> QIcon:
        """
        Создание иконки по ее внутреннему идентификатору. Подробнее см. :attr:`all_icon_names`
        
        Args:
            name: Идентификатор действия
        """
        return QIcon.fromTheme(name)
