from typing import Optional

from PySide2.QtCore import Qt
from PySide2.QtGui import QKeyEvent, QMouseEvent, QPaintEvent, QPainter

from axipy import MapTool, view_manager, ActiveToolPanel, selection_manager
from axipy.concurrent import ProgressSpecification, ProgressGuiFlags
from axipy.gui.map_tool import SelectToolBehavior
from . import helper
from .helper import print_


class MapToolExt(MapTool):
    first_load = True
    first_unload = True

    global_connections = []
    active_view_connections = []

    def __init__(self) -> None:
        super().__init__()

        self.connections = []

    def load(self) -> None:
        print_(f"load {self}\n")

        if MapToolExt.first_load:
            self.load_once()
            MapToolExt.first_load = False

    def load_once(self) -> None:
        MapToolExt.first_unload = True

    def unload(self) -> None:

        print_(f"unload {self}")
        n = 0
        for connection in self.connections:
            connection.disconnect()
            n += 1
        print_(f"cleared {n}")

        if MapToolExt.first_unload:
            self.unload_once()
            MapToolExt.first_unload = False

        if not MapToolExt.first_load:
            MapToolExt.first_load = True

        self.redraw()

    def unload_once(self) -> None:
        g, a = 0, 0
        for connection in self.global_connections:
            connection.disconnect()
            g += 1
        self.global_connections.clear()

        for connection in self.active_view_connections:
            connection.disconnect()
            a += 1
        self.active_view_connections.clear()

        print_(f"cleared g={g}, a={a}")

        self.redraw()

    def keyPressEvent(self, event: QKeyEvent) -> bool:
        if event.key() == Qt.Key_Escape:
            self.reset()
            return self.BlockEvent
        else:
            return self.PassEvent

    def block(self) -> bool:
        return bool(self.view.widget != view_manager.active.widget)

    def spec(self, desc: str) -> ProgressSpecification:
        return ProgressSpecification(window_title=self.title, description=desc, flags=ProgressGuiFlags.CANCELABLE)

    def __str__(self) -> str:
        return self.view.title


class PanelMapTool(MapToolExt):

    def __init__(self, custom=False) -> None:
        super().__init__()
        PanelMapTool.custom = custom

    def load_once(self) -> None:
        super().load_once()

        if not hasattr(self, "widget"):
            raise AttributeError("no widget attr for panel")

        if self.custom:
            panel = ActiveToolPanel().make_custom(
                self.title, self.observer_id, self.widget)
        else:
            panel = ActiveToolPanel().make_acceptable(
                self.title, self.observer_id, self.widget)
            if hasattr(self, "tool_panel_accepted"):
                panel.accepted.connect(self.tool_panel_accepted)
            else:
                raise AttributeError("PanelMapTool has no tool_panel_accepted method")
            panel.try_enable()

        PanelMapTool.panel = panel

        panel.panel_was_closed.connect(MapTool.reset)

        panel.activate()

    def unload_once(self) -> None:
        if hasattr(PanelMapTool, "panel") and PanelMapTool.panel is not None:
            PanelMapTool.panel.deactivate()
            w = PanelMapTool.panel.widget
            if hasattr(w, "unload_once"):
                w.unload_once()
            PanelMapTool.panel = None

        super().unload_once()


class SelectorMapTool(PanelMapTool):

    def __init__(self) -> None:
        super().__init__()

        self.selector = SelectToolBehavior(self)

    def load_once(self) -> None:
        super().load_once()

        if hasattr(self.panel.widget, "selection_changed"):
            self.global_connections.append(helper.Connection(
                selection_manager.changed, self.panel.widget.selection_changed))
            self.panel.widget.selection_changed()

        if hasattr(self.panel.widget, "active_changed"):
            self.global_connections.append(helper.Connection(
                view_manager.active_changed, self.panel.widget.active_changed))
            self.panel.widget.active_changed()

    def unload(self) -> None:
        # Важно явно удалить selector при unload
        self.selector = None
        super().unload()

    # Input events

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        if self.selector.keyPressEvent(event):
            return self.BlockEvent
        super().keyPressEvent(event)

    def mouseMoveEvent(self, event: QMouseEvent):
        self.selector.mouseMoveEvent(event)
        return self.PassEvent

    def mousePressEvent(self, event: QMouseEvent) -> Optional[bool]:
        return self.selector.mousePressEvent(event)

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        return self.selector.mouseReleaseEvent(event)

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        self.selector.paintEvent(event, painter)
        return self.PassEvent
