from dataclasses import dataclass
from enum import Enum, auto
from functools import wraps
from pathlib import Path
from typing import Optional, Callable, Any, Iterator, Tuple, overload, Type, TYPE_CHECKING

from PySide2.QtCore import Qt, QFileInfo, Slot, Signal, QLocale
from PySide2.QtGui import QIcon, QKeyEvent
from PySide2.QtWidgets import (QFileDialog, QDialog, QDialogButtonBox, QMessageBox, QWidget, QProgressBar)
from axipy import (tr, CurrentSettings, task_manager, provider_manager,
                   ChooseCoordSystemDialog, CoordSystem, Notifications, AxipyAnyCallableTask, DefaultSettings,
                   data_manager, Table, AxipyProgressHandler, Schema, Source, Destination, Feature, Rect,
                   FloatCoord, Raster, DataObject)

from .bounding_rect_dialog_native import BoundingRectDialogCustom
from .ui.export_to_file_dialog import Ui_Dialog

if TYPE_CHECKING:
    from .__init__ import ExportToFile


def try_except(notify_msg: str) -> Callable:
    def decorate(func: Callable) -> Callable:

        @wraps(func)
        def wrapper(*args, **kwargs) -> Any:
            try:
                return func(*args, **kwargs)
            except Exception as e:
                print(e)
                if notify_msg:
                    self = args[0]
                    title = self._title
                    Notifications.push(title, self._plugin.tr(notify_msg) + f" ({e})", Notifications.Critical)

        return wrapper

    return decorate


def copy_rect(rect: Rect) -> Rect:
    return Rect(rect.xmin, rect.ymin, rect.xmax, rect.ymax)


class Exporter:
    @dataclass
    class SkippedNotification:
        first_file_name: Path
        n: int = 1

    class ExportMode(Enum):
        DEFAULT = auto()
        """Обычный экспорт, со сменой проекции и пересчетом координат."""
        PRESERVE_GEOMETRY = auto()
        """Не пересчитывать координаты при смене проекции."""
        OVERRIDE_INP_CS = auto()
        """Переопределить входную проекцию."""

    def __init__(
            self,
            paths: Tuple[Path],
            *,
            out_path: Path,
            preserve_geometry: bool,
            override_inp_cs: bool,
            widget_dialog: 'ExportToFileDialog',
    ) -> None:
        self._paths: Tuple[Path] = paths
        self._out_folder: Path = out_path
        self._preserve_geometry: bool = preserve_geometry
        self._override_inp_cs: bool = override_inp_cs
        self._widget_dialog: 'ExportToFileDialog' = widget_dialog

        self._overwrite: bool = False
        self._skipped_notification: Optional[Exporter.SkippedNotification] = None

    def export_process(
            self,
            ph: AxipyProgressHandler,
    ) -> 'ExportToFileDialog.TaskResult':
        paths_length = len(self._paths)
        successfully_exported_files = 0
        for i, inp_path in enumerate(self._paths):
            if i == 1:
                self._widget_dialog.signal_pbar_tables_set_max.emit(paths_length)
                self._widget_dialog.signal_pbar_tables_add_value.emit()

            out_path = self._out_folder / inp_path.name

            skip_existing = not self._overwrite and out_path.exists()

            if skip_existing:
                if self._skipped_notification is None:
                    self._skipped_notification = self.SkippedNotification(out_path)
                else:
                    self._skipped_notification.n += 1
            else:
                try:
                    self.export_single(
                        ph,
                        inp_path,
                        out_path,
                    )
                except (Exception,) as e:
                    self._widget_dialog.signal_notify.emit(
                        f"Ошибка в процессе конвертации: {e}",
                        int(Notifications.Critical)
                    )
                    raise e
                else:
                    successfully_exported_files += 1

            self._widget_dialog.signal_pbar_tables_add_value.emit()
            if ph.is_canceled():
                self._widget_dialog.signal_notify.emit("Конвертация отменена.", int(Notifications.Warning))
                break

        return ExportToFileDialog.TaskResult(paths_length, successfully_exported_files, self._skipped_notification)

    @staticmethod
    @overload
    def ensure_opened_data_object(file_path: Path, data_object_type: Type[Table]) -> Optional[Table]:
        ...

    @staticmethod
    @overload
    def ensure_opened_data_object(file_path: Path, data_object_type: Type[Raster]) -> Optional[Raster]:
        ...

    @staticmethod
    @overload
    def ensure_opened_data_object(file_path: Path, data_object_type: Type[DataObject]) -> Optional[DataObject]:
        ...

    @staticmethod
    def ensure_opened_data_object(file_path, data_object_type=DataObject):
        """Поиск уже открытой таблицы"""
        all_data_objects: Iterator[data_object_type] = filter(
            lambda data_object_arg: isinstance(data_object_arg, data_object_type), data_manager.all_objects
        )
        for data_object in all_data_objects:
            prop = data_object.properties

            file_name = prop.get("fileName", '')
            if file_name and Path(file_name) == file_path:
                return data_object

    @staticmethod
    def copy_coord_system(cs: CoordSystem, *, rect: Rect = None) -> CoordSystem:
        cs = CoordSystem.from_string(cs.to_string())
        if rect is not None:
            cs.rect = rect
        return cs

    @staticmethod
    def process_table_change_cs(table: Table, inp_cs: CoordSystem) -> Iterator[Feature]:

        def change_cs(f: Feature) -> Feature:
            g = f.geometry
            if g is not None:
                g.coordsystem = inp_cs
                f.geometry = g
            return f

        return map(change_cs, table.items())

    def export_single(
            self,
            ph: AxipyProgressHandler,
            inp_path: Path,
            out_path: Path,
    ) -> None:
        table = None
        is_hidden = False
        try:
            # Поиск уже открытой таблицы
            table = self.ensure_opened_data_object(inp_path, Table)

            # Открытие временной таблицы
            if table is None:
                table = self.get_source_tab_from_path(inp_path, hidden=True).open()
                is_hidden = True

            if table is None:
                raise Exception(f"Can't open table {inp_path}")

            if self._preserve_geometry:
                export_mode = self.ExportMode.PRESERVE_GEOMETRY
            elif self._override_inp_cs:
                export_mode = self.ExportMode.OVERRIDE_INP_CS
            else:
                export_mode = self.ExportMode.DEFAULT

            if export_mode == self.ExportMode.PRESERVE_GEOMETRY:
                # В этом случае берется охват входной таблицы, чтобы сохранить точность координат
                out_prj_bounds = copy_rect(table.coordsystem.rect)
                items = self.process_table_change_cs(table, self._widget_dialog.out_cs)
            elif export_mode == self.ExportMode.OVERRIDE_INP_CS:
                out_prj_bounds = self._widget_dialog.out_prj_bounds
                items = self.process_table_change_cs(table, self._widget_dialog.inp_cs)
            elif export_mode == self.ExportMode.DEFAULT:
                out_prj_bounds = self._widget_dialog.out_prj_bounds
                items = table.items()
            else:
                raise RuntimeError("Internal")

            self._widget_dialog.signal_pbar_features_set_max.emit(table.count())

            def func_callback(_f: Feature, _i: int):
                self._widget_dialog.signal_pbar_features_add_value.emit()
                if ph.is_canceled():
                    return False

            out_cs_copy = self.copy_coord_system(
                self._widget_dialog.out_cs,
                # В этом случае берется охват входной таблицы, чтобы сохранить точность координат
                rect=out_prj_bounds,
            )
            schema = table.schema
            schema.coordsystem = out_cs_copy
            destination = self.get_destination_tab_from_path(out_path, schema)
            destination.export(items, func_callback=func_callback)
        except Exception as e:
            raise e
        finally:
            if table is not None and is_hidden:
                task_manager.run_in_gui(table.close)

    @staticmethod
    def get_source_tab_from_path(path: Path, hidden: bool = True) -> Source:
        source = provider_manager.tab.get_source(str(path))
        source["hidden"] = hidden
        return source

    @staticmethod
    def get_destination_tab_from_path(path: Path, schema: Schema = None) -> Destination:
        if schema is None:
            schema = Schema()
        destination = provider_manager.tab.get_destination(str(path), schema)
        return destination


class ExportToFileDialog(QDialog):
    signal_notify = Signal(str, int)

    signal_pbar_tables_set_max = Signal(int)
    signal_pbar_tables_add_value = Signal()

    signal_pbar_features_set_max = Signal(int)
    signal_pbar_features_add_value = Signal()

    @dataclass
    class TaskResult:
        all_files: int
        successfully_exported_files: int
        skipped: Exporter.SkippedNotification

    def __init__(self, plugin: 'ExportToFile', parent=None) -> None:
        super().__init__(parent)

        self._plugin: 'ExportToFile' = plugin
        self._title: str = self._plugin.tr("Экспорт таблиц")

        # ui file
        self._ui = Ui_Dialog()
        self._ui.setupUi(self)

        current_cs = CoordSystem.current()
        self.inp_cs: CoordSystem = current_cs
        self.out_cs: CoordSystem = current_cs
        # Копирование Rect, чтобы он не был связан с объектом КС
        self.out_prj_bounds = current_cs.rect

        self._selected_filter: Optional[str] = None
        self._overwrite: bool = False
        self._pbar_range_init: bool = False

        self._out_folder: Optional[Path] = None

        self._task: Optional[AxipyAnyCallableTask] = None

        self.signal_notify.connect(self.slot_notify)
        self.signal_pbar_tables_set_max.connect(self.slot_pbar_tables_set_max)
        self.signal_pbar_tables_add_value.connect(self.slot_pbar_tables_add_value)
        self.signal_pbar_features_set_max.connect(self.slot_pbar_features_set_max)
        self.signal_pbar_features_add_value.connect(self.slot_pbar_features_add_value)

        self._ui_ok: Optional[QDialogButtonBox.StandardButton.Ok] = None
        self._ui_cancel: Optional[QDialogButtonBox.StandardButton.Cancel] = None

        self.init_ui()

    def init_ui(self) -> None:
        self._ui.pbar_tables.hide()
        self._ui.pbar_features.hide()

        self.setWindowTitle(self._title)
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"

        self._ui.tb_select_files.setIcon(QIcon.fromTheme("open"))

        self._ui.tb_select_files.clicked.connect(self.on_open_files_clicked)
        self._ui.tb_delete_selected.setIcon(QIcon.fromTheme("delete"))
        self._ui.tb_delete_selected.clicked.connect(self.on_delete_selected_clicked)

        def on_selection_changed() -> None:
            self._ui.tb_delete_selected.setEnabled(bool(len(self._ui.list_source_tables.selectedItems())))

        self._ui.list_source_tables.itemSelectionChanged.connect(on_selection_changed)

        last_save_path = self.get_existing_last_save_path()
        self._out_folder = last_save_path
        self._ui.le_out_folder.setText(str(last_save_path))
        self._ui.pb_choose_out_folder.clicked.connect(self.choose_out_folder)
        self._ui.pb_inp_cs.clicked.connect(lambda: self.on_projection_clicked(is_input=True))
        self._ui.pb_out_cs.clicked.connect(lambda: self.on_projection_clicked(is_input=False))
        self._ui.pb_out_bounds.clicked.connect(self.on_projection_rect_clicked_native)

        def on_cb_clicked(checked: bool) -> None:
            if checked:
                self._ui.gb_inp_cs.setChecked(False)
            self.check_pb_bounds_state()

        def on_gb_clicked(checked: bool) -> None:
            if checked:
                self._ui.cb_preserve_geomery.setChecked(False)
            self.check_pb_bounds_state()

        self._ui.gb_inp_cs.clicked.connect(on_gb_clicked)
        self._ui.cb_preserve_geomery.clicked.connect(on_cb_clicked)

        self._ui_ok = self._ui.buttonBox.button(QDialogButtonBox.Ok)
        self._ui_ok.setText(self._plugin.tr("Экспорт"))
        self._ui_ok.setEnabled(False)
        self._ui_cancel = self._ui.buttonBox.button(QDialogButtonBox.Cancel)
        self._ui_cancel.setText(self._plugin.tr("Закрыть"))

    def keyPressEvent(self, event: QKeyEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self.on_delete_selected_clicked()
        super().keyPressEvent(event)

    def accept(self) -> None:
        self.export_process_prepare()

    def reject(self) -> None:

        if (
                self._task is not None and
                self._task.progress_handler().is_running() and
                not self._task.progress_handler().is_canceled()
        ):
            self._task.progress_handler().cancel()
            return None

        if self._task is None or not self._task.progress_handler().is_running():
            super().reject()

    @property
    def inp_cs(self) -> CoordSystem:
        return self._inp_cs

    @inp_cs.setter
    def inp_cs(self, cs: CoordSystem):
        self._inp_cs = cs
        self._ui.pb_inp_cs.setText(cs.name)

    @property
    def out_cs(self) -> CoordSystem:
        return self._out_cs

    @out_cs.setter
    def out_cs(self, cs: CoordSystem):
        self._out_cs = cs
        # Происходит копия
        self.out_prj_bounds = cs.rect
        # Обновление представления
        self._ui.pb_out_cs.setText(cs.name)

    @property
    def out_prj_bounds(self) -> Rect:
        return self._out_prj_bounds

    @out_prj_bounds.setter
    def out_prj_bounds(self, rect: Rect):
        # Копирование Rect, чтобы он не был связан с объектом КС
        self._out_prj_bounds = copy_rect(rect)
        self._ui.pb_out_bounds.setText(self.format_rect_as_string(self.out_prj_bounds))

    @staticmethod
    def format_rect_as_string(rect: Rect) -> str:
        params = {"precision": 6, "locale": QLocale.English}
        xmin = FloatCoord(rect.xmin).as_string(**params)
        ymin = FloatCoord(rect.ymin).as_string(**params)
        xmax = FloatCoord(rect.xmax).as_string(**params)
        ymax = FloatCoord(rect.ymax).as_string(**params)
        return f"({xmin}, {ymin}, {xmax}, {ymax})"

    def check_pb_bounds_state(self) -> None:
        is_checked = self._ui.cb_preserve_geomery.isChecked()
        self._ui.pb_out_bounds.setEnabled(not is_checked)

    def check_enabled_export(self) -> None:
        if (
                self._ui.list_source_tables.count() > 0 and
                self._out_folder is not None
        ):
            enabled = True
        else:
            enabled = False

        self._ui_ok.setEnabled(enabled)

    def read_inp_paths(self) -> Tuple[Path]:
        list_widget = self._ui.list_source_tables
        paths = (list_widget.item(i).text() for i in range(list_widget.count()))
        return tuple(Path(elem) for elem in paths)

    @try_except("Ошибка открытия файлов.")
    def on_open_files_clicked(self) -> None:
        last_open_path = CurrentSettings.LastOpenPath
        if not Path(last_open_path).exists():
            last_open_path = DefaultSettings.LastOpenPath

        file_names, selected_filter = QFileDialog.getOpenFileNames(
            parent=self,
            caption="Выбор файлов для экспорта",
            dir=str(last_open_path),
            filter="Файлы TAB (*.tab)",
            selectedFilter=self._selected_filter,  # selectedFilter
        )
        if len(file_names) <= 0:
            return None

        self._selected_filter = selected_filter
        CurrentSettings.LastOpenPath = QFileInfo(file_names[0]).absolutePath()
        path_list = self.read_inp_paths()
        for fn in file_names:
            if fn not in path_list:
                self._ui.list_source_tables.addItem(fn)
        self.check_enabled_export()

    def on_delete_selected_clicked(self) -> None:
        selected_items = self._ui.list_source_tables.selectedItems()
        for item in selected_items:
            self._ui.list_source_tables.takeItem(self._ui.list_source_tables.row(item))
        self.check_enabled_export()

    @staticmethod
    def get_existing_last_save_path() -> Path:
        last_save_path = CurrentSettings.LastSavePath
        if not Path(last_save_path).exists():
            last_save_path = DefaultSettings.LastSavePath
        return last_save_path

    @Slot()
    def choose_out_folder(self) -> None:
        last_save_path = self.get_existing_last_save_path()
        out_folder = QFileDialog.getExistingDirectory(
            self,
            caption=tr("Выбор выходной папки"),
            dir=str(last_save_path),
            options=QFileDialog.ShowDirsOnly,
        )
        if not out_folder:
            return None
        self._ui.le_out_folder.setText(out_folder)
        self._out_folder = Path(out_folder)
        CurrentSettings.LastSavePath = out_folder

        self.check_enabled_export()

    def on_projection_clicked(self, *, is_input: bool) -> None:
        if is_input:
            cs = self.inp_cs
        else:
            cs = self.out_cs
        dialog = ChooseCoordSystemDialog(cs)

        @Slot(int)
        def finished(return_code: int):
            if return_code == QDialog.Accepted:
                result_cs = dialog.chosenCoordSystem()
                if is_input:
                    self.inp_cs = result_cs
                else:
                    self.out_cs = result_cs

                if result_cs.non_earth:
                    self.on_projection_rect_clicked_native()

        dialog.finished.connect(finished)
        dialog.open()

    @Slot()
    def on_projection_rect_clicked_native(self) -> None:
        # additional check, needed to skip choosing bounds for non earth projection
        if self._ui.cb_preserve_geomery.isChecked():
            return None

        dialog = BoundingRectDialogCustom(self.out_cs.unit, self.out_prj_bounds)

        @Slot(int)
        def finished(return_code: int):
            if return_code == QDialog.Accepted:
                self.out_prj_bounds = dialog.rect
                dialog.deleteLater()

        dialog.finished.connect(finished)
        dialog.open()

    def toggle_ui(self, state: bool) -> None:
        objects = self.children()
        widgets: Iterator[QWidget] = filter(lambda x: isinstance(x, QWidget), objects)
        for widget in widgets:
            if widget == self._ui.buttonBox:
                continue
            widget.setEnabled(state)

        self._ui_ok.setEnabled(state)

    @staticmethod
    def prepare_pbar(pbar: QProgressBar):
        """Делает прогресс бесконечным, для первого файла."""
        pbar.setValue(0)
        pbar.setMaximum(0)
        pbar.show()

    @try_except("Не удалось запустить процесс конвертации.")
    def export_process_prepare(self) -> None:
        if not self._out_folder.exists():
            mbox = QMessageBox(
                QMessageBox.Critical,  # icon
                tr("Ошибка"),  # title
                tr("Выбранной выходной папки не существует, пожалуйста, выберите другую папку."),  # text
                parent=self,
            )
            mbox.open()
            return None

        self.toggle_ui(False)
        self._ui_ok.setEnabled(False)
        self._ui_cancel.setText(tr("Отмена"))

        for pbar in (self._ui.pbar_tables, self._ui.pbar_features):
            self.prepare_pbar(pbar)

        override_inp_cs = self._ui.gb_inp_cs.isChecked()
        preserve_geometry = self._ui.cb_preserve_geomery.isChecked()

        exporter = Exporter(
            self.read_inp_paths(),
            out_path=self._out_folder,
            preserve_geometry=preserve_geometry,
            override_inp_cs=override_inp_cs,
            widget_dialog=self
        )

        if self._task is None:
            self._task = AxipyAnyCallableTask(exporter.export_process)
            self._task.progress_handler().finished.connect(self.slot_on_finished)
            task_manager.start_task(self._task)

    @Slot(int)
    def slot_pbar_tables_set_max(self, max_: int) -> None:
        self._ui.pbar_tables.setMaximum(max_)

    @Slot()
    def slot_pbar_tables_add_value(self) -> None:
        pbar = self._ui.pbar_tables
        v = pbar.value()
        pbar.setValue(v + 1)

    @Slot(int)
    def slot_pbar_features_set_max(self, max_: int) -> None:
        self._ui.pbar_features.setMaximum(max_)

    @Slot()
    def slot_pbar_features_add_value(self) -> None:
        pbar = self._ui.pbar_features
        v = pbar.value()
        pbar.setValue(v + 1)

    @Slot(str, int)
    def slot_notify(self, msg: str, msg_type: int) -> None:
        Notifications.push(self._title, msg, msg_type)

    @Slot()
    def slot_on_finished(self) -> None:
        task_result: ExportToFileDialog.TaskResult = self._task.progress_handler().result
        self._task.progress_handler().finished.disconnect(self.slot_on_finished)
        self._task = None

        self._ui.pbar_tables.hide()
        self._ui.pbar_features.hide()
        self._ui_cancel.setText(tr("Закрыть"))
        if isinstance(task_result, self.TaskResult):
            skipped_notification = task_result.skipped

            if skipped_notification is not None:
                Notifications.push(
                    self._title,
                    f"Файл '{skipped_notification.first_file_name}' уже существует в выходной папке. Файл пропущен. "
                    f"(Всего пропущено {skipped_notification.n}).",
                    Notifications.Warning,
                )
            Notifications.push(
                self._title,
                self._plugin.tr(f"Конвертация завершена. "
                                f"Сконвертировано файлов: "
                                f"{task_result.successfully_exported_files} из {task_result.all_files}."),
            )

        self.toggle_ui(True)
        self.check_pb_bounds_state()
        self.check_enabled_export()
