from typing import Optional

import axipy
from axipy import Plugin
from axipy.da import Feature, Table, GeometryType, PointCompatStyle, \
    CollectionStyle, Style
from axipy.gui import view_manager
from .strategy_base import SelectStrategy
from ..data_types import NamedSearchPatterns, SearchGroupPatterns, \
    SearchPatterns
from ..widgets.select_by_style_widget import SelectByStyleWidget
from ..widgets.style_widget_interface import SelectByStyleWidgetInterface


class MapinfoPointStrategy(SelectStrategy):
    """
    Стратегия выделения для символов со стилем MapInfo
    """

    def __init__(self, f: Feature, table: Table) -> None:
        super().__init__(f, table)
        self.__geometries = [
            GeometryType.Point,
            GeometryType.MultiPoint,
            GeometryType.GeometryCollection
        ]
        # название свойства : номер в строке стиля
        self.__symbol = axipy.tr("Символ")
        self.__color = axipy.tr("Цвет")
        self.__size = axipy.tr("Размер")

    def fill_widget(self, widget: 'SelectByStyleWidget'):
        widget.search_geometries = {gtype: True for gtype in self.__geometries}
        widget.search_patterns = NamedSearchPatterns(self.search_patterns(), self.string_pattern_group_key())

    def search_patterns(self) -> SearchPatterns:
        return {self.__symbol: True, self.__color: True, self.__size: True}

    def identify(self, style: Style) -> bool:
        return isinstance(style, PointCompatStyle)

    def string_pattern_group_key(self) -> str:
        return 'mapinfo_point_key'

    def settings_widget(self, iface: Plugin) -> SelectByStyleWidgetInterface:
        widget = SelectByStyleWidget(view_manager.global_parent, iface)
        widget.put_content_on_top(True)
        self.fill_widget(widget)
        return widget

    def extract_style_from_collection(self, collection_style: CollectionStyle) -> Optional[Style]:
        point_style = collection_style.point  # type: Style
        if point_style:
            return point_style.clone()
        return point_style

    def is_supported_style(self, target: PointCompatStyle, source: PointCompatStyle,
                           group_patterns: SearchGroupPatterns) -> bool:
        assert (isinstance(target, PointCompatStyle))
        assert (isinstance(source, PointCompatStyle))
        patterns = group_patterns[self.string_pattern_group_key()]
        # символ
        if patterns[self.__symbol] and target.symbol != source.symbol:
            return False
        # цвет
        if patterns[self.__color] and target.color != source.color:
            return False
        # размер
        if patterns[self.__size] and target.size != source.size:
            return False
        return True
