import codecs
import logging
import shutil
import zipfile
from pathlib import PurePosixPath, Path

from typing import Union

from .dependencies import Dependencies, PipDependencyInstaller
from .parser import IParser, MANIFEST_FILE_NAME, ENTRY_POINT

ARCHIVE_EXTENSION = 'axp'
_ARCHIVE_SUFFIX = '.' + ARCHIVE_EXTENSION


def _to_text_fileobject(fileobject_from_zip):
    return codecs.getreader('utf-8')(fileobject_from_zip)


class PackageParser(IParser):
    def parse(self, location_path: Union[str, Path]) -> dict:
        with zipfile.ZipFile(location_path) as archive:
            return self.read_archive(archive)

    def read_archive(self, archive: zipfile.ZipFile) -> dict:
        def is_root_content(path):
            return (path.endswith('/') and path.count('/') == 1) or '/' not in path

        root_content = set(filter(is_root_content, archive.namelist()))
        if (
                len(root_content) != 1 or
                not next(iter(root_content)).endswith('/')
        ):
            raise RuntimeError('Archive must contain one root directory')
        plugin_dir = PurePosixPath(next(iter(root_content)))
        try:
            entry_point = plugin_dir / ENTRY_POINT
            _ = archive.getinfo(str(entry_point))
        except KeyError:
            raise RuntimeError(f'Module must contain an entry point: {ENTRY_POINT}')
        try:
            manifest_path = plugin_dir / MANIFEST_FILE_NAME
            manifest_path_info = archive.getinfo(str(manifest_path))
        except KeyError:
            raise RuntimeError('Module must contain a manifest file')
        plugin_id = plugin_dir.name
        with archive.open(manifest_path_info) as manifest_file:
            text_fileobject = _to_text_fileobject(manifest_file)
            result = self.read_config(text_fileobject)
        result['id'] = plugin_id
        return result

    @classmethod
    def _get_id(cls, archive: zipfile.ZipFile) -> str:
        def is_root_content(path):
            return (path.endswith('/') and path.count('/') == 1) or '/' not in path

        root_content = set(filter(is_root_content, archive.namelist()))
        if (
                len(root_content) != 1 or
                not next(iter(root_content)).endswith('/')
        ):
            raise RuntimeError('Archive must contain one root directory')
        plugin_dir = PurePosixPath(next(iter(root_content)))
        return plugin_dir.name

    @staticmethod
    def supports(location_path: Union[str, Path]) -> bool:
        return location_path.endswith(_ARCHIVE_SUFFIX)

    @staticmethod
    def install(package_file, destination: Path):
        module_destination = destination / 'modules'
        dependencies_destination = destination / 'dependencies' / 'site-packages'
        assert module_destination.is_dir()
        with zipfile.ZipFile(package_file, 'r') as package:
            # extact package
            package.extractall(module_destination)
            module_id = PackageParser._get_id(package)
            module_rootdir = module_destination / module_id
            logging.info(f'Install package: {module_rootdir}')
            # install deps
            deps = Dependencies.find(module_rootdir)
            if deps:
                deps_installer = PipDependencyInstaller()
                try:
                    deps_installer.install(deps, dependencies_destination)
                except Exception as e:
                    shutil.rmtree(module_rootdir)
                    raise e
