"""
Запуск редактора стилей линий.
"""
import platform
import subprocess
from PySide2.QtWidgets import QApplication
from axipy import Position, Plugin, tr, Notifications
from pathlib import Path
from typing import Optional


class StrokeStyleEditorRunnerPlugin(Plugin):
    def __init__(self) -> None:
        self._title: str = self.tr("Редактор стилей линий")
        self._popen: Optional[subprocess.Popen] = None
        self._exec_file: Path = Path(QApplication.instance().applicationDirPath()) / "StrokeStyleEditor"

        if platform.system() == "Windows":
            self._exec_file = self._exec_file.with_suffix(".exe")

        if not self._exec_file.is_file():
            raise FileNotFoundError(self._exec_file)

        self._action = self.create_action(
            self.tr("Редактор стилей линий"),
            icon="://icons/share/64px/axioma_stroke.png",
            on_click=self.run_editor,
            tooltip=self.tr("Запуск редактора стилей линий"),
        )
        position = Position(tr("Основные"), tr("Настройки"))
        position.add(self._action)

    def unload(self) -> None:
        if self._action:
            self._action.remove()

        # Suppresses warning at delete, because detached process is wanted in this case
        if self._popen is not None:
            self._popen.returncode = 0

            self._popen.stdout.close()
            self._popen.stderr.close()

    def run_editor(self) -> None:
        if self._popen is None or self._popen.poll() is not None:
            if self._popen is not None:
                self._popen.stdout.close()
                self._popen.stderr.close()

            self._popen = subprocess.Popen(
                [str(self._exec_file)],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
            )

            try:
                outs, errs = self._popen.communicate(timeout=1)
                print(outs, errs)
            except subprocess.TimeoutExpired:
                ...

            returncode = self._popen.returncode
            if returncode is not None and returncode != 0:
                Notifications.push(self._title, "Не удалось запустить редактор.", Notifications.Critical)
        else:
            Notifications.push(self._title, "Редактор уже запущен.", Notifications.Warning)
