import sys

import configparser
import os
import site
from pathlib import Path
from ._utils import _try_except_silent


def _get_writable_app_data_location() -> Path:
    # импорт должен быть локальным
    from PySide2.QtCore import QStandardPaths, QDir
    app_data_location = QStandardPaths.writableLocation(QStandardPaths.AppDataLocation)
    if app_data_location == "":
        raise RuntimeError("app_data_location cannot be determined.")
    app_data_location = QDir(app_data_location).absolutePath()
    return Path(app_data_location)


def _get_axioma_version_from_config() -> str:
    config = configparser.ConfigParser()
    config.read(Path(__file__).parent / "sitecustomize.ini")
    axioma_major_version = config["DEFAULT"]["axioma_major_version"]
    return axioma_major_version


def _get_axioma_app_data_location() -> Path:
    app_data_location = _get_writable_app_data_location()
    if app_data_location.name != "Axioma.GIS":
        app_data_location = app_data_location / "ESTI" / "Axioma.GIS"
    axioma_version = _get_axioma_version_from_config()
    return app_data_location / f"v{axioma_version}"


def _get_axioma_python_dependencies_location() -> Path:
    axioma_app_data_location = _get_axioma_app_data_location()
    return axioma_app_data_location / "installed_modules" / "dependencies" / "site-packages"


@_try_except_silent()
def _add_dependencies_to_pythonpath() -> None:
    """
    Функция добавляет папку с зависимыми модулями в PYTHONPATH и регистрирует ее как site директорию.
    (В site директории, все файлы *.pth используются для расширения PYTHONPATH).
    """
    deps_folder = _get_axioma_python_dependencies_location()
    deps_folder_str = str(deps_folder)

    if deps_folder not in (Path(elem) for elem in sys.path):
        sys.path.insert(0, deps_folder_str)

    site.addsitedir(deps_folder_str)

    if os.environ.get("VIRTUAL_ENV", None) is None:
        os.environ["PIP_TARGET"] = str(deps_folder)
