from typing import Union, List, Optional, Tuple, Any, Iterator

from axipy.cpp_cs import ShadowLinearUnit, ShadowAreaUnit, ShadowUnit

from axipy._internal._decorator import _deprecated_by
from axipy._internal._metaclass import _MappingMetaDocumentation, _MappingMetaExtended
from axipy._internal._utils import _NoInitReadOnlyMeta

__all__ = [
    'Unit',
    'LinearUnit',
    'LinearUnits',
    'AreaUnit',
    'AreaUnits',
    'UnitValue',
]

_linear_dict = {
    "km": "kilometer",
    "m": "meter",
    "mm": "millimeter",
    "cm": "centimeter",
    "mi": "mile",
    "nmi": "nautical_mile",
    "inch": "inch",
    "ft": "foot",
    "yd": "yard",
    "survey_ft": "usFoot",
    "li": "link",
    "ch": "chain",
    "rd": "rod",
    "degree": "degree",
}

_area_dict = {
    "sq_mm": "sq_millimeter",
    "sq_cm": "sq_centimeter",
    "sq_m": "sq_meter",
    "sq_km": "sq_kilometer",
    "sq_mi": "sq_mile",
    "sq_nmi": "sq_nautical_mile",
    "sq_inch": "sq_inch",
    "sq_ft": "sq_foot",
    "sq_yd": "sq_yard",
    "sq_survey_ft": "sq_usFoot",
    "acre": "acre",
    "hectare": "hectare",
    "sq_li": "sq_link",
    "sq_ch": "sq_chain",
    "sq_rd": "sq_rod",
    "perch": "perch",
    "rood": "rood",
}

_units_dict = _linear_dict.copy()
_units_dict.update(_area_dict)


class _UnitBase(_NoInitReadOnlyMeta):
    class _DeprecDesc:

        def __set_name__(self, owner, name):
            self._name = name

        def __init__(self):
            self._inst = None

        def __get__(self, obj, obj_type):
            if self._inst is not None:
                return self._inst
            shadow_name = _units_dict.get(self._name, None)
            if shadow_name is None:
                return None

            shadow_obj = None
            wrapper = None
            if hasattr(ShadowLinearUnit, shadow_name):
                shadow_obj = getattr(ShadowLinearUnit, shadow_name)
                wrapper = LinearUnit
            elif hasattr(ShadowAreaUnit, shadow_name):
                shadow_obj = getattr(ShadowAreaUnit, shadow_name)
                wrapper = AreaUnit
            self._inst = wrapper._wrap(shadow_obj())
            return self._inst

    km = _DeprecDesc()
    """
    Километры

    :meta private:
    """
    m = _DeprecDesc()
    """
    Метры

    :meta private:
    """
    mm = _DeprecDesc()
    """
    Миллиметры

    :meta private:
    """
    cm = _DeprecDesc()
    """
    Сантиметры

    :meta private:
    """
    mi = _DeprecDesc()
    """
    Мили

    :meta private:
    """
    nmi = _DeprecDesc()
    """
    Морские мили

    :meta private:
    """
    inch = _DeprecDesc()
    """
    Дюймы

    :meta private:
    """
    ft = _DeprecDesc()
    """
    Футы

    :meta private:
    """
    yd = _DeprecDesc()
    """
    Ярды

    :meta private:
    """
    survey_ft = _DeprecDesc()
    """
    Топографические футы

    :meta private:
    """
    li = _DeprecDesc()
    """
    Линки

    :meta private:
    """
    ch = _DeprecDesc()
    """
    Чейны

    :meta private:
    """
    rd = _DeprecDesc()
    """
    Роды

    :meta private:
    """
    degree = _DeprecDesc()
    """
    Градусы

    :meta private:
    """

    sq_mm = _DeprecDesc()
    """
    Квадратные миллиметры

    :meta private:
    """
    sq_cm = _DeprecDesc()
    """
    Квадратные сантиметры

    :meta private:
    """
    sq_m = _DeprecDesc()
    """
    Квадратные метры

    :meta private:
    """
    sq_km = _DeprecDesc()
    """
    Квадратные километры

    :meta private:
    """
    sq_mi = _DeprecDesc()
    """
    Квадратные мили

    :meta private:
    """
    sq_nmi = _DeprecDesc()
    """
    Квадратные морские мили

    :meta private:
    """
    sq_inch = _DeprecDesc()
    """
    Квадратные дюймы

    :meta private:
    """
    sq_ft = _DeprecDesc()
    """
    Квадратные футы

    :meta private:
    """
    sq_yd = _DeprecDesc()
    """
    Квадратные ярды

    :meta private:
    """
    sq_survey_ft = _DeprecDesc()
    """
    Квадратные топографические футы

    :meta private:
    """
    acre = _DeprecDesc()
    """
    Акры

    :meta private:
    """
    hectare = _DeprecDesc()
    """
    Гектары

    :meta private:
    """
    sq_li = _DeprecDesc()
    """
    Квадратные линки

    :meta private:
    """
    sq_ch = _DeprecDesc()
    """
    Квадратные чейны

    :meta private:
    """
    sq_rd = _DeprecDesc()
    """
    Квадратные роды

    :meta private:
    """
    perch = _DeprecDesc()
    """
    Перчи

    :meta private:
    """
    rood = _DeprecDesc()
    """
    Руды

    :meta private:
    """

    class _DescAll:

        def __set_name__(self, owner, name):
            self._name = name

        def __init__(self):
            self._inst = None

        def __get__(self, obj, objtype=None):
            if self._inst is not None:
                return self._inst
            if self._name == "all_linear":
                # No degree in deprecated version
                _linear_dict_old = {k: v for k, v in _linear_dict.items() if k != "degree"}
                return [getattr(LinearUnit, key) for key in _linear_dict_old.keys()]
            elif self._name == "all_area":
                return [getattr(AreaUnit, key) for key in _area_dict.keys()]

    all_linear = _DescAll()

    all_area = _DescAll()


class Unit(_UnitBase):

    @classmethod
    def _wrap(cls, shadow: ShadowUnit):
        result = cls.__new__(cls)
        result._shadow = shadow
        return result

    @property
    def name(self) -> str:
        """Краткое наименование единиц измерения."""
        return self._shadow.name()

    @property
    def localized_name(self) -> str:
        """Локализованное краткое наименование единиц измерения."""
        return self._shadow.localized_name()

    @property
    def description(self) -> str:
        """Краткое описание."""
        return self._shadow.description()

    @property
    def conversion(self) -> float:
        """Коэффициент преобразования в метры."""
        return self._shadow.conversion()

    def to_unit(self, unit: Union['LinearUnit', 'AreaUnit'], value: float = 1) -> float:
        """
        Перевод значения в другие единицы измерения.

        Args:
            unit: Единицы измерения, в которые необходимо перевести значение.
            value: Значение для перевода.

        """
        if type(self) == type(unit):
            return self._shadow.conversion(unit._shadow) * value
        else:
            raise ValueError('Unit type mismatch')

    def __str__(self):
        return self.name

    def __repr__(self):
        return f"<axipy.{self.__class__.__name__} name={self.name}>"


class LinearUnit(Unit):
    """
    Линейная единица измерения.
    
    Используется для работы с координатами объектов или расстояний.

    Note:
        Получить экземпляр можно через класс :class:`axipy.LinearUnits` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/cs/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    def __eq__(self, other):
        if isinstance(other, LinearUnit):
            return self.name == other.name
        return False

    @classmethod
    @_deprecated_by("LinearUnits.values()")
    def list_all(cls) -> 'List[LinearUnit]':
        """
        Возвращает перечень всех линейных единиц измерения.

        :meta private:
        """
        return cls.all_linear

    @classmethod
    @_deprecated_by("LinearUnits[name]")
    def by_name(cls, name: str) -> Optional['LinearUnit']:
        """
        Возвращает единицу измерения по ее наименованию.
        
        Args:
            name: Наименование :attr:`name`, :attr:`localized_name` или :attr:`description`

        :meta private:
        """
        return next(filter(lambda v: name in [v.localized_name, v.name, v.description], LinearUnit.list_all()), None)

    @staticmethod
    def from_area_unit(area_unit: 'AreaUnit'):
        """
        Возвращает единицу измерения расстояния, соответствующую единице измерения площадей.
        """
        if isinstance(area_unit, AreaUnit):
            name = area_unit.name.replace("sq_", "")
            return LinearUnits.get(name)


class AreaUnit(Unit):
    """
    Единица измерения площадей.

    Note:
        Получить экземпляр можно через класс :class:`axipy.AreaUnits` по соответствующему атрибуту.

    .. literalinclude:: /../../tests/doc_examples/cs/test_example_unit.py
        :caption: Пример создания
        :pyobject: test_run_example_unit_create
        :lines: 2-
        :dedent: 4
    """

    def __eq__(self, other):
        if isinstance(other, AreaUnit):
            return self.name == other.name
        return False

    @classmethod
    @_deprecated_by("AreaUnits.values()")
    def list_all(cls) -> 'List[AreaUnit]':
        """
        Возвращает перечень всех площадных единиц измерения.

        :meta private:
        """
        return cls.all_area

    @classmethod
    @_deprecated_by("AreaUnits[name]")
    def by_name(cls, name: str) -> Optional['AreaUnit']:
        """
        Возвращает единицу измерения по ее наименованию.

        Args:
            name: Наименование :attr:`name`, :attr:`localized_name` или :attr:`description`

        :meta private:
        """
        return next(filter(lambda v: name in [v.localized_name, v.name, v.description], AreaUnit.list_all()), None)

    @staticmethod
    def from_linear_unit(linear_unit: 'LinearUnit'):
        """
        Возвращает единицу измерения площадей, соответствующую единице измерения расстояний.
        """
        if isinstance(linear_unit, LinearUnit):
            name = "sq_" + linear_unit.name
            return AreaUnits.get(name)


class _LinearUnitsMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator:
        return iter(LinearUnits._desc_keys)

    def __len__(cls):
        return len(LinearUnits._desc_keys)

    def __getitem__(cls, key):
        try:
            result = getattr(LinearUnits, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class LinearUnits(_MappingMetaDocumentation, metaclass=_LinearUnitsMappingMetaExtended):
    """
    Единицы измерения расстояний.
    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    class _KeysDesc:

        def __init__(self):
            self._keys = None

        def __get__(self, instance, owner):
            if self._keys is None:
                self._keys = [k for k, v in LinearUnits.__dict__.items() if isinstance(v, LinearUnits._LinearDesc)]
            return self._keys

    _desc_keys = _KeysDesc()

    class _LinearDesc:

        def __set_name__(self, owner, name):
            self._name = name

        def __init__(self):
            self._inst = None

        def __get__(self, obj, obj_type):
            if self._inst is not None:
                return self._inst
            shadow_name = _linear_dict.get(self._name)
            shadow_obj = getattr(ShadowLinearUnit, shadow_name)()
            self._inst = LinearUnit._wrap(shadow_obj)
            return self._inst

    km: LinearUnit = _LinearDesc()
    """Километры"""
    m: LinearUnit = _LinearDesc()
    """Метры"""
    mm: LinearUnit = _LinearDesc()
    """Миллиметры"""
    cm: LinearUnit = _LinearDesc()
    """Сантиметры"""
    mi: LinearUnit = _LinearDesc()
    """Мили"""
    nmi: LinearUnit = _LinearDesc()
    """Морские мили"""
    inch: LinearUnit = _LinearDesc()
    """Дюймы"""
    ft: LinearUnit = _LinearDesc()
    """Футы"""
    yd: LinearUnit = _LinearDesc()
    """Ярды"""
    survey_ft: LinearUnit = _LinearDesc()
    """Топографические футы"""
    li: LinearUnit = _LinearDesc()
    """Линки"""
    ch: LinearUnit = _LinearDesc()
    """Чейны"""
    rd: LinearUnit = _LinearDesc()
    """Роды"""
    degree: LinearUnit = _LinearDesc()
    """Градусы"""

    @classmethod
    def items(cls) -> List[Tuple[str, Any]]:
        """
        Возвращает список кортежей ключ-значение, где ключи это атрибуты класса, а значения это объекты класса
        :class:`axipy.LinearUnit`.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[Any]:
        """
        Возвращает список значений, где значения это объекты класса
        :class:`axipy.LinearUnit`.
        """
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Any = None):
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class _AreaUnitsMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator:
        return iter(AreaUnits._desc_keys)

    def __len__(cls):
        return len(AreaUnits._desc_keys)

    def __getitem__(cls, key):
        try:
            result = getattr(AreaUnits, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class AreaUnits(_MappingMetaDocumentation, metaclass=_AreaUnitsMappingMetaExtended):
    """
    Единицы измерения площадей.
    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    class _KeysDesc:

        def __init__(self):
            self._keys = None

        def __get__(self, instance, owner):
            if self._keys is None:
                self._keys = [k for k, v in AreaUnits.__dict__.items() if isinstance(v, AreaUnits._AreaDesc)]
            return self._keys

    _desc_keys = _KeysDesc()

    class _AreaDesc:

        def __set_name__(self, owner, name):
            self._name = name

        def __init__(self):
            self._inst = None

        def __get__(self, obj, obj_type):
            if self._inst is not None:
                return self._inst
            shadow_name = _area_dict.get(self._name)
            shadow_obj = getattr(ShadowAreaUnit, shadow_name)()
            self._inst = AreaUnit._wrap(shadow_obj)
            return self._inst

    sq_mm: AreaUnit = _AreaDesc()
    """Квадратные миллиметры"""
    sq_cm: AreaUnit = _AreaDesc()
    """Квадратные сантиметры"""
    sq_m: AreaUnit = _AreaDesc()
    """Квадратные метры"""
    sq_km: AreaUnit = _AreaDesc()
    """Квадратные километры"""
    sq_mi: AreaUnit = _AreaDesc()
    """Квадратные мили"""
    sq_nmi: AreaUnit = _AreaDesc()
    """Квадратные морские мили"""
    sq_inch: AreaUnit = _AreaDesc()
    """Квадратные дюймы"""
    sq_ft: AreaUnit = _AreaDesc()
    """Квадратные футы"""
    sq_yd: AreaUnit = _AreaDesc()
    """Квадратные ярды"""
    sq_survey_ft: AreaUnit = _AreaDesc()
    """Квадратные топографические футы"""
    acre: AreaUnit = _AreaDesc()
    """Акры"""
    hectare: AreaUnit = _AreaDesc()
    """Гектары"""
    sq_li: AreaUnit = _AreaDesc()
    """Квадратные линки"""
    sq_ch: AreaUnit = _AreaDesc()
    """Квадратные чейны"""
    sq_rd: AreaUnit = _AreaDesc()
    """Квадратные роды"""
    perch: AreaUnit = _AreaDesc()
    """Перчи"""
    rood: AreaUnit = _AreaDesc()
    """Руды"""

    @classmethod
    def items(cls) -> List[Tuple[str, Any]]:
        """
        Возвращает список кортежей ключ-значение, где ключи это атрибуты класса, а значения это объекты класса
        :class:`axipy.AreaUnit`.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[Any]:
        """
        Возвращает список значений, где значения это объекты класса
        :class:`axipy.AreaUnit`.
        """
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Any = None):
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class UnitValue:
    """
    Контейнер, который хранит значение вместе с его единицей измерения.
    
    Args:
        value: Значение.
        unit: Единица измерения, в которой содержится значение.
            Если значение не указано, принимаются метры :attr:`LinearUnits.m`.

    Пример::

        unit =  UnitValue(2, LinearUnits.km)
        print(unit)
        >>> 2 km
    """

    def __init__(self, value: float = 1, unit: Optional[Unit] = None):
        self._value = value
        if unit is None:
            self._unit = LinearUnits.m
        else:
            self._unit = unit

    @property
    def unit(self) -> Unit:
        """Единица измерения."""
        return self._unit

    @unit.setter
    def unit(self, unit: Unit):
        self._unit = unit

    @property
    def value(self) -> float:
        """Значение."""
        return self._value

    @value.setter
    def value(self, value: float):
        self._value = value

    def to_unit(self, unit) -> 'UnitValue':
        new_value = self.unit.to_unit(unit, self.value)
        return UnitValue(new_value, unit)

    def __str__(self):
        return f"{self.value} {self.unit}"

    def __repr__(self):
        return f"<axipy.{self.__class__.__name__} value={self.value} unit={self.unit}>"

    # Arithmetic operators

    def __neg__(self):
        """-self"""
        return UnitValue(-self.value, self.unit)

    def __pos__(self):
        """+self"""
        return self

    def __add__(self, other):
        """self + other"""
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value + other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value + new_other_value, self.unit)

    def __radd__(self, other):
        """other + self"""
        other, self = self, other
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value + other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value + new_other_value, self.unit)

    def __sub__(self, other):
        """self - other"""
        return self + -other

    def __rsub__(self, other):
        """other - self"""
        return -self + other

    def __mul__(self, other):
        """self * other"""
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value * other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value * new_other_value, self.unit)

    def __rmul__(self, other):
        """other * self"""
        other, self = self, other
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value * other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value * new_other_value, self.unit)

    def __truediv__(self, other):
        """self / other"""
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value / other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value / new_other_value, self.unit)

    def __rtruediv__(self, other):
        """other / self"""
        other, self = self, other
        if type(self) == type(other):
            if self.unit == other.unit:
                return UnitValue(self.value / other.value, self.unit)
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return UnitValue(self.value / new_other_value, self.unit)

    # Comparison

    def __eq__(self, other):
        if type(self) == type(other):
            if self.unit == other.unit:
                return self.value == other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value == new_other_value

        return False

    def __lt__(self, other):
        if type(self) == type(other):
            if self.unit == other.unit:
                return self.value < other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value < new_other_value

        return False

    def __le__(self, other):
        if type(self) == type(other):
            if self.unit == other.unit:
                return self.value <= other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value <= new_other_value

        return False

    def __gt__(self, other):
        if type(self) == type(other):
            if self.unit == other.unit:
                return self.value > other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value > new_other_value

        return False

    def __ge__(self, other):
        if type(self) == type(other):
            if self.unit == other.unit:
                return self.value >= other.value
            else:
                new_other_value = other.unit.to_unit(self.unit, other.value)
                return self.value >= new_other_value

        return False
