import logging
import os
import platform
import subprocess
from pathlib import Path

from typing import Optional

from .dependency_installer import DependencyInstaller, InstallError, Dependencies

__all__ = [
    'PipDependencyInstaller',
]


def python_interpreter() -> Optional[Path]:
    result = Path(os.environ['AXIOMA_PYTHON_INTERPRETER'])
    return result if result.is_file() else None


class PipDependencyInstaller(DependencyInstaller):

    @property
    def __flags(self):
        if platform.system() == 'Windows':
            return  {'creationflags': subprocess.CREATE_NO_WINDOW}
        else:
            return {}
        
    @property
    def __executable(self):
        executable = python_interpreter()
        if not executable:
            raise InstallError('Python interpreter is not found')
        return executable
    
    def install_by_name(self, package: str, dependencies_destination: Path):
        p = dependencies_destination.absolute().as_posix()
        logging.info(f'Dependency catalog: {p}')
        logging.info(f'--- Package deps: {package} ---')
        try:
            exit_data = subprocess.run(
                [str(self.__executable), '-m', 'pip', 'install', f'--target={p}', package],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                **self.__flags
            )
            exit_string = exit_data.stdout if exit_data.returncode == 0 else exit_data.stderr
            for m in Dependencies._to_list(exit_string.decode()):
                logging.info(m)
            if exit_data.returncode != 0:
                raise InstallError('Error while installing dependent packages')
        except subprocess.CalledProcessError as e:
            logging.error(f"Execution error:\"{e.cmd}\"")
            raise e
        except Exception as e:
            raise e

    def install(self, dependency: Dependencies, dependencies_destination: Path):
        for package in dependency.packages():
            self.install_by_name(package, dependencies_destination)

