from itertools import filterfalse

import os
import re
import traceback
from PySide2.QtCore import QDir, Qt, Slot, Signal
from PySide2.QtGui import QIcon, QKeyEvent
from PySide2.QtWidgets import QDialog, QPushButton, QDialogButtonBox, QFileDialog, QTableWidgetItem, QHeaderView, \
    QWidget, QMessageBox, QProgressBar
from axipy import Plugin, data_manager, CurrentSettings, \
    ExportParameters, provider_manager, Source, DataProvider, task_manager, AxipyProgressHandler, Notifications, \
    Table, AxipyAnyCallableTask, Feature
from pathlib import Path
from typing import List, Dict, Tuple, Optional, Union
from .ui.qdialog_main import Ui_Dialog

RE_EMPTY = re.compile(r"^\s*$")


class QDialogMain(QDialog):
    signal_pbar_set_max: Signal = Signal(QProgressBar, int)
    signal_pbar_add_value: Signal = Signal(QProgressBar)

    def __init__(self, plugin: Plugin, parent=None) -> None:
        super().__init__(parent)
        self._plugin = plugin
        self.title = self._plugin.tr("Модуль 'Экспорт в бд'")

        self._ui = Ui_Dialog()
        self._ui.setupUi(self)

        self._ui_ok: QPushButton = self._ui.buttonBox.button(QDialogButtonBox.Ok)
        self._ui_cancel: QPushButton = self._ui.buttonBox.button(QDialogButtonBox.Cancel)

        self.init_ui()

        self.on_data_sources_updated()
        data_manager._shadow.dataSourcesUpdated.connect(self.on_data_sources_updated)

        self._task: Optional[AxipyAnyCallableTask] = None

        self._postgre_id: str = ""
        self._oracle_id: str = ""

    def init_ui(self) -> None:
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"

        self._ui.tb_refresh_db.setIcon(QIcon.fromTheme("refresh_db"))
        self._ui.tb_refresh_db.clicked.connect(self.on_data_sources_updated)

        self._ui.tb_select_files.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_select_files.clicked.connect(self.on_open_files_clicked)

        self._ui.tb_delete_selected.setIcon(QIcon.fromTheme("delete"))
        self._ui.tb_delete_selected.clicked.connect(self.on_delete_selected_clicked)

        tb = self._ui.table_widget_tables
        tb.horizontalHeader().setSectionResizeMode(QHeaderView.Stretch)
        tb.itemSelectionChanged.connect(self.on_selection_changed)

        self._ui.le_error_file_name.setText(QDir.temp().filePath("export_error.sql"))
        self._ui.tb_error_file_name.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_error_file_name.clicked.connect(self.on_choice_error_file_name_clicked)

        self._ui.le_log_file_name.setText(QDir.temp().filePath("export_log.sql"))
        self._ui.tb_log_file_name.setIcon(QIcon.fromTheme("open"))
        self._ui.tb_log_file_name.clicked.connect(self.on_choice_log_file_name_clicked)

        self._ui.cb_out_database.currentIndexChanged.connect(self.on_cb_out_database_index_changed)

        self._ui.pbar_tables.hide()
        self._ui.pbar_features.hide()

        self._ui_ok.setText(self._plugin.tr("Экспорт"))
        self._ui_ok.setEnabled(False)

        self._ui.buttonBox.accepted.connect(self.accept)
        self._ui.buttonBox.rejected.connect(self.reject)

    def toggle_ui(self, state: bool) -> None:
        widgets = self.children()
        widgets = filter(lambda x: isinstance(x, QWidget), widgets)
        for widget in widgets:
            if widget == self._ui.buttonBox:
                continue
            getattr(widget, "setEnabled")(state)

        self._ui_ok.setEnabled(state)

    def accept(self) -> None:
        self.export_process()

    def reject(self) -> None:
        if self._task is None:
            super().reject()
        else:
            self.rejected.emit()

    def get_postgre_id(self) -> Optional[str]:
        if self._postgre_id == "":
            try:
                self._postgre_id = provider_manager.postgre.id
            except (Exception,):
                ...

        return self._postgre_id

    def get_oracle_id(self) -> Optional[str]:
        if self._oracle_id == "":
            try:
                self._oracle_id = provider_manager.oracle.id
            except (Exception,):
                ...

        return self._oracle_id

    def on_data_sources_updated(self) -> None:
        # logging.debug("on_data_sources_updated()")
        try:
            prev = self._ui.cb_out_database.currentText()
            self._ui.cb_out_database.blockSignals(True)
            self._ui.cb_out_database.clear()
            names: list = data_manager._shadow.getDbSourceDestNames()
            for name in names:
                params = data_manager._shadow.getDbSourceMap(name)
                self._ui.cb_out_database.addItem(name, params)
            self._ui.cb_out_database.blockSignals(False)
            if prev in names:
                self._ui.cb_out_database.setCurrentIndex(names.index(prev))
            else:
                self.on_cb_out_database_index_changed(self._ui.cb_out_database.currentIndex())
            self.check_enabled_export()
        except Exception as ex:
            print(ex)

    def on_cb_out_database_index_changed(self, index) -> None:
        prev = self._ui.cb_out_database_schema.currentText()
        self._ui.cb_out_database_schema.clear()
        if index == -1:
            return None

        i = self._ui.cb_out_database.currentIndex()
        map_ = self._ui.cb_out_database.itemData(i)

        list_owners = map_["listOwners"]
        current_schema = map_["currentSchema"]

        self._ui.cb_out_database_schema.blockSignals(True)
        self._ui.cb_out_database_schema.addItems(list_owners)
        if prev in list_owners:
            self._ui.cb_out_database_schema.setCurrentIndex(list_owners.index(prev))
        else:
            self._ui.cb_out_database_schema.setCurrentIndex(list_owners.index(current_schema))
        self._ui.cb_out_database_schema.blockSignals(False)

    def keyPressEvent(self, event: QKeyEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self.on_delete_selected_clicked()
        else:
            super().keyPressEvent(event)

    def on_open_files_clicked(self) -> None:
        last_open_path = CurrentSettings.LastOpenPath
        file_names, filter_ = QFileDialog.getOpenFileNames(
            self,
            "Выбор файлов для экспорта",
            str(last_open_path),
            "Файлы TAB (*.tab);;Все файлы (*.*)",
        )

        if file_names:
            CurrentSettings.LastOpenPath = Path(file_names[0]).parent
            self.add_opened_files(file_names)
            self.check_enabled_export()

    def add_opened_files(self, files: List[str]) -> None:
        tb = self._ui.table_widget_tables
        row_count = tb.rowCount()

        def file_in_table(file_arg) -> bool:
            for i_arg in range(row_count):
                if tb.item(i_arg, 0).text() == file_arg:
                    return True

            return False

        files = tuple(filterfalse(file_in_table, files))

        n = len(files)

        tb.setRowCount(row_count + n)
        for i in range(n):
            text = files[i]
            tb.setItem(i + row_count, 0, QTableWidgetItem(text))
            tb.setItem(i + row_count, 1, QTableWidgetItem(Path(text).stem))

    def on_selection_changed(self) -> None:
        self._ui.tb_delete_selected.setEnabled(bool(len(self._ui.table_widget_tables.selectedItems())))

    def check_enabled_export(self) -> None:
        enabled = bool(
            self._ui.table_widget_tables.rowCount() > 0 and
            self._ui.cb_out_database.currentIndex() != -1
        )
        self._ui_ok.setEnabled(enabled)

    def on_delete_selected_clicked(self) -> None:
        tb = self._ui.table_widget_tables
        for range_ in tb.selectedRanges():
            top = range_.topRow()
            count = range_.rowCount()
            for i in range(count):
                tb.removeRow(top)

        self.check_enabled_export()

    def on_choice_error_file_name_clicked(self) -> None:
        file_name = QFileDialog.getSaveFileName(
            self,
            "Сохранение файла ошибок",
            self._ui.le_error_file_name.text(),
            "Файлы ошибок (*.sql)",
        )
        if file_name[0]:
            self._ui.le_error_file_name.setText(file_name[0])

    def on_choice_log_file_name_clicked(self) -> None:
        file_name = QFileDialog.getSaveFileName(
            self,
            "Сохранение файла лога команд",
            self._ui.le_log_file_name.text(),
            "Файлы команд (*.sql)"
        )
        if file_name[0]:
            self._ui.le_log_file_name.setText(file_name[0])

    def export_process(self) -> None:
        self.toggle_ui(False)
        try:
            export_parameters = ExportParameters()
            export_parameters.geometryAsText = self._ui.cb_geom_as_text.isChecked()
            export_parameters.dropTable = self._ui.cb_drop_table.isChecked()
            export_parameters.createIndex = self._ui.cb_create_index.isChecked()
            export_parameters.mapCatalog = self._ui.cb_register_mapcatalog.isChecked()
            if self._ui.gb_error_file_name.isChecked():
                export_parameters.errorFile = self._ui.le_error_file_name.text()
            if self._ui.gb_log_file_name.isChecked():
                export_parameters.logFile = self._ui.le_log_file_name.text()

            chosen_database_schema = self._ui.cb_out_database_schema.currentText()
            map_: Dict[str, str] = self._ui.cb_out_database.currentData()

            current_schema = map_["currentSchema"]
            if (
                    map_["provider"] == self.get_oracle_id() and
                    export_parameters.createIndex and
                    chosen_database_schema != current_schema
            ):
                QMessageBox.critical(
                    self,
                    "Ошибка",
                    "При экспорте в схему БД oracle отличную от текущей, "
                    "создание пространственного индекса не поддерживается. "
                    f"(Текущая схема = {current_schema})",
                )
                raise Exception("Export canceled.")

            # Получение таблиц и alias
            tb = self._ui.table_widget_tables
            row_count = tb.rowCount()
            tables = {Path(tb.item(row, 0).text()): tb.item(row, 1).text() for row in range(row_count)}

            self._ui.pbar_tables.setValue(0)
            self._ui.pbar_tables.setMaximum(0)
            self._ui.pbar_features.setValue(0)
            self._ui.pbar_features.setMaximum(0)
            self._ui.pbar_tables.show()
            self._ui.pbar_features.show()

            data_manager._shadow.dataSourcesUpdated.disconnect(self.on_data_sources_updated)

            self._task = AxipyAnyCallableTask(
                self.export_task,
                row_count,
                tables,
                export_parameters,
                map_,
                chosen_database_schema,
            )

            ph = self._task.progress_handler()
            ph.finished.connect(self.on_task_finished)
            self.rejected.connect(ph.cancel)

            self.signal_pbar_set_max.connect(self.slot_pbar_set_max)
            self.signal_pbar_add_value.connect(self.slot_pbar_add_value)

        except Exception as ex:
            Notifications.push(self.title, f"Ошибка при старте экспорта: {ex}", Notifications.Critical)
            self.toggle_ui(True)
        else:
            task_manager.start_task(self._task)

    @staticmethod
    def ensure_opened_table(table_path: Path) -> Optional[Table]:
        """Поиск уже открытой таблицы"""
        table = None
        all_tables = filter(lambda do_: isinstance(do_, Table), data_manager.all_objects)
        for table_ in all_tables:
            prop = table_.properties
            if "tabFile" in prop and Path(prop["tabFile"]) == Path(table_path):
                table = table_
        return table

    def export_task(
            self,
            ph: AxipyProgressHandler,
            count: int,
            tables: Dict[Path, str],
            export_parameters: ExportParameters,
            map_: Dict[str, str],
            chosen_database_schema: str,
    ) -> None:

        original_log, tmp_log = self.init_log("logFile", export_parameters)
        original_log_errors, tmp_log_errors = self.init_log("errorFile", export_parameters)

        is_pbar_max_set = False
        success = 0

        for table_path, table_alias in tables.items():
            try:
                self.export_one_table(
                    ph,
                    table_path,
                    table_alias,
                    map_,
                    chosen_database_schema,
                    export_parameters,
                    original_log,
                    original_log_errors,
                )
            except (Exception,) as e:
                traceback.print_exc()
                Notifications.push(
                    self.title,
                    f"Файл '{table_path.name}' пропущен. Ошибка: '{e}'.",
                    Notifications.Warning,
                )
            else:
                success += 1
            finally:
                self.clean_tmp_log(export_parameters.logFile, tmp_log)
                self.clean_tmp_log(export_parameters.errorFile, tmp_log_errors)

                if not is_pbar_max_set:
                    self.signal_pbar_set_max.emit(self._ui.pbar_tables, count)
                is_pbar_max_set = True

                self.signal_pbar_add_value.emit(self._ui.pbar_tables)

                if ph.is_canceled():
                    self.notify_count(count, success)
                    return None

        self.notify_count(count, success)

    def export_one_table(
            self,
            ph: AxipyProgressHandler,
            table_path: Path,
            table_alias: str,
            map_: dict,
            chosen_database_schema: str,
            export_parameters: ExportParameters,
            original_log: Optional[str],
            original_log_errors: Optional[str],
    ):
        if RE_EMPTY.match(table_alias):
            table_alias = table_path.stem

        table_path: Path = Path(table_path)
        extension: str = table_path.suffix[1:]
        data_provider: DataProvider = provider_manager.find_by_extension(extension)

        source: Source = data_provider.get_source(filepath=str(table_path))
        source["hidden"] = True
        table: Optional[Table] = None

        close_needed = False

        try:
            table = self.ensure_opened_table(table_path)
            if table is None:
                table = source.open()
                close_needed = True

            dest_kwargs = {
                "schema": table.schema,
                "db_name": map_["db_name"],
                "host": map_["host"],
                "user": map_["user"],
                "port": map_["port"],
                "password": "",

                "export_params": export_parameters,
            }

            provider = map_["provider"]

            if provider == self.get_postgre_id():
                dest_kwargs["dataobject"] = f"{chosen_database_schema}.{table_alias}"  # Схема.Имя выходной таблицы
                dest = provider_manager.postgre.get_destination(**dest_kwargs)
            elif provider == self.get_oracle_id():
                current_schema = map_["currentSchema"]
                if current_schema == chosen_database_schema:
                    dest_kwargs["dataobject"] = table_alias
                else:
                    dest_kwargs["dataobject"] = f"{chosen_database_schema}.{table_alias}"  # Схема.Имя выходной таблицы
                dest = provider_manager.oracle.get_destination(**dest_kwargs)
            else:
                raise RuntimeError("Destination provider not found")

            features_count = table.count()
            self.signal_pbar_set_max.emit(self._ui.pbar_features, features_count)

            def feature_callback(f: Feature, row_id: int) -> Union[None, bool]:
                self.signal_pbar_add_value.emit(self._ui.pbar_features)
                if ph.is_canceled():
                    return False

            dest.export(table.items(), func_callback=feature_callback)
        except Exception as e:
            raise e
        finally:
            if table is not None and close_needed:
                task_manager.run_in_gui(table.close)

            if export_parameters.logFile:
                QDialogMain.append_log(original_log)
            if export_parameters.errorFile:
                QDialogMain.append_log(original_log_errors)

    @staticmethod
    def init_log(attr: str, export_parameters: ExportParameters, ) -> Tuple[Optional[str], Optional[str]]:
        original_log_arg = getattr(export_parameters, attr)
        if original_log_arg:
            # create empty with overwrite
            open(original_log_arg, mode='w', encoding="utf-8").close()
            tmp_log_arg = str(Path(original_log_arg).with_suffix(".tmp"))
            setattr(export_parameters, attr, tmp_log_arg)
            return original_log_arg, tmp_log_arg
        else:
            return None, None

    @staticmethod
    def append_log(path: str) -> None:
        tmp_path = Path(path).with_suffix(".tmp")
        if not tmp_path.exists():
            return None

        with open(path, mode="a", encoding="utf-8") as x:
            with open(str(tmp_path), mode='r', encoding="utf-8") as y:
                x.writelines(y.readlines())

    @staticmethod
    def clean_tmp_log(param: Optional[str], file: Optional[str]) -> None:
        if param and file:
            if os.path.exists(file):
                os.remove(file)

    @Slot(QProgressBar, int)
    def slot_pbar_set_max(self, pbar: QProgressBar, max_: int) -> None:
        pbar.setMaximum(max_)

    @Slot(QProgressBar)
    def slot_pbar_add_value(self, pbar: QProgressBar) -> None:
        v = pbar.value()
        pbar.setValue(v + 1)

    @Slot()
    def on_task_finished(self) -> None:
        # logging.debug("Task finished")

        ph = self._task.progress_handler()
        ph.finished.disconnect(self.on_task_finished)
        self.rejected.disconnect(ph.cancel)

        self.signal_pbar_set_max.disconnect(self.slot_pbar_set_max)
        self.signal_pbar_add_value.disconnect(self.slot_pbar_add_value)

        self._ui.pbar_tables.hide()
        self._ui.pbar_features.hide()

        self._task.progress_handler().deleteLater()
        # self.task.deleteLater()  # cpp object is already deleted at this moment
        self._task = None

        data_manager._shadow.dataSourcesUpdated.connect(self.on_data_sources_updated)

        self.toggle_ui(True)

    def notify_count(self, count: int, success: int) -> None:
        Notifications.push(
            self.title,
            self._plugin.tr(
                f"Экспортировано таблиц {success}/{count}."
            )
        )
