"""
Дуга по 3 точкам.
"""
from typing import List

from PySide2.QtCore import Qt, QPointF, QRectF, QPoint, QSizeF
from PySide2.QtGui import QPainter, QPen

from axipy import Notifications
from axipy import Plugin
from axipy import tr, LineString
from axipy.da import Feature, Style
from axipy.mi import Arc
from axipy.utl import Pnt, Rect
from .. import circle_utils
from ..circle_utils import CircleMapTool
from ..helper import print_exc_


class ArcByPoints(CircleMapTool):

    def __init__(self, iface: Plugin, title: str) -> None:
        super().__init__(iface, title)

    def prepare_to_draw(self, painter: QPainter, points: List[QPoint]):
        prepare_to_draw_qt_arc(painter, points)

    def prepare_to_insert(self) -> None:
        try:
            result = circle_utils.center_and_radius_of_circle(self.points)
            if result is None:
                raise Exception("None")

            center, radius = result
            if radius == 0:
                self.insert_line()
                self.clear_and_redraw()
                return None

            start_angle, end_angle = circle_utils.start_end_angles(self.points, center)
            self.insert_arc(center, radius, start_angle, end_angle)
        except (Exception,):
            print_exc_()
            Notifications.push(self.title,
                               tr("Не удалось построить дугу по 3-м точкам. Попробуйте ещё раз, изменив координаты."),
                               Notifications.Critical)
        finally:
            self.clear_and_redraw()

    def insert_arc(self, c, r, a1, a2) -> None:
        rect = Rect(c.x - r, c.y - r, c.x + r, c.y + r)

        arc = Arc(rect, startAngle=a1, endAngle=a2, cs=self.view.coordsystem)
        style = Style.from_mapinfo("Pen (1, 2, 0)")
        f_arc = Feature(geometry=arc, style=style)
        feature = f_arc

        span_angle = a2 - a1
        span_angle = round(span_angle)
        if span_angle < 0:
            span_angle += 360
        if span_angle == 0:
            self.insert_line()
            return None

        circle_utils.insert_feature(feature)

        Notifications.push(self.title, tr("Дуга успешно добавлена."), Notifications.Success)

    def insert_line(self) -> None:
        line_str = LineString(self.points[::2], cs=self.view.coordsystem)
        style = Style.from_mapinfo("Pen (1, 2, 0)")
        f_line_str = Feature(geometry=line_str, style=style)
        feature = f_line_str

        circle_utils.insert_feature(feature)

        Notifications.push(self.title, tr("Линия успешно добавлена."), Notifications.Success)


def prepare_to_draw_qt_arc(painter: QPainter, points: List[QPoint]) -> None:
    result = circle_utils.center_and_radius_of_circle(points)
    if result is None:
        return None

    center, radius = result
    if radius == 0:
        painter.drawLine(*points[::2])
        return None

    start_angle, end_angle = circle_utils.start_end_angles(points, center, device=True)

    # Запомнить старый стиль
    pen = painter.pen()
    old_pen = QPen(pen)
    # Задать новый стиль
    pen.setStyle(Qt.DashLine)
    painter.setPen(pen)

    # Нарисовать
    draw_qt_arc(painter, center, radius, start_angle, end_angle, points)

    # Возвратить старый стиль
    painter.setPen(old_pen)


def draw_qt_arc(painter: QPainter, c: Pnt, r: float, a1: float, a2: float, points: List[QPoint]) -> None:
    max_angle_to_draw_line = 5

    q_point_f = QPointF(c.x - r, c.y - r)
    size = 2 * r
    q_size_f = QSizeF(size, size)
    q_rect_f = QRectF(q_point_f, q_size_f)

    span_angle = a2 - a1
    if span_angle < 0:
        span_angle += 360

    span_angle_int = round(span_angle)

    if span_angle_int <= max_angle_to_draw_line:
        painter.drawLine(*points[::2])
        return None
    qt_span_angle = span_angle_int * 16

    qt_start_angle = round(a1) * 16

    painter.drawArc(q_rect_f, qt_start_angle, qt_span_angle)
