from functools import wraps
from typing import Optional, List

import axipy.app
from PySide2.QtCore import Signal, QObject
from PySide2.QtWidgets import QWidget
from axipy._internal._decorator import InitOnce
from axipy.cpp_gui import ShadowWidgetManager
from axipy.da import Table
from axipy.render import Map, Report

from .gui_class import gui_instance
from .view import View, MapView, TableView, ReportView, LegendView


def _add_view(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        result = func(*args, **kwargs)

        mainwindow = axipy.app.mainwindow
        if isinstance(result, View) and mainwindow.is_valid:
            window = mainwindow.add(result)
            window.show()

        return result
    return wrapper


__all__ = ["ViewManager", "view_manager"]


class ViewManager:
    """Менеджер содержимого окон.

    Note:
        Используйте готовый экземпляр этого класса :attr:`view_manager`.

    .. literalinclude:: /../../tests/doc_examples/gui/test_example_view_manager.py
        :caption: Пример использования
        :pyobject: example_view_manager
        :lines: 2-
        :dedent: 4

    """

    class _ViewManagerSignals(QObject):
        mainwindow_activated = Signal()

    def __init__(self):
        self._signals = ViewManager._ViewManagerSignals()
        self.__mainwindow_activated = self._signals.mainwindow_activated

    @property
    def mainwindow_activated(self) -> Signal:
        """
        Возникает когда главное окно приложения инициализировано и активировано. Данное событие
        может использоваться в плагинах когда есть необходимость обратиться к главному окну приложения.
        Но ввиду того, что сами плагины загружаются до инициализации главного окна, данную процедуру
        можно выполнить используя данное событие (:attr:`axipy.AxiomaInterface.window`).

        :rtype: Signal[]
        """
        return self.__mainwindow_activated


    @InitOnce
    def _view_manager(self):
        return ShadowWidgetManager(gui_instance._shadow)

    @property
    def shadow(self):
        return self._view_manager()

    @property
    def active_changed(self) -> Signal:
        """
        Активное окно изменилось.

        :rtype: Signal[]
        """
        return self._view_manager().activeWidgetChanged

    @property
    def count_changed(self) -> Signal:
        """
        Количество окон изменилось.

        :rtype: Signal[]
        """
        return self._view_manager().countChanged

    def activate(self, view: View):
        """Делает заданное окно активным.

        Args:
            view: Содержимое окна.
        """
        self._view_manager().activate(view._shadow)

    def close(self, view: View):
        """Закрывает окно.

        Args:
            view: Содержимое окна.
        """
        self._view_manager().close(view._shadow)

    def close_all(self):
        """Закрывает все окна."""
        for w in self.views:
            self.close(w)

    @property
    def active(self) -> Optional[View]:
        """Текущее активное окно.
        
        Return:
          None, если нет активных окон.
        """
        return View._wrap(self._view_manager().active())

    @active.setter
    def active(self, view: View):
        self._view_manager().activate(view._shadow)

    @property
    def views(self) -> List[View]:
        """Список всех известных окон."""
        return list(iter(self))

    def __filter_view(self, _nameType):
        return list(filter(lambda v: isinstance(v, _nameType), iter(self)))

    @property
    def mapviews(self) -> List[MapView]:
        """Список всех окон с картами.
        
        Пример::

            for v in view_manager.mapviews:
                print('Widget:', v.title)

            >>> Widget: Карта: world
            >>> Widget: Карта: rus_obl
        """
        return self.__filter_view(MapView)

    @property
    def reportviews(self) -> List[ReportView]:
        """Список всех окон с отчетами."""
        return self.__filter_view(ReportView)

    @property
    def tableviews(self) -> List[TableView]:
        """Список всех окон с таблицами просмотра."""
        return self.__filter_view(TableView)

    @property
    def legendviews(self) -> List[LegendView]:
        """Список всех окон с легендами."""
        return self.__filter_view(LegendView)

    @property
    def count(self) -> int:
        """Количество окон."""
        return self.__len__()

    def __len__(self):
        return len(self.views)

    def __getitem__(self, key: str) -> View:
        """Возвращает существующее окно по имени.

        Raises:
            KeyError: Окно с заданным именем не найдено.
        """
        found = self._find_by_name(key)
        if found is None:
            raise KeyError(f"Key '{key}' not found")
        return found

    def __contains__(self, key: str) -> bool:
        """Проверяет существование окна с заданным именем.

        Args:
            key: Имя окна.
        """
        found = self._find_by_name(key)
        return found is not None

    def _find_by_name(self, name: str):
        for view in self.views:
            if view.widget.windowTitle() == name:
                return View._wrap(view)
        return None

    def __iter__(self):
        return (View._wrap(view) for view in self._view_manager().views())

    @_add_view
    def create_mapview(self, map: Map) -> MapView:
        """Создает окно из для переданного объекта карты.

        Args:
            map: Карта.

        Note:

            Переданная карта копируется.

        Return:
            Окно карты.
        """
        if not isinstance(map, Map):
            raise TypeError('Parameter is not Map instance.')
        return View._wrap(gui_instance._shadow.createMapView(map._shadow))

    @_add_view
    def create_tableview(self, table: Table) -> TableView:
        """Создает окно в виде табличного представления из объекта данных.

        Args:
            table: Таблица.

        Return:
            Окно таблицы.
        """
        if not isinstance(table, Table):
            raise TypeError('Parameter is not Table instance.')
        return View._wrap(gui_instance._shadow.createTableView(table._shadow))

    @_add_view
    def create_reportview(self, report: Report = None) -> ReportView:
        """Создает окно с планом отчета.

        Args:
            report: План отчета. Если не передан, то создается по умолчанию.

        Return:
            Окно отчета.
        """
        if report is not None and not isinstance(report, Report):
            raise TypeError('Parameter is not Report instance.')
        r = report._shadow if report is not None else None
        return View._wrap(gui_instance._shadow.createReportView(r))

    @_add_view
    def create_legendview(self, mapview: MapView) -> LegendView:
        """Создает окно легенды для карты.

        Args:
            mapview: Окно с картой.

        Return:
            Окно с легендой.
        """
        return View._wrap(gui_instance._shadow.createLegendView(mapview._shadow))

    @property
    def global_parent(self) -> QWidget:
        """Может использоваться как 'parent' при использовании стандартных диалогов Qt. Использование данного свойства
        решает проблему, когда окно показывается в панели задач как отдельное приложение.
        
        Пример::
         
            if QMessageBox.question(view_manager.global_parent, 'Вопрос', 'Отменить действие?') == QMessageBox.Yes:
                pass
        """
        return gui_instance._shadow.global_parent()

    def create_layer_control(self):
        return gui_instance._shadow.createLayerControlWidget()

    def create_data_catalog(self):
        return gui_instance._shadow.createDataCatalogWidget()


view_manager = ViewManager()
