from typing import Optional, Any, Union, List, Tuple, Iterator

from PySide2.QtCore import Signal
from axipy.cpp_core_core import StateObserver, DefaultKeys as ShadowDefaultKeys, ValueObserver

from axipy._internal._exceptions import _generate_type_error
from axipy._internal._metaclass import _MappingMetaExtended, _MappingMetaDocumentation
from axipy._internal._shadow_instance_factory import _ShadowManager
from .state_manager_ import DefaultKeys
from axipy._internal._utils import _NoInit

__all__ = [
    "Observer",
    "ObserverManager",
]


class Observer(_NoInit):
    """
    Наблюдатель
    """

    def __init__(self) -> None:
        self._shadow: Optional[ValueObserver] = None
        self._name: Optional[str] = None
        super().__init__()

    @classmethod
    def _wrap(cls, shadow: ValueObserver, name: str) -> 'Observer':
        v = cls.__new__(cls)
        v._shadow = shadow
        v._name = name
        return v

    @property
    def value(self) -> Any:
        """
        Устанавливает или возвращает значение наблюдателя.
        При изменении значения испускается сигнал :attr:`changed`.
        """
        return self._shadow.value()

    @value.setter
    def value(self, value: Any):
        """
        Устанавливает значение.
        При изменении значения испускается сигнал :attr:`changed`.

        Args:
            value: Новое значение.

        """
        self._shadow.setValue(value)

    @property
    def name(self) -> str:
        """Возвращает имя наблюдателя."""
        return self._name

    @property
    def changed(self) -> Signal:
        """
        Сигнал об изменении значения.

        :rtype: Signal[Any]

        ::

            from axipy import ObserverManager

            def print_func(value):
                print(value)

            ObserverManager.Selection.changed.connect(print_func)

        """
        return self._shadow.changed

    def __repr__(self) -> str:
        if self._shadow is None:
            value = None
        else:
            value = self.value

        return f"<axipy.{self.__class__.__name__} name={self.name} value={value}>"


class _CustomMappingMetaExtended(_MappingMetaExtended):
    class _ShadowDesc:

        def __init__(self) -> None:
            self._s = None

        def __get__(self, _obj, _objtype=None) -> StateObserver:
            if self._s is None:
                self._s = _ShadowManager.state_observer
            return self._s

    _shadow: StateObserver = _ShadowDesc()

    def __iter__(cls) -> Iterator:
        return iter(cls._shadow.keys())

    def __len__(cls) -> int:
        return len(cls._shadow.keys())

    def __getitem__(cls, key: str) -> Observer:
        shadow = cls._shadow.find(key)
        if shadow is None:
            raise KeyError(key)
        else:
            return Observer._wrap(shadow, key)


class ObserverManager(_MappingMetaDocumentation, metaclass=_CustomMappingMetaExtended):
    """
    Наблюдатели за состоянием.
    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Поддерживает обращение по индексу.
    """

    class _ShadowDesc:

        def __init__(self) -> None:
            self._s = None

        def __get__(self, _obj, _objtype=None) -> StateObserver:
            if self._s is None:
                self._s = _ShadowManager.state_observer
            return self._s

    _shadow: StateObserver = _ShadowDesc()

    class _Desc:

        def __set_name__(self, owner, name: str):
            self._name = name

        def __get__(self, _obj, _obj_name) -> Observer:
            value_observer = _ShadowManager.state_observer.find(self._name)
            return Observer._wrap(value_observer, self._name)

    Selection: Observer = _Desc()
    """Есть выборка"""

    Editable: Observer = _Desc()
    """Активная карта имеет редактируемый слой"""

    SelectionEditable: Observer = _Desc()
    """Карта имеет редактируемый слой и есть выделенные объекты на одном из слоев карты"""

    SelectionEditableIsSame: Observer = _Desc()
    """Карта имеет редактируемый слой и выборку на этом слое"""

    ActiveView: Observer = _Desc()
    """Есть активное окно"""

    ActiveMapView: Observer = _Desc()
    """Есть активное окно карты"""

    ActiveTableView: Observer = _Desc()
    """Есть активное окно таблицы"""

    HasTables: Observer = _Desc()
    """Открыта хотя бы одна таблица"""

    @classmethod
    def items(cls) -> List[Tuple[str, Observer]]:
        """
        Возвращает список кортежей ключ-значение, где ключи это имена наблюдателей, а значения это объекты класса
        :class:`axipy.Observer`.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это имена наблюдателей."""
        return super().keys()

    @classmethod
    def values(cls) -> List[Observer]:
        """
        Возвращает список значений, где значения это объекты класса
        :class:`axipy.Observer`.
        """
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Any = None) -> Optional[Observer]:
        """
        Возвращает значение по ключу, где ключ это имя наблюдателя, а значение это объект класса
        :class:`axipy.Observer`. Если ключа нет, возвращается значение по умолчанию.
        """
        return super().get(key, default_value)

    @classmethod
    def remove(cls, name: str) -> None:
        """
        Удаляет наблюдатель по имени.

        :param name: Имя наблюдателя.
        """

        cls._shadow.remove(name)

    @classmethod
    def create(cls, name: str, init_value: Any) -> Observer:
        """
        Создает наблюдатель.

        :param name: Имя наблюдателя.
        :param init_value: Начальное значение наблюдателя.
        :return: Наблюдатель.
        """
        if not isinstance(name, str):
            raise _generate_type_error(type(name), str)
        return Observer._wrap(cls._shadow.create(name, init_value), name)

    @staticmethod
    def _to_name(key: Union[str, DefaultKeys, Observer, None]) -> Optional[str]:
        if key is None:  # Когда наблюдатель не нужен
            return None
        elif isinstance(key, Observer):
            return key.name
        elif isinstance(key, str):
            return key
        elif isinstance(key, (DefaultKeys, ShadowDefaultKeys.Key)):
            return ShadowDefaultKeys.names()[key]
        else:
            raise TypeError(f"Wrong observer_name type '{type(key), key}'")
