from enum import Enum

from PySide2.QtCore import QSize
from PySide2.QtGui import QFont, QColor

from axipy.da import Point

__all__ = [
    "LabelOverlap",
    "LabelBackgroundType",
    "LabelLinePosition",
    "LabelAreaPosition",
    "LabelAreaInterior",
    "LabelHorizontalAlign",
    "LabelLayoutPosition",
    "LabelLayout",
    "Label",
    "CustomLabelEndType",
    "CustomLabelProperties",
]


class LabelOverlap(int, Enum):
    """Стратегия при наложении подписей :attr:`Label.placementPolicy`."""

    AllowOverlap = 0
    """Допускать перекрытие меток (по умолчанию)."""
    DisallowOverlap = 1
    """Не допускать перекрытие меток."""
    OtherPosition = 2
    """Пробовать найти для метки новую позицию."""


class LabelBackgroundType(int, Enum):
    """Фон подписи :attr:`Label.backgroundType`."""

    Empty = 0
    """Отсутствует"""
    Outline = 1
    """Кайма"""
    Frame = 2
    """Фломастер"""


class LabelLinePosition(int, Enum):
    """Режим подписей для линий :attr:`Label.linePosition`."""

    Horizontal = 0
    """Горизонтально"""
    Parallel = 1
    """Вдоль сегмента"""
    FollowPath = 2
    """Вдоль линии"""


class LabelAreaPosition(int, Enum):
    """Режим подписей для областей :attr:`Label.areaPosition`."""

    Centroid = 0
    """У центроида"""
    Horizontal = 1
    """Горизонтально"""
    Vertical = 2
    """Вертикально"""
    Automatic = 3
    """Автоматически"""


class LabelAreaInterior(int, Enum):
    """Выбор участка для областей :attr:`Label.areaInterior`."""

    Max = 0
    """Наибольший"""
    Centroid = 1
    """Ближайший к центроиду"""


class LabelHorizontalAlign(int, Enum):
    """Горизонтальное выравнивание подписи :attr:`Label.horizontalAlign`."""

    Flat = 1,
    """По центру пологого участка (можно задать угол на вкладке Дополнительно)"""
    Center = 2,
    """Середина подписи совпадает с центром линии"""
    Begin = 3,
    """Подпись располагается в начале линии"""
    End = 4
    """Подпись располагается в конце линии"""


class LabelLayoutPosition(int, Enum):
    """Относительное положение подписи :attr:`LabelLayout.position`."""

    Center = 0,
    """По центру"""
    TopLeft = 1,
    """Сверху слева"""
    Top = 2,
    """Сверху"""
    TopRight = 3,
    """Сверху справа"""
    Right = 4,
    """Справа"""
    BottomRight = 5,
    """Снизу справа"""
    Bottom = 6,
    """Снизу"""
    BottomLeft = 7,
    """Снизу слева"""
    Left = 8
    """Слева"""


class LabelLayout:
    """Положение подписей :class:`Label`."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, _shadow):
        result = cls.__new__(cls)
        result._shadow = _shadow
        return result

    @property
    def position(self) -> LabelLayoutPosition:
        """Расположение подписи."""
        return LabelLayoutPosition(self._shadow.get_pos())

    @position.setter
    def position(self, p: LabelLayoutPosition):
        self._shadow.set_pos(p)

    @property
    def visible(self) -> bool:
        """Видимость подписи для данного типа геометрии. По умолчанию True"""
        return self._shadow.get_visible()

    @visible.setter
    def visible(self, v: bool):
        self._shadow.set_visible(v)

    @property
    def offset(self) -> QSize:
        """Смещение. Интервал значений (0..100)"""
        return self._shadow.get_offset()

    @offset.setter
    def offset(self, v: QSize):
        self._shadow.set_offset(v)


class Label:
    """
    Метки слоя. Доступны через свойство векторного слоя :attr:`VectorLayer.label`.

    .. literalinclude:: /../../tests/doc_examples/render/test_example_layer.py
        :caption: Пример использования.
        :pyobject: test_run_example_layer_label
        :lines: 3-
        :dedent: 4
    """

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, _shadow):
        result = cls.__new__(cls)
        result._shadow = _shadow
        return result

    @property
    def text(self) -> str:
        """Наименование атрибута таблицы либо выражение для метки, которое может основываться на одном или нескольких атрибутах."""
        return self._shadow.get_label()

    @text.setter
    def text(self, n: str):
        self._shadow.set_label(n)

    @property
    def visible(self) -> bool:
        """Управляет видимостью меток."""
        return self._shadow.get_label_visible()

    @visible.setter
    def visible(self, v: bool):
        self._shadow.set_label_visible(v)

    @property
    def placementPolicy(self) -> LabelOverlap:
        """Принцип наложения меток на слой карты. По умолчанию LabelOverlap.AllowOverlap"""
        return LabelOverlap(self._shadow.get_label_placement_policy())

    @placementPolicy.setter
    def placementPolicy(self, n: LabelOverlap):
        self._shadow.set_label_placement_policy(n)

    @property
    def color(self) -> QColor:
        """Цвет шрифта меток."""
        return self._shadow.get_label_color()

    @color.setter
    def color(self, v: QColor):
        self._shadow.set_label_color(v)

    @property
    def opacity(self) -> int:
        """Прозрачность (0..100). По умолчанию 100 (Непрозрачно)."""
        return self._shadow.get_label_opacity()

    @opacity.setter
    def opacity(self, v: int):
        self._shadow.set_label_opacity(v)

    @property
    def rangeEnabled(self) -> bool:
        """Показывать в пределах. Если True, используются свойства :attr:`rangeMin` и :attr:`rangeMax`.
    По умолчанию False."""
        return self._shadow.get_label_rangeEnabled()

    @rangeEnabled.setter
    def rangeEnabled(self, v: bool):
        self._shadow.set_label_rangeEnabled(v)

    @property
    def rangeMin(self) -> float:
        """Минимальный предел показа с метрах при включенном свойстве :attr:`rangeEnabled`."""
        return self._shadow.get_label_rangeMin()

    @rangeMin.setter
    def rangeMin(self, v: float):
        self._shadow.set_label_rangeMin(v)

    @property
    def rangeMax(self) -> float:
        """Максимальный предел показа с метрах при включенном свойстве :attr:`rangeEnabled`."""
        return self._shadow.get_label_rangeMax()

    @rangeMax.setter
    def rangeMax(self, v: float):
        self._shadow.set_label_rangeMax(v)

    @property
    def font(self) -> QFont:
        """Шрифт."""
        return self._shadow.get_label_font()

    @font.setter
    def font(self, v: QFont):
        self._shadow.set_label_font(v)

    @property
    def spacing(self) -> bool:
        """Разрядка. По умолчанию False"""
        return self._shadow.get_label_spacing()

    @spacing.setter
    def spacing(self, v: bool):
        self._shadow.set_label_spacing(v)

    @property
    def shadow(self) -> bool:
        """Тень. По умолчанию False"""
        return self._shadow.get_label_shadow()

    @shadow.setter
    def shadow(self, v: bool):
        self._shadow.set_label_shadow(v)

    @property
    def backgroundType(self) -> LabelBackgroundType:
        """Фон подписи. По умолчанию отсутствует"""
        return LabelBackgroundType(self._shadow.get_label_backgroundType())

    @backgroundType.setter
    def backgroundType(self, v: LabelBackgroundType):
        self._shadow.set_label_backgroundType(v)

    @property
    def backgroundColor(self) -> QColor:
        """Цвет фона"""
        return self._shadow.get_label_backgroundColor()

    @backgroundColor.setter
    def backgroundColor(self, v: QColor):
        self._shadow.set_label_backgroundColor(v)

    @property
    def backgroundSize(self) -> int:
        """Толщина фона в пунктах."""
        return self._shadow.get_label_backgroundSize()

    @backgroundSize.setter
    def backgroundSize(self, v: int):
        self._shadow.set_label_backgroundSize(v)

    @property
    def lineKeepDirection(self) -> bool:
        """Направление текста строится вдоль направления линии. По умолчанию False."""
        return self._shadow.get_label_lineKeepDirection()

    @lineKeepDirection.setter
    def lineKeepDirection(self, v: bool):
        self._shadow.set_label_lineKeepDirection(v)

    @property
    def linePosition(self) -> LabelLinePosition:
        """Режим подписей для линий. По умолчанию LabelLinePosition.FollowPath."""
        return self._shadow.get_label_linePosition()

    @linePosition.setter
    def linePosition(self, v: LabelLinePosition):
        self._shadow.set_label_linePosition(v)

    @property
    def areaPosition(self) -> LabelAreaPosition:
        """Режим подписей для областей. По умолчанию LabelAreaPosition.Horizontal."""
        return self._shadow.get_label_areaPosition()

    @areaPosition.setter
    def areaPosition(self, v: LabelAreaPosition):
        self._shadow.set_label_areaPosition(v)

    @property
    def areaInterior(self) -> LabelAreaInterior:
        """Режим подписей для областей. По умолчанию LabelAreaInterior.Centroid."""
        return self._shadow.get_label_areaInterior()

    @areaInterior.setter
    def areaInterior(self, v: LabelAreaInterior):
        self._shadow.set_label_areaInterior(v)

    @property
    def pointLayout(self) -> LabelLayout:
        """Положение подписей для точек."""
        return LabelLayout._wrap(self._shadow.get_label_pointLayout())

    @pointLayout.setter
    def pointLayout(self, v: LabelLayout):
        self._shadow.set_label_pointLayout(v._shadow)

    @property
    def lineLayout(self) -> LabelLayout:
        """Положение подписей для линий."""
        return LabelLayout._wrap(self._shadow.get_label_lineLayout())

    @lineLayout.setter
    def lineLayout(self, v: LabelLayout):
        self._shadow.set_label_lineLayout(v._shadow)

    @property
    def areaLayout(self) -> LabelLayout:
        """Положение подписей для областей."""
        return LabelLayout._wrap(self._shadow.get_label_areaLayout())

    @areaLayout.setter
    def areaLayout(self, v: LabelLayout):
        self._shadow.set_label_areaLayout(v._shadow)

    @property
    def useClip(self) -> bool:
        """Использовать динамические подписи. По умолчанию False."""
        return self._shadow.get_label_useClip()

    @useClip.setter
    def useClip(self, v: bool):
        self._shadow.set_label_useClip(v)

    @property
    def supressDuplicates(self) -> bool:
        """Запретить повтор подписей. Подписи с одинаковым текстом на этом слое будут отображаться 
        один раз. По умолчанию False.
        """
        return self._shadow.get_label_supressDuplicates()

    @supressDuplicates.setter
    def supressDuplicates(self, v: bool):
        self._shadow.set_label_supressDuplicates(v)

    @property
    def horizontalAlign(self) -> LabelHorizontalAlign:
        """Горизонтальное выравнивание подписей. По умолчанию `LabelHorizontalAlign.Flat`"""
        return LabelHorizontalAlign(self._shadow.get_label_horizontalAlign())

    @horizontalAlign.setter
    def horizontalAlign(self, v: LabelHorizontalAlign):
        self._shadow.set_label_horizontalAlign(v)

    @property
    def overhang(self) -> int:
        """Максимальный свес для линии (в %)."""
        return self._shadow.get_label_overhang()

    @overhang.setter
    def overhang(self, v: int):
        self._shadow.set_label_overhang(v)


class CustomLabelEndType(int, Enum):
    """
    Тип выноски для метки :attr:`CustomLabelProperties.endType`.
    """

    EndNone = 0
    """Не отображать (по умолчанию)"""
    Line = 1
    """Линия"""
    Arrow = 2
    """Стрелка"""


class CustomLabelProperties:
    """Свойства выносной метки. Используется при задании параметров положения метки карты
    :meth:`CustomLabels.set` или получения их :meth:`CustomLabels.get`.
    """
    def __init__(self):
        self.__dict = dict()
    
    @property
    def expression(self) -> str:
        """Текст. Если не задано, используется базовая формула для слоя."""
        if 'expression' in self.__dict:
            return self.__dict['expression']
        return None
    
    @expression.setter
    def expression(self, v: str):
        self.__dict['expression'] = v

    @property
    def angle(self) -> float:
        """Угол поворота текста в градусах"""
        if 'angle' in self.__dict:
            return self.__dict['angle']
        return 0
    
    @angle.setter
    def angle(self, v: float):
        self.__dict['angle'] = v

    @property
    def endType(self) -> CustomLabelEndType:
        """Тип выноски. По умолчанию `CustomLabelEndType.EndNone`"""
        if 'endType' in self.__dict:
            return CustomLabelEndType(self.__dict['endType'])
        return CustomLabelEndType.EndNone
    
    @endType.setter
    def endType(self, v: CustomLabelEndType):
        self.__dict['endType'] = v

    @property
    def position(self) -> Point:
        """Переопределенное положение метки. Если используется по умолчанию, возвращается None.
        Если при задании не указана система координат, используется СК слоя, которому принадлежит метка.
        """
        from axipy.da import Geometry
        if 'pos' in self.__dict:
            p = self.__dict['pos']
            if 'coord' in p:
                cs = self.__dict['cs'] if 'cs' in self.__dict else None
                return Geometry.from_wkt(p['coord'], cs)
        return None
    
    @position.setter
    def position(self, p: Point):
        v = dict()
        v['coord'] = p.to_wkt()
        if p.coordsystem:
            v['cs'] = p.coordsystem.to_string()
        self.__dict['pos'] = v

    def __repr__(self):
        return str(self.__dict)

    def _to_dict(self):
        return self.__dict

    def _set_dict(self, d):
        self.__dict = d
