from typing import List, Optional, Union

from axipy.cs import CoordSystem

__all__ = [
    "Attribute",
    "Schema"
]


class Attribute:
    """Атрибут схемы таблицы.

    Используется для создания и инспектирования атрибутов и схем :class:`axipy.Schema`.
    Для создания атрибутов используйте функции :meth:`string`, :meth:`decimal` и другие.

    Args:
        name: Название.
        typedef: Описание типа.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания
        :pyobject: example_attribute_create
        :lines: 2-
        :dedent: 4
    """

    _TYPES = ['bool', 'date', 'time', 'datetime',
              'string', 'double', 'int', 'decimal']
    DEFAULT_STRING_LENGTH = 80
    DEFAULT_DECIMAL_LENGTH = 10
    DEFAULT_DECIMAL_PRECISION = 5

    def __init__(self, name: str, typedef: str):
        self._name = name
        self._typedef = typedef

    @property
    def name(self) -> str:
        """Имя атрибута."""
        return self._name

    @property
    def typedef(self) -> str:
        """Описание типа.

        Строка вида ``<тип>[:длина][.точность]``.
        """
        return self._typedef

    @property
    def length(self) -> int:
        """Длина атрибута."""
        return self._type_length(self._typedef)

    @property
    def precision(self) -> int:
        """Точность."""
        return self._type_precision(self._typedef)

    @property
    def type_string(self) -> str:
        """Тип в виде строки без длины и точности."""
        return next(filter(lambda t: self._typedef.startswith(t), self._TYPES))

    @staticmethod
    def string(name: str, length: int = DEFAULT_STRING_LENGTH) -> 'Attribute':
        """Создает атрибут строкового типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута.
        """
        return Attribute(name, f'string:{length}')

    @staticmethod
    def decimal(name: str, length: int = DEFAULT_DECIMAL_LENGTH,
                precision: int = DEFAULT_DECIMAL_PRECISION) -> 'Attribute':
        """Создает атрибут десятичного типа.

        Args:
            name: Имя атрибута.
            length: Длина атрибута. Количество символов, включая запятую.
            precision: Число знаков после запятой.
        """
        return Attribute(name, f'decimal:{length}.{precision}')

    @staticmethod
    def integer(name: str) -> 'Attribute':
        """Создает атрибут целого типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'int')

    @staticmethod
    def float(name: str) -> 'Attribute':
        """Создает атрибут вещественного типа.

        То же, что и :meth:`double`

        Args:
            name: Имя атрибута.
        """
        return Attribute.double(name)

    @staticmethod
    def double(name: str) -> 'Attribute':
        """Создает атрибут вещественного типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'double')

    @staticmethod
    def bool(name: str) -> 'Attribute':
        """Создает атрибут логического типа.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'bool')

    @staticmethod
    def date(name: str) -> 'Attribute':
        """Создает атрибут типа дата.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'date')

    @staticmethod
    def time(name: str) -> 'Attribute':
        """Создает атрибут типа время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'time')

    @staticmethod
    def datetime(name: str) -> 'Attribute':
        """Создает атрибут типа дата и время.

        Args:
            name: Имя атрибута.
        """
        return Attribute(name, 'datetime')

    def __eq__(self, other: Union['Attribute', str]) -> bool:
        if isinstance(other, str):
            return self._name == other
        return self._name == other._name and self._typedef == other._typedef

    @staticmethod
    def _type_length(type_definition: str) -> int:
        pos = type_definition.rfind(':')
        if pos < 0:
            return 0
        start = pos + 1
        end = type_definition.rfind('.')
        end = len(type_definition) if end < 0 else end
        length = type_definition[start:end]
        return int(length)

    @staticmethod
    def _type_precision(type_definition: str) -> int:
        pos = type_definition.rfind('.')
        if pos < 0:
            return 0
        start = pos + 1
        length = type_definition[start:]
        return int(length)


class Schema(list):
    """Схема таблицы. Представляет собой список атрибутов :class:`axipy.Attribute`.
    Организован в виде :class:`list` и свойством `coordsystem`.
    При задании `coordsystem` создается геометрический атрибут.

    Args:
        *attributes: Атрибуты.
        coordsystem: Система координат для геометрического атрибута. Может быть задана или в виде строки (подробнее :meth:`axipy.cs.CoordSystem.from_string`)
                или как объект СК :class:`axipy.cs.CoordSystem`.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания
        :pyobject: example_schema_create
        :lines: 2-
        :dedent: 4

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример создания из списка
        :pyobject: example_schema_create_from_list
        :lines: 2-
        :dedent: 4

    Имеет стандартные функции работы со списком.

    .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
        :caption: Пример операций
        :pyobject: example_schema_attrs
        :lines: 2-
        :dedent: 4

    """

    _COORDSYSTEM = 'coordsystem'
    _PROPERTIES = 'properties'

    def __init__(self, *attributes: Attribute, coordsystem: Union[str, CoordSystem] = None):
        self.coordsystem = coordsystem
        if type(attributes) == tuple and len(attributes) in [1, 2] and type(attributes[0]) == list:
            super().__init__(attributes[0])
            if (len(attributes) == 2 and type(attributes[1]) == CoordSystem):
                self.coordsystem = attributes[1]
        else:
            super().__init__(attributes)

    @classmethod
    def _from_dict(cls, dictionary: dict):
        attrs = [Attribute(*attr) for attr in dictionary[cls._PROPERTIES]]
        crs = dictionary[cls._COORDSYSTEM] if cls._COORDSYSTEM in dictionary else None
        return Schema(*attrs, coordsystem=crs)

    def to_dict(self) -> dict:
        result = dict()
        if self._coordsystem is not None:
            result.update(
                {self._COORDSYSTEM: self._coordsystem._shadow.get_str()})
        result.update(
            {self._PROPERTIES: [[attr.name, attr.typedef] for attr in self]})
        return result

    @property
    def attribute_names(self) -> List[str]:
        """Возвращает список имен атрибутов."""
        return [a.name for a in self]

    def insert(self, index: int, attr: Attribute):
        """Вставляет атрибут.

        Args:
            index: Индекс, по которому производится вставка.
            attr: Атрибут.
        """
        if not isinstance(attr, Attribute):
            raise TypeError(f'Argument is not of type {Attribute}.')
        super().insert(index, attr)

    def __contains__(self, name: str) -> bool:
        return super().__contains__(name)

    @property
    def coordsystem(self) -> Optional[CoordSystem]:
        """Система координат.

        Returns:
            None, если СК отсутствует.

        See also:
            :attr:`axipy.Table.is_spatial`

        .. literalinclude:: /../../tests/doc_examples/da/test_example_schema.py
            :caption: Пример использования
            :pyobject: example_schema_coordsystem
            :lines: 2-
            :dedent: 4
        """
        return self._coordsystem

    @coordsystem.setter
    def coordsystem(self, cs: Union[CoordSystem, str]):
        if isinstance(cs, CoordSystem):
            self._coordsystem = cs
        elif isinstance(cs, str):
            self._coordsystem = CoordSystem.from_string(cs)
        else:
            self._coordsystem = None

    def by_name(self, n: str) -> Attribute:
        '''Возвращает атрибут по его имени. Если такого имени не существует, возвращает None.
        
        Args:
            n: Наименование атрибута.
        '''
        for a in filter(lambda f: f.name.lower() == n.lower(), self):
            return a
        return None
    
    def index_by_name(self, n: str) -> int:
        """
        Возвращает индекс по имени атрибута
        
        Args:
            n: Наименование атрибута.
        """
        attr = self.by_name(n)
        if attr is None:
            return -1
        return self.index(attr)

    def __len__(self):
        return super().__len__()
