from axipy.cpp_core_dp import Converter, Feature as _ShadowFeature

from typing import Iterator, Dict, Callable, Union
from .opener import opener_instance as _opener
from ..data_object import Table, DataObject
from ..feature import Feature
from ..schema import Schema

__all__ = [
    "Source",
    "Destination",
    "ExportParameters",
]


class Source(dict):
    """
    Источник данных.

    Используется для открытия данных или для указания источника при конвертации.

    Пример открытия::

        table = source.open()

    Пример конвертации::

        destination.export_from(source)

    Note:
        Не все провайдеры поддерживают открытие и конвертацию. См. описание
        конкретного провайдера данных.
    """

    def __init__(self, *args):
        if type(self) is Source:
            raise NotImplementedError
        _merged = dict()
        for d in args:
            _merged.update(d)
        super().__init__(_merged)

    @classmethod
    def _table_file(cls, filepath: str) -> Dict:
        return {'src': filepath}

    @classmethod
    def _table_db(cls, db_name: str, host: str, user: str, password: str,
                  port: str) -> Dict:
        return {
            'src': host,
            'db': db_name,
            'user': user,
            'password': password,
            'port': port
        }

    @classmethod
    def _provider(cls, provider_id: str) -> Dict:
        return {'provider': provider_id}

    @classmethod
    def _alias(cls, alias: str) -> Dict:
        if alias:
            return {'alias': alias}
        return {}

    @classmethod
    def _prj(cls, prj: str) -> Dict:
        if prj:
            return {'prj': prj}
        return {}

    def open(self) -> DataObject:
        """Открывает объект данных."""
        return _opener.open(self)


class Destination(dict):
    """
    Назначение объекта данных.

    Используется для создания данных или для указания назначения при конвертации.

    Пример создания::

        table = destination.create_open()

    Пример конвертации::

        destination.export_from(source)

    Note:
        Не все провайдеры поддерживают создание и конвертацию. См. описание
        конкретного провайдера данных.
    """

    def __init__(self, schema: Schema, *args):
        if type(self) is Destination:
            raise NotImplementedError
        _merged = {'schema': schema.to_dict()}
        for d in args:
            _merged.update(d)
        super().__init__(_merged)

    def create_open(self) -> DataObject:
        """Создает и открывает объект данных."""
        return _opener.create(self)

    @staticmethod
    def _feature_callback_generator(func_callback: Callable, table: Iterator[Feature]) -> _ShadowFeature:
        for idx, f in enumerate(table):
            r = func_callback(f, idx)
            if r is None:
                yield f._shadow
            else:
                return None

    def export(self, features: Iterator[Feature], func_callback: Callable[[Feature, int], Union[None, bool]] = None):
        """
        Создает объект данных и экспортирует в него записи.

        Args:
            features: Записи.
            func_callback: Функция, которая будет вызываться после экспорта каждой записи.
                В определении должны быть параметры следующих типов:

                    * feature :class:`Feature` - текущая запись
                    * row :class:`int` - порядковый номер

                Возможно прерывание процесса экспорта, для этого нужно вернуть False в func_callback.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_export.py
            :pyobject: test_run_example_export_features
            :lines: 2-7, 11-
            :dedent: 4
            :caption: Пример экспорта данных
        """
        self.export_from_table(features, copy_schema=False, func_callback=func_callback)

    def export_from_table(self, table: Table, copy_schema: bool = False,
                          func_callback: Callable[[Feature, int], Union[None, bool]] = None):
        """
        Создает объект данных и экспортирует в него записи из таблицы.

        Args:
            table: Таблица.
            copy_schema: Копировать схему источника без изменений.
            func_callback: Функция, которая будет вызываться после экспорта каждой записи.
                В определении должны быть параметры следующих типов:

                    * feature :class:`Feature` - текущая запись
                    * row :class:`int` - порядковый номер

                Возможно прерывание процесса экспорта, для этого нужно вернуть False в func_callback.

        .. literalinclude:: /../../tests/doc_examples/da/test_example_export.py
            :pyobject: example_export_tab
            :lines: 2-
            :dedent: 4
            :caption: Пример экспорта таблицы с прогрессом

        .. literalinclude:: /../../tests/doc_examples/da/test_example_export.py
            :pyobject: example_export_csv
            :lines: 2-
            :dedent: 4
            :caption: Пример экспорта таблицы в формат CSV
        """

        dest: dict = self.copy()
        if copy_schema:
            dest.update({'schema': table.schema.to_dict()})

        if func_callback is None:
            Converter.saveAs(
                (f._shadow for f in table),
                dest,
            )
        else:
            Converter.saveAs(
                self._feature_callback_generator(func_callback, table),
                dest,
            )

    def export_from(self, source: Source, copy_schema: bool = False):
        """
        Создает объект данных и экспортирует в него записи из источника данных.

        Args:
            source: Источник данных.
            copy_schema: Копировать схему источника без изменений.

        """
        dest: dict = self.copy()
        if copy_schema:
            del dest['schema']
        Converter.convert(source, dest)


class ExportParameters:
    """
    Дополнительные параметры экспорта в таблицу базы данных.
    """
    createIndex: bool = True
    """
    Создавать пространственный индекс
    """
    dropTable: bool = False
    """
    Предварительно удалять существующую таблицу, если она присутствует в БД
    """
    geometryColumnName: str = None
    """
    Наименование геометрической колонки
    """
    renditonColumnName: str = None
    """
    Наименование колонки с оформлением
    """
    srid: int = 0
    """
    Значение SRID
    """
    logFile: str = None
    """
    Наименование файла, куда будут прописываться успешно выполненные команды
    """
    errorFile: str = None
    """
    Наименование файла, куда будут прописываться команды по вставке записей, не принятых сервером
    """
    mapCatalog: bool = True
    """
    Регистрация импортируемой таблицы в mapinfo.mapinfo_mapcatalog
    """
    geometryAsText: bool = False
    """
    Геометрию экспортировать как текстовые объекты
    """
    fixGeometry: bool = False
    """
    Пробовать исправлять инвалидную геометрию
    """

    def to_dict(self):
        res = {
            'createIndex': self.createIndex,
            'dropTable': self.dropTable,
            'mapCatalog': self.mapCatalog,
            'geometryAsText': self.geometryAsText,
            'fixGeometry': self.fixGeometry
        }
        if self.geometryColumnName:
            res['geometryColumnName'] = self.geometryColumnName
        if self.renditonColumnName:
            res['renditonColumnName'] = self.renditonColumnName
        if self.srid > 0:
            res['srid'] = self.srid
        if self.logFile:
            res['logFile'] = self.logFile
        if self.errorFile:
            res['errorFile'] = self.errorFile
        return res
