from typing import List

from axipy.da.data_object import Table
from .data_provider import DataProvider
from .source import Source, Destination
from ..schema import Schema

__all__ = [
    "ShapeSource",
    "ShapeDestination",
    "ShapeDataProvider",
]


class ShapeSource(Source):
    pass


class ShapeDestination(Destination):
    pass


class ShapeDataProvider(DataProvider):
    """Векторный провайдер SHP.
    
    Note:
        Ссылку на провайдер можно получить через глобальную переменную :attr:`axipy.provider_manager.shp`.
    """
    _identifier = 'OgrDataProvider'

    def get_source(self, filepath: str, encoding: str = 'utf8', prj: str = None, alias: str = None) -> Source:
        """Создает источник данных.

        Args:
            filepath: Путь к файлу.
            encoding: Кодировка.
            prj: Строка Системы Координат.
        """
        return ShapeSource(
            Source._provider(self.id),
            Source._table_file(filepath),
            Source._alias(alias),
            Source._prj(prj),
            {'charset': encoding}
        )

    def open(self, filepath: str, encoding: str = 'utf8', prj: str = None, alias: str = None) -> Table:
        """Открывает объект данных.

        Пример::

            shp = provider_manager.shp.open('world.shp', prj='1, 104')

        Args:
            filepath: Путь к файлу.
            encoding: Кодировка.
            prj: Строка Системы Координат.
            alias: Псевдоним для открываемой таблицы.
        """
        return self.get_source(filepath, encoding, prj, alias).open()

    def get_destination(self, filepath: str, schema: Schema,
                        encoding: str = 'utf8') -> Destination:
        """Создает назначение объекта данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
            encoding: Кодировка.
        """
        return ShapeDestination(schema,
                                Source._provider(self.id),
                                Source._table_file(filepath),
                                {'charset': encoding}
                                )

    def create_open(self, filepath: str, schema: Schema,
                    encoding: str = 'utf8') -> Table:
        """Создает и открывает объект данных.

        Args:
            filepath: Путь к файлу.
            schema: Схема таблицы.
            encoding: Кодировка.
        """
        return self.get_destination(filepath, schema, encoding).create_open()

    def open_temporary(self, schema: Schema) -> Table:
        """Создает и открывает временную таблицу.

        Args:
            schema: Схема таблицы.
        """
        return ShapeDestination(schema,
                                Source._provider(self.id),
                                Source._table_file(''),
                                ).create_open()

    def file_extensions(self) -> List[str]:
        return list(filter(lambda v: v in ['shp', 'shz'], super().file_extensions()))
