import traceback
from PySide2.QtCore import Qt, QFileInfo, Slot, Signal, QEvent
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import (QFileDialog, QDialog, QDialogButtonBox, QMessageBox, QLineEdit, QProgressBar,
                               QToolButton, QGroupBox)
from axipy import (tr, CurrentSettings, task_manager, provider_manager,
                   ChooseCoordSystemDialog, CoordSystem, Notifications, AxipyAnyCallableTask, DefaultSettings,
                   data_manager, Table, Plugin, AxipyProgressHandler)
from pathlib import Path
from typing import Optional, List
from .bounding_rect_dialog import BoundingRectDialog
from .ui.export_to_file_dialog import Ui_Dialog


class ExportToFileDialog(QDialog):
    _add_progress_signal = Signal()
    _init_progress_range_signal = Signal()
    _notify_signal = Signal(str, int)

    class _SkippedNotification:

        def __init__(self, first_file_name: Path) -> None:
            self.first_file_name: Path = first_file_name
            self.n: int = 1

    def __init__(self, plugin: Plugin, parent=None) -> None:
        super().__init__(parent)

        self._plugin = plugin
        self._title = self._plugin.tr("Экспорт таблиц")

        # ui file
        self._ui = Ui_Dialog()
        self._ui.setupUi(self)
        # ui elements
        self._ui_le_out_folder: QLineEdit = self._ui.le_out_folder
        self._ui_tbSelectFiles: QToolButton = self._ui.tbSelectFiles
        self._ui_pbar: QProgressBar = self._ui.pbar
        self._ui_groupBoxProjection: QGroupBox = self._ui.groupBoxProjection

        self._coord_system = CoordSystem.from_string(CoordSystem.current().to_string())
        self._selected_filter: Optional[str] = None
        self._overwrite = False
        self._pbar_range_init = False
        self._skipped_notification: Optional[ExportToFileDialog._SkippedNotification] = None

        self._out_folder: Optional[Path] = None

        self._task: Optional[AxipyAnyCallableTask] = None

        self._add_progress_signal.connect(self._add_progress)
        self._init_progress_range_signal.connect(self._init_progress_range)
        self._notify_signal.connect(self._notify)

        self._ok_button: Optional[QDialogButtonBox.StandardButton.Ok] = None
        self._cancel_button: Optional[QDialogButtonBox.StandardButton.Cancel] = None

        self._init_ui()

    def _init_ui(self) -> None:
        self._ui_pbar.hide()

        self.setWindowTitle(self._title)
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"

        self._ui.tbSelectFiles.setIcon(QIcon.fromTheme("open"))

        def on_open_files_clicked() -> None:
            try:
                last_open_path_ = CurrentSettings.LastOpenPath
                if not Path(last_open_path_).exists():
                    last_open_path_ = DefaultSettings.LastOpenPath

                file_names, selected_filter = QFileDialog.getOpenFileNames(
                    parent=self,
                    caption="Выбор файлов для экспорта",
                    dir=str(last_open_path_),
                    filter="Файлы TAB (*.tab);;Все файлы (*.*)",
                    selectedFilter=self._selected_filter,  # selectedFilter
                )
                if len(file_names) <= 0:
                    return None

                self._selected_filter = selected_filter
                CurrentSettings.LastOpenPath = QFileInfo(file_names[0]).absolutePath()
                path_list = self._get_path_list()
                for fn in file_names:
                    if fn not in path_list:
                        self._ui.listSourceTables.addItem(fn)
                self._check_enabled_export()
            except Exception as e:
                Notifications.push(self._title, f"Ошибка открытия файлов: {e}", Notifications.Critical)

        self._ui.tbSelectFiles.clicked.connect(on_open_files_clicked)
        self._ui.tbDeleteSelected.setIcon(QIcon.fromTheme("delete"))
        self._ui.tbDeleteSelected.clicked.connect(self._on_delete_selected_clicked)

        def on_selection_changed() -> None:
            self._ui.tbDeleteSelected.setEnabled(bool(len(self._ui.listSourceTables.selectedItems())))

        self._ui.listSourceTables.itemSelectionChanged.connect(on_selection_changed)

        def get_existing_last_save_path() -> Path:
            last_save_path_ = CurrentSettings.LastSavePath
            if not Path(last_save_path_).exists():
                last_save_path_ = DefaultSettings.LastSavePath
            return last_save_path_

        last_save_path = get_existing_last_save_path()
        self._out_folder = last_save_path
        self._ui_le_out_folder.setText(str(last_save_path))

        def choose_out_folder() -> None:
            last_save_path_ = get_existing_last_save_path()
            out_folder_ = QFileDialog.getExistingDirectory(
                self,
                caption=tr("Выбор выходной папки"),
                dir=str(last_save_path_),
                options=QFileDialog.ShowDirsOnly,
            )

            if out_folder_:
                self._ui.le_out_folder.setText(out_folder_)
                self._out_folder = Path(out_folder_)
                CurrentSettings.LastSavePath = out_folder_

                self._check_enabled_export()

        self._ui.pb_choose_out_folder.clicked.connect(choose_out_folder)

        self._ui.pbChoiseProjection.setText(self._coord_system.name)

        self._ok_button = self._ui.buttonBox.button(QDialogButtonBox.Ok)
        self._ok_button.setText(self._plugin.tr("Экспорт"))
        self._ok_button.setEnabled(False)
        self._cancel_button = self._ui.buttonBox.button(QDialogButtonBox.Cancel)
        self._cancel_button.setText(self._plugin.tr("Закрыть"))

        def on_projection_change_clicked() -> None:
            # Диалог не корректно работает с текущей КС или первой инициализацией без None
            if self._coord_system == CoordSystem.current():
                cs = None
            else:
                cs = self._coord_system
            dialog = ChooseCoordSystemDialog(cs)

            @Slot(int)
            def finished(return_code: int):
                if return_code == QDialog.Accepted:
                    result_cs = dialog.chosenCoordSystem()
                    self._coord_system = result_cs
                    self._ui.pbChoiseProjection.setText(result_cs.name)

                    if result_cs.non_earth:
                        on_projection_rect_change_clicked()

            dialog.finished.connect(finished)
            dialog.open()

        self._ui.pbChoiseProjection.clicked.connect(on_projection_change_clicked)

        def on_projection_rect_change_clicked() -> None:
            dialog = BoundingRectDialog(self._coord_system, self)
            dialog.open()

        self._ui.pbBoundingRect.clicked.connect(on_projection_rect_change_clicked)

    def keyPressEvent(self, event: QEvent) -> None:
        if event.key() == Qt.Key_Delete:
            self._on_delete_selected_clicked()

        super().keyPressEvent(event)

    def accept(self) -> None:
        self._export_process()

    def reject(self) -> None:

        if (
                self._task is not None and
                self._task.progress_handler().is_running() and
                not self._task.progress_handler().is_canceled()
        ):
            self._task.progress_handler().cancel()
            return None

        if self._task is None or not self._task.progress_handler().is_running():
            super().reject()

    def _check_enabled_export(self) -> None:
        if (
                self._ui.listSourceTables.count() > 0 and
                self._out_folder is not None
        ):
            enabled = True
        else:
            enabled = False

        self._ok_button.setEnabled(enabled)

    def _get_path_list(self) -> List[str]:
        list_widget = self._ui.listSourceTables
        return [list_widget.item(i).text() for i in range(list_widget.count())]

    def _on_delete_selected_clicked(self) -> None:
        selected_items = self._ui.listSourceTables.selectedItems()
        for item in selected_items:
            self._ui.listSourceTables.takeItem(self._ui.listSourceTables.row(item))
        self._check_enabled_export()

    def _toggle_gui_availability(self, state: bool) -> None:
        objects = (
            self._ui.tbSelectFiles,
            self._ui.tbDeleteSelected,
            self._ui.listSourceTables,
            self._ui.le_out_folder,
            self._ui.pb_choose_out_folder,
            self._ui_groupBoxProjection,
        )
        for obj in objects:
            obj.setEnabled(state)

    def _export_process(self) -> None:
        if not self._out_folder.exists():
            mbox = QMessageBox(
                QMessageBox.Critical,  # icon
                tr("Ошибка"),  # title
                tr("Выбранной выходной папки не существует, пожалуйста, выберите другую папку."),  # text
                parent=self,
            )
            mbox.open()
            return None

        self._toggle_gui_availability(False)
        self._ok_button.setEnabled(False)
        self._cancel_button.setText(tr("Отмена"))

        self._skipped_notification = None

        self._pbar_range_init = False
        self._ui_pbar.setRange(0, 0)
        self._ui_pbar.show()

        def export_in_thread(ph: AxipyProgressHandler, path_list: List[str]):

            for path in path_list:
                table = None
                is_hidden = False
                try:
                    out_file_name = self._out_folder / Path(path).name
                    if not self._overwrite and out_file_name.exists():
                        if self._skipped_notification is None:
                            self._skipped_notification = ExportToFileDialog._SkippedNotification(out_file_name)
                        else:
                            self._skipped_notification.n += 1
                    else:
                        # Поиск уже открытой таблицы
                        all_tables = filter(lambda do_: isinstance(do_, Table), data_manager.all_objects)
                        for table_ in all_tables:
                            prop = table_.properties
                            if "tabFile" in prop and Path(prop["tabFile"]) == Path(path):
                                table = table_
                                break
                        # Открытие временной таблицы
                        if table is None:
                            table = provider_manager.open_hidden({'src': path, 'provider': 'TabDataProvider'})
                            is_hidden = True
                        if table is None:
                            raise Exception(f"Can't open table {path}")
                        schema = table.schema
                        schema.coordsystem = self._coord_system
                        destination = provider_manager.tab.get_destination(str(out_file_name), schema)
                        destination.export_from_table(table)
                except Exception as e:
                    self._notify_signal.emit(f"Ошибка в процессе конвертации: {e}", int(Notifications.Critical))
                    return None
                finally:
                    try:
                        if table is not None and is_hidden:
                            task_manager.run_in_gui(table.close)
                    except (Exception,):
                        traceback.print_exc()
                        return None

                self._init_progress_range_signal.emit()
                self._add_progress_signal.emit()

                if ph.is_canceled():
                    self._notify_signal.emit("Конвертация отменена.", int(Notifications.Warning))
                    break

        if self._task is None:
            self._task = AxipyAnyCallableTask(export_in_thread, self._get_path_list())
            self._task.progress_handler().finished.connect(self._on_finished)
            task_manager.start_task(self._task)

    @Slot()
    def _init_progress_range(self) -> None:
        if not self._pbar_range_init:
            path_list = self._get_path_list()
            self._ui_pbar.setMaximum(len(path_list))
            self._ui_pbar.setValue(1)
            self._pbar_range_init = True

    @Slot()
    def _add_progress(self) -> None:
        self._ui_pbar.setValue(self._ui_pbar.value() + 1)

    @Slot(str, int)
    def _notify(self, msg: str, msg_type: int) -> None:
        Notifications.push(self._title, msg, msg_type)

    @Slot()
    def _on_finished(self) -> None:
        self._task.progress_handler().finished.disconnect(self._on_finished)
        self._task = None

        self._ui_pbar.hide()
        self._cancel_button.setText(tr("Закрыть"))

        if self._skipped_notification is not None:
            Notifications.push(
                self._title,
                f"Файл '{self._skipped_notification.first_file_name}' уже существует в выходной папке. Файл пропущен. "
                f"(Всего пропущено {self._skipped_notification.n}).",
                Notifications.Warning,
            )
        Notifications.push(
            self._title,
            self._plugin.tr("Конвертация завершена."),
        )

        self._toggle_gui_availability(True)
        self._check_enabled_export()
