import os
import sys
from pathlib import Path
from types import TracebackType
from typing import Type

__all__ = []


def _install_excepthook() -> None:
    # install custom exception hook

    def axioma_excepthook(extype: Type[BaseException], value: BaseException, traceback: TracebackType) -> None:
        print("Unhandled Python Exception:", file=sys.stderr)
        sys.__excepthook__(extype, value, traceback)

    sys.excepthook = axioma_excepthook


# function uses paths, relative to file location
def _setup_environment() -> None:
    if "AXIOMA_HOME" in os.environ:
        return None

    dev = False
    home = None

    relative_position = 2

    if os.name == "nt":
        installpath = Path(__file__).parents[relative_position + 2]
    elif sys.platform.startswith("darwin"):
        installpath = Path(__file__).parents[relative_position + 6] / "Frameworks"
    else:
        installpath = Path(__file__).parents[relative_position + 3] / "bin"

    devpath = Path(__file__).parents[relative_position]

    def contains_core_binary(path: Path) -> bool:
        """
        Possible values:
            core.dll libcore.so.* libcore.*.dylib
        """
        if not path.is_dir():
            raise Exception("Not a dir")

        for arg in path.iterdir():
            name = arg.name
            if name.startswith("core.") or name.startswith("libcore."):
                return True
        return False

    if contains_core_binary(installpath):
        home = installpath
        # dev = False
    elif contains_core_binary(devpath):
        home = devpath
        dev = True

    if dev and sys.platform == "win32":
        environ = os.environ
        if "PATH" in environ:
            for elem in environ["PATH"].split(";"):
                os.add_dll_directory(elem)

    if home is None:
        raise RuntimeError("Can't find Axioma installation")

    def insert_first_environ_path(key: str, environ_path: str) -> None:
        pathsep = os.pathsep
        env_dict = os.environ

        value = env_dict.get(key, None)
        if value is None:
            env_dict[key] = environ_path
        elif not isinstance(value, str):
            raise TypeError("environ value not a string")
        elif value.startswith(pathsep):
            env_dict[key] = environ_path + env_dict[key]
        else:
            env_dict[key] = environ_path + pathsep + env_dict[key]

    if dev:
        insert_first_environ_path("PATH", str(home.parents[2] / "third-party" / "out" / "bin"))

    insert_first_environ_path("PATH", str(home))
    os.environ["AXIOMA_HOME"] = str(home)

    try:
        if sys.version_info >= (3, 8) and sys.platform == 'win32':
            os.add_dll_directory(str(home))
    except (Exception,):
        ...

    if not dev:
        if sys.platform.startswith("darwin"):
            os.environ["QT_PLUGIN_PATH"] = str(home.parents[1] / "PlugIns")
        else:
            os.environ["QT_PLUGIN_PATH"] = str(home / "qtplugins")


def _setup_linux() -> None:
    path = Path(__file__)
    if sys.platform == "linux" and "AXIOMA_HOME" in os.environ and "site-packages" in path.parts:
        axioma_base = str(Path(os.environ["AXIOMA_HOME"]).parent)

        os.environ["AXIOMA_PLUGINS"] = axioma_base + "/bin/plugins"
        os.environ["AXIOMA_DATA"] = axioma_base + "/share/axioma"
        os.environ["GDAL_DATA"] = axioma_base + "/share/gdal"
        os.environ["PROJ_LIB"] = axioma_base + "/share/proj"
        os.environ["GDAL_DRIVER_PATH"] = axioma_base + "/bin/gdalplugins"
        os.environ["QT_PLUGIN_PATH"] = axioma_base + "/bin/qtplugins"
        axioma_python_home = axioma_base + "/python"
        py_lib = axioma_python_home + "/lib/python3.8"
        os.environ["AXIOMA_PYTHON_BINDINGS"] = py_lib + "/site-packages"
        os.environ["AXIOMA_PYTHON_BASEPATH"] = \
            f"{py_lib}:{py_lib}/plat-linux:{py_lib}/lib-dynload:{py_lib}/site-packages"
        os.environ["AXIOMA_PYTHON_PLUGINS"] = axioma_base + "/bin/python_plugins"
        os.environ["AXIOMA_PYTHON_INTERPRETER"] = axioma_python_home + "/bin/python3"


def _setup_all() -> None:
    _install_excepthook()
    _setup_environment()
    _setup_linux()
