from PySide2.QtCore import QPointF, QPoint, QLocale
from axipy.cpp_common import CommonUtils

from axipy.utl.float_formatter import FloatCoord
from typing import Optional, Union, List

__all__ = ["Pnt"]


class _Pnt:
    @staticmethod
    def _fixPnt(v: tuple) -> 'Pnt':
        """
        Преобразование из tuple в Pnt, если это возможно.
        """
        if isinstance(v, tuple) and len(v) == 2:
            return Pnt(v[0], v[1])
        return v

    @classmethod
    def _point_value_to_qt(cls, v: Union[QPointF, tuple, 'Pnt', List]) -> Union[QPointF, List[QPointF]]:
        if isinstance(v, tuple) and len(v) == 2:
            return QPointF(v[0], v[1])
        elif isinstance(v, Pnt):
            return v.to_qt()
        elif isinstance(v, list):
            return [cls._point_value_to_qt(p) for p in v]
        elif isinstance(v, QPointF):
            return v
        raise ValueError(f'Bad value {v}')

    @classmethod
    def _point_value_from_qt(cls, v: Union[QPointF, tuple, 'Pnt', List]) -> Union['Pnt', List['Pnt']]:
        if isinstance(v, tuple) and len(v) == 2:
            return Pnt(v[0], v[1])
        elif isinstance(v, QPointF):
            return Pnt.from_qt(v)
        elif isinstance(v, list):
            return [cls._point_value_from_qt(p) for p in v]
        elif isinstance(v, Pnt):
            return v
        raise ValueError(f'Bad value {v}')


class Pnt(_Pnt):
    """
    Точка без геопривязки. Может быть использована в качестве параметра геометрии (точки полигона)
    или при получении параметров,
    где результат представлен в виде точки (центр карты или элемента отчета).

    .. literalinclude:: /../../tests/doc_examples/utl/test_example_pnt.py
        :caption: Создание точки.
        :pyobject: test_run_example_pnt
        :start-after: # start init
        :end-before: # finish init
        :dedent: 4

    """

    def __init__(self, x: float, y: float) -> None:
        """
        Конструктор класса.

        Args:
            x: X координата.
            y: Y координата.

        """
        self._point = QPointF(x, y)

    @classmethod
    def from_qt(cls, p: Union[QPointF, QPoint]) -> Optional['Pnt']:
        """Преобразует из формата Qt. Если класс не соответствует, возвращает None.
        
        Args:
            p: Преобразуемая точка.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_pnt.py
            :caption:  Пример.
            :pyobject: test_run_example_pnt
            :start-after: # start from_qt
            :end-before: # finish from_qt
            :dedent: 4

        """
        if not isinstance(p, (QPointF, QPoint)):
            return None
        obj = cls.__new__(cls)
        obj._point = p if isinstance(p, QPointF) else QPointF(p)
        return obj

    def to_qt(self) -> QPointF:
        """
        Преобразование в формат Qt.

        .. literalinclude:: /../../tests/doc_examples/utl/test_example_pnt.py
            :caption:  Пример.
            :pyobject: test_run_example_pnt
            :start-after: # start to_qt
            :end-before: # finish to_qt
            :dedent: 4

        """
        return self._point

    @property
    def x(self) -> float:
        """
        Устанавливает или возвращает координату X.
        """
        return self._point.x()

    @x.setter
    def x(self, v: float):
        self._point.setX(v)

    @property
    def y(self) -> float:
        """
        Устанавливает или возвращает координату Y.
        """
        return self._point.y()

    @y.setter
    def y(self, v: float):
        self._point.setY(v)

    @classmethod
    def eq_approx(cls, point1: 'Pnt', point2: 'Pnt', precision: float = 1e-12) -> bool:
        """
        Сравнивает две точки с заданной точностью.

        Args:
            point1: Первая точка сравнения
            point2: Вторая точка сравнения
            precision: Точность сравнения

        Return:
            True если точки равны
        """
        pnt1 = Pnt._point_value_to_qt(point1)
        pnt2 = Pnt._point_value_to_qt(point2)
        return CommonUtils.approx_compare_points(pnt1, pnt2, precision)

    def __eq__(self, other) -> bool:
        if isinstance(other, Pnt):
            return CommonUtils.approx_compare_points(self._point, other._point)
        if isinstance(other, tuple) and len(other) == 2:
            p2 = Pnt._point_value_to_qt(other)
            return CommonUtils.approx_compare_points(self._point, p2)
        return False

    def __repr__(self) -> str:
        x = FloatCoord(self.x).as_string(precision=15, locale=QLocale.English)
        y = FloatCoord(self.y).as_string(precision=15, locale=QLocale.English)
        return f"{__class__.__name__}({x}, {y})"

    def __str__(self) -> str:
        return "({} {})".format(self.x, self.y)
