"""
Настройки Аксиомы.
"""
from PySide2.QtGui import QColor
from axipy.cpp_core_core import DefaultSettingsList, ShadowSettings

from abc import abstractmethod, ABC
from axipy._internal._metaclass import _MappingMetaExtended, _MappingMetaDocumentation
from axipy._internal._shadow_instance_factory import _ShadowManager
from enum import Enum
from pathlib import Path
from typing import List, Tuple, Any, Iterator, MutableMapping, Mapping, TypeVar, Dict, Union

__all__ = [
    "DefaultSettings",
    "CurrentSettings",
    "AxiomaLanguage",
]


class AxiomaLanguage(str, Enum):
    """
    Язык Аксиомы.
    """
    ru = "ru"
    """Русский язык"""
    en = "en"
    """Английский язык"""


class _ShadowSettingsManager:
    Value = TypeVar("Value")
    ShadowValue = TypeVar("ShadowValue")

    _path_keys = (
        "LastSavePath",
        "LastOpenPath",
        "LastPathWorkspace",
        "DefaultPathCache",
        "PenCatalog",
        "BrushCatalog",
        "SymbolCatalog",
    )

    class _MappingReprStr(ABC):

        @abstractmethod
        def items(self) -> List[Tuple[str, Path]]:
            ...

        def __repr__(self) -> str:
            return repr(self.items())

        def __str__(self) -> str:
            return str(self.items())

    class _CustomMutableMapping(MutableMapping, _MappingReprStr):

        def __init__(self, dict_arg: dict):
            self._dict_arg = dict_arg

        @staticmethod
        def _reset_to_default() -> None:
            dict_ = dict(DefaultSettings.UserDataPaths.items())
            list_of_lists = [[k, str(v)] for k, v in dict_.items()]
            _ShadowSettingsManager.set_value("UserDataPaths", list_of_lists)

        @staticmethod
        def _ensure_existing_dir(v: Union[Path, str]) -> None:

            def check_is_dir(v_arg: Path):
                if not v_arg.is_dir():
                    raise TypeError("Needs path to existing dir")

            if isinstance(v, Path):
                check_is_dir(v)
            elif isinstance(v, str):
                v = Path(v)
                check_is_dir(v)
            else:
                raise TypeError("Needs Path or str value")

        def _update_setting(self) -> None:
            list_of_lists = [[k, str(v)] for k, v in self._dict_arg.items()]
            _ShadowSettingsManager.set_value("UserDataPaths", list_of_lists)

        def __setitem__(self, __k, __v) -> None:
            self._ensure_existing_dir(__v)
            self._dict_arg[__k] = __v
            self._update_setting()

        def __delitem__(self, __k) -> None:
            del self._dict_arg[__k]
            self._update_setting()

        def __getitem__(self, __k) -> Path:
            return self._dict_arg[__k]

        def __len__(self) -> int:
            return len(self._dict_arg)

        def __iter__(self) -> Iterator:
            return iter(self._dict_arg)

        def keys(self) -> List[str]:
            return list(super().keys())

        def values(self) -> List[Path]:
            return list(super().values())

        def items(self) -> List[Tuple[str, Path]]:
            return list(super().items())

    class _CustomMapping(Mapping, _MappingReprStr):

        def __init__(self, dict_arg: dict):
            self._dict_arg = dict_arg

        def __getitem__(self, __k) -> Path:
            return self._dict_arg[__k]

        def __len__(self) -> int:
            return len(self._dict_arg)

        def __iter__(self) -> Iterator:
            return iter(self._dict_arg)

        def keys(self) -> List[str]:
            return list(super().keys())

        def values(self) -> List[Path]:
            return list(super().values())

        def items(self) -> List[Tuple[str, Path]]:
            return list(super().items())

    @staticmethod
    def _shadow_value_to_value(key, shadow_value: ShadowValue, default) -> Value:
        value = shadow_value

        if key == "UserDataPaths" and isinstance(value, list):
            value = {k: Path(v) for k, v in value}
            if default:
                value = _ShadowSettingsManager._CustomMapping(value)
            else:
                value = _ShadowSettingsManager._CustomMutableMapping(value)

        elif key in _ShadowSettingsManager._path_keys:
            value = Path(value)

        return value

    @staticmethod
    def _value_to_shadow_value(_key, value: Value) -> ShadowValue:
        shadow_value = value
        return shadow_value

    @staticmethod
    def _to_shadow_key(key: str) -> DefaultSettingsList.Key:
        return getattr(DefaultSettingsList, key)

    @staticmethod
    def get_value(key: str, /, is_default: bool) -> Value:
        if is_default:
            func = ShadowSettings.defaultValue
        else:
            func = ShadowSettings.value
        shadow_value = func(_ShadowSettingsManager._to_shadow_key(key))
        value = _ShadowSettingsManager._shadow_value_to_value(key, shadow_value, is_default)
        return value

    @staticmethod
    def set_value(key: str, value: Value) -> None:

        # Автоматическая проверка простых типов
        passed_type = type(value)
        expected_type = type(_ShadowSettingsManager.get_value(key, is_default=True))
        if expected_type in (bool, int, float, str):
            if passed_type != expected_type:
                raise TypeError(f'Key type mismatch; expected {expected_type} but got {passed_type} instead.')

        if key in _ShadowSettingsManager._path_keys:
            value = str(value)

        shadow_value = _ShadowSettingsManager._value_to_shadow_value(key, value)
        ShadowSettings.setValue(_ShadowSettingsManager._to_shadow_key(key), shadow_value)


class _Settings(_MappingMetaDocumentation):

    class _KeysDesc:

        def __init__(self) -> None:
            self._keys = None

        def __get__(self, instance, owner) -> List:
            if self._keys is None:
                self._keys = [k for k, v in _Settings.__dict__.items() if isinstance(v, _Settings._Desc)]
            return self._keys

    _desc_keys = _KeysDesc()

    class _Desc:
        """
        Дескриптор управляет доступом к атрибутам, для классов DefaultSettings и CurrentSettings.
        """

        def __set_name__(self, owner, name: str) -> None:
            self._name = name

        def __get__(self, obj, objtype=None) -> Any:
            """
            Can't compare objtype, because of infinite recursion.
            """
            if type(objtype) == _DefaultSettingsMappingMetaExtended:
                value = _ShadowSettingsManager.get_value(self._name, is_default=True)
            elif type(objtype) == _CurrentSettingsMutableMappingMeta:
                value = _ShadowSettingsManager.get_value(self._name, is_default=False)
            else:
                raise RuntimeError
            return value

    class _LangDesc:

        def __get__(self, obj, objtype=None) -> AxiomaLanguage:
            if type(objtype) == _DefaultSettingsMappingMetaExtended:
                result_str = "ru"
            elif type(objtype) == _CurrentSettingsMutableMappingMeta:
                result_str = _ShadowManager.core.translationLanguage()
            else:
                raise RuntimeError

            result = AxiomaLanguage[result_str]
            return result

    Language: 'AxiomaLanguage' = _LangDesc()
    """Язык Аксиомы"""
    SilentCloseWidget: bool = _Desc()
    """Подтверждать закрытие несохраненных данных"""
    SnapSensitiveRadius: int = _Desc()
    """Привязка узлов - размер"""
    EditNodeColor: QColor = _Desc()
    """Узлы при редактировании - цвет"""
    EditNodeSize: int = _Desc()
    """Узлы при редактировании - размер"""
    NearlyGeometriesTopology: bool = _Desc()
    """Перемещать узлы соседних объектов при редактировании"""
    NodesUpdateMode: bool = _Desc()
    """Объединять историю изменения узлов в режиме Форма"""
    ShowDrawingToolTip: bool = _Desc()
    """Показывать данные при рисовании"""
    CreateTabAfterOpen: bool = _Desc()
    """Создавать TAB при открытии"""
    RenameDataObjectFromTab: bool = _Desc()
    """Переименовывать открытый объект по имени TAB файла"""
    LastSavePath: Path = _Desc()
    """Последний пусть сохранения"""
    UseLastSelectedFilter: bool = _Desc()
    """Запоминать последний фильтр в диалоге открытия файлов"""
    SelectByInformationTool: bool = _Desc()
    """Инструмент «Информация» выбирает объект"""
    SaveAsToOriginalFileFolder: bool = _Desc()
    """Сохранять копию в каталог с исходным файлом"""
    LastNameFilter: str = _Desc()
    """Последний использованный фильтр файлов"""
    SensitiveMouse: int = _Desc()
    """Чувствительность мыши в пикселях"""
    ShowSplashScreen: bool = _Desc()
    """Отображать экран загрузки"""
    RulerColorLine: QColor = _Desc()
    """Линейка - цвет линии"""
    UseAntialiasing: bool = _Desc()
    """Использовать сглаживание при отрисовке"""
    ShowDegreeTypeNumeric: bool = _Desc()
    """Отображать градусы в формате Десятичное значение"""
    DrawCoordSysBounds: bool = _Desc()
    """Отображать границы мира"""
    PreserveScaleMap: bool = _Desc()
    """Сохранять масштаб при изменении размеров окна"""
    ShowMapScaleBar: bool = _Desc()
    """Показывать масштабную линейку"""
    ShowScrollOnMapView: bool = _Desc()
    """Показывать полосы прокрутки"""
    LoadLastWorkspace: bool = _Desc()
    """Загружать при старте последнее рабочее пространство"""
    ShowMeshLayout: bool = _Desc()
    """Отображать сетку привязки"""
    MeshSizeLayout: float = _Desc()
    """Размер ячейки"""
    SnapToMeshLayout: bool = _Desc()
    """Привязывать элементы отчета к сетке"""
    ShowMeshLegend: bool = _Desc()
    """Отображать сетку привязки"""
    MeshSizeLegend: float = _Desc()
    """Размер ячейки"""
    SnapToMeshLegend: bool = _Desc()
    """Привязывать к сетке"""
    LastOpenPath: Path = _Desc()
    """Последний каталог откуда открывались данные"""
    LastPathWorkspace: Path = _Desc()
    """Последний каталог к рабочему набору"""
    DefaultPathCache: Path = _Desc()
    """Каталог с кэшированными данными"""
    UserDataPaths: Dict[str, Path] = _Desc()
    """Список пользовательских каталогов с названиями"""
    EnableSmartTabs: bool = _Desc()
    """Умное переключение вкладок"""
    DistancePrecision: int = _Desc()
    """Точность по умолчанию для расстояний и площадей"""
    PenCatalog: Path = _Desc()
    """Каталог со стилями линий"""
    BrushCatalog: Path = _Desc()
    """Каталог со стилями заливок"""
    SymbolCatalog: Path = _Desc()
    """Каталог со стилями точек"""
    UseNativeFileDialog: bool = _Desc()
    """Пользоваться системными файловыми диалогами"""

    @classmethod
    def items(cls) -> List[Tuple[str, Any]]:
        """
        Возвращает список кортежей ключ-значение, где ключи - это атрибуты класса, a значения - это значения настроек.
        """
        return super().items()

    @classmethod
    def keys(cls) -> List[str]:
        """Возвращает список ключей, где ключи это атрибуты класса."""
        return super().keys()

    @classmethod
    def values(cls) -> List[Any]:
        """Возвращает список значений, где значения это значения настроек."""
        return super().values()

    @classmethod
    def get(cls, key: str, default_value: Any = None):
        """Возвращает значение по ключу."""
        return super().get(key, default_value)


class _DefaultSettingsMappingMetaExtended(_MappingMetaExtended):

    def __iter__(cls) -> Iterator:
        return iter(_Settings._desc_keys)

    def __len__(cls) -> int:
        return len(_Settings._desc_keys)

    def __getitem__(cls, key: str) -> Any:
        try:
            result = getattr(DefaultSettings, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result


class DefaultSettings(_Settings, metaclass=_DefaultSettingsMappingMetaExtended):
    """
    Настройки Аксиомы по умолчанию.
    Класс является статическим словарем, доступным только для чтения (:class:`collections.abc.Mapping`).
    Методы и атрибуты класса такие же как у класса :class:`axipy.CurrentSettings`, но доступны только для чтения.
    """


class _CurrentSettingsMutableMappingMeta(_DefaultSettingsMappingMetaExtended):

    def __getitem__(cls, key: str) -> Any:
        try:
            result = getattr(CurrentSettings, key)
        except AttributeError:
            raise KeyError(key)
        else:
            return result

    @staticmethod
    def _set(key: str, value: Any) -> None:
        if key in (
                "UserDataPaths",
                "Language",
        ):
            raise TypeError(f"Can't set attribute '{key}'.")
        _ShadowSettingsManager.set_value(key, value)

    def __setattr__(cls, key: str, value: Any) -> None:
        _CurrentSettingsMutableMappingMeta._set(key, value)

    def __setitem__(cls, key: str, value: Any) -> None:
        _CurrentSettingsMutableMappingMeta._set(key, value)


class CurrentSettings(_Settings, metaclass=_CurrentSettingsMutableMappingMeta):
    """
    Текущие настройки Аксиомы.
    Класс является статическим словарем (:class:`collections.abc.MutableMapping`).
    Поддерживает обращение по индексу.
    Для получения настроек по умолчанию, используется класс :class:`axipy.DefaultSettings`.

    .. literalinclude:: /../../tests/doc_examples/test_example_settings.py
        :caption: Пример работы с настройками.
        :pyobject: test_example_settings
        :start-after: # start
        :end-before: # finish
        :dedent: 4
    """

    @staticmethod
    def reset() -> None:
        """Сброс значений всех текущих настроек на значения по умолчанию."""
        for k, v in DefaultSettings.items():
            if k == "UserDataPaths":
                CurrentSettings.UserDataPaths._reset_to_default()
                continue
            elif k == "Language":
                continue
            CurrentSettings[k] = v
