from typing import Union

from PySide2.QtCore import Signal
from PySide2.QtGui import QColor
from axipy.cpp_render import ShadowLayer, ShadowVectorLayer, ShadowRasterLayer, ShadowThematicList, ShadowThematic, \
    ShadowCosmeticLayer

from axipy._internal._decorator import _deprecated_by
from axipy._internal._shadow_instance_factory import _ShadowManager
from axipy.cs import CoordSystem
from axipy.da import DataObject, Style
from axipy.utl import Rect
from .label import Label

__all__ = [
    "Layer",
    "ThematicLayer",
    "ListThematic",
    "VectorLayer",
    "CosmeticLayer",
    "RasterLayer",
]


class Layer:
    """Абстрактный базовый класс для слоя карты.

    Для создания нового экземпляра для векторного или растрового источника данных необходимо использовать метод :meth:`Layer.create`.
    Для тематических слоев - использовать соответствующие им конструкторы.
    """

    def __init__(self):
        super().__init__()
        raise NotImplementedError

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result._shadow = shadow
        return result

    @classmethod
    def _wrap(cls, _shadow: ShadowLayer):
        if isinstance(_shadow, ShadowCosmeticLayer):
            return CosmeticLayer._wrap_typed(_shadow)
        if isinstance(_shadow, ShadowVectorLayer):
            return VectorLayer._wrap_typed(_shadow)
        if isinstance(_shadow, ShadowRasterLayer):
            return RasterLayer._wrap_typed(_shadow)
        if isinstance(_shadow, ShadowThematic):
            return ShadowThematic._wrap_typed(_shadow)
        return None

    @classmethod
    def create(cls, dataObject: DataObject) -> 'Layer':
        """Создает слой на базе открытой таблицы или растра.

        Args:
            dataObject: Таблица или растр. В зависимости от переданного объекта будет создан
                :class:`VectorLayer`  или :class:`RasterLayer`.

        .. literalinclude:: /../../tests/doc_examples/render/test_example_layer.py
            :caption: Пример создания слоя на базе файла.
            :pyobject: test_run_example_layer
            :lines: 3-
            :dedent: 4
        """
        if dataObject is None or dataObject._shadow is None:
            raise ValueError("Object is not valid. Unable to create layer.")
        render = _ShadowManager.render
        from axipy.da import Raster, Table
        if not render:
            raise RuntimeError("axipy is not initialized")
        if isinstance(dataObject, Raster):
            shadow = ShadowRasterLayer(render, dataObject._shadow)
        elif dataObject.is_spatial:
            if isinstance(dataObject, Table):
                shadow = ShadowVectorLayer(render, dataObject._shadow)
            else:
                shadow = ShadowRasterLayer(render, dataObject._shadow)
        else:
            raise ValueError("Non spatial table")
        if shadow is None:
            msg = 'Cannot create layer.'
            if dataObject._shadow is not None and not dataObject._shadow.isSpatial():
                msg = msg + ' Table is not spatial.'
            raise ValueError(msg)
        res = cls._wrap(shadow)
        if res is None:
            raise TypeError("Invalid layer type")
        return res

    @property
    def data_changed(self) -> Signal:
        """
        Сигнал об изменении контента слоя.

        :rtype: Signal[]
        """
        return self._shadow.dataChanged

    @property
    def need_redraw(self) -> Signal:
        """
        Сигнал о необходимости перерисовать слой.

        :rtype: Signal[]
        """
        return self._shadow.needRedraw

    @property
    def title(self) -> str:
        """Наименование слоя."""
        return self._shadow.get_name()

    @title.setter
    def title(self, n: str):
        self._shadow.set_name(n)

    @property
    def coordsystem(self) -> CoordSystem:
        """Координатная система, в которой находятся данные, отображаемые слоем."""
        return CoordSystem._wrap(self._shadow.cs())

    @property
    @_deprecated_by('get_bounds')
    def bounds(self) -> Rect:
        return self.get_bounds()

    def get_bounds(self) -> Rect:
        """Возвращает область, в которую попадают все данные, которые могут быть отображены на слое."""
        return Rect.from_qt(self._shadow.bounds())

    @property
    def data_object(self) -> DataObject:
        """Источник данных для слоя."""
        return DataObject._wrap(self._shadow.data_object())

    @property
    def opacity(self) -> int:
        """Прозрачность слоя в составе карты. Доступные значения от 0 до 100."""
        return self._shadow.get_opacity()

    @opacity.setter
    def opacity(self, n: int):
        self._shadow.set_opacity(n)

    @property
    def zoom_restrict(self) -> bool:
        """Будет ли использоваться ограничение по отображению. Если установлено True, то для ограничения отображения слоя в 
        зависимости от масштаба используются значения свойств `zoom_min` и `zoom_max`"""
        return self._shadow.zoomRestrict()

    @zoom_restrict.setter
    def zoom_restrict(self, v: bool):
        self._shadow.setZoomRestrict(v)

    @property
    def min_zoom(self) -> float:
        """Минимальная ширина окна, при котором слой отображается на карте. Учитывается только при установленном `zoom_restrict=True`"""
        return self._shadow.zoomMin()

    @min_zoom.setter
    def min_zoom(self, v: float):
        self._shadow.setZoomMin(v)

    @property
    def max_zoom(self) -> float:
        """Максимальная ширина окна, при котором слой отображается на карте. Учитывается только при установленном `zoom_restrict=True`"""
        return self._shadow.zoomMax()

    @max_zoom.setter
    def max_zoom(self, v: float):
        self._shadow.setZoomMax(v)

    def __eq__(self, other) -> bool:
        if not isinstance(other, Layer):
            return False
        return ShadowLayer.isEquals(self._shadow, other._shadow if other._shadow is not None else None)

    def __str__(self):
        return str(type(self))

    @property
    def visible(self):
        """Управляет видимостью слоя.
        
        Выключение видимости верхнего слоя для активной карты::

            if view_manager.active is not None:
                view_manager.active.map.layers[0].visible = False
        """
        return self._shadow.is_visible()

    @visible.setter
    def visible(self, v):
        self._shadow.set_visible(v)

    @property
    def is_valid(self) -> bool:
        '''Проверка на валидность объекта. Слой мог быть удален, как пример, в связи с закрытием таблицы'''
        if hasattr(self, '_shadow'):
            return self._shadow is not None
        return False


class ThematicLayer(Layer):
    """Абстрактный класс слоя с тематическим оформлением векторного слоя карты на базе атрибутивной информации.
    """
    @classmethod
    def _wrap(cls, shadow: ShadowThematic):
        from .thematic_layer import (RangeThematicLayer,
                                     PieThematicLayer,
                                     BarThematicLayer,
                                     SymbolThematicLayer,
                                     DensityThematicLayer,
                                     IndividualThematicLayer)
        from axipy.cpp_render import (ShadowThematicRange,
                                      ShadowThematicPie,
                                      ShadowThematicBar,
                                      ShadowThematicSymbol,
                                      ShadowThematicIndividual,
                                      ShadowThematicDensity)
        if isinstance(shadow, ShadowThematicRange):
            return RangeThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicPie):
            return PieThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicBar):
            return BarThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicSymbol):
            return SymbolThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicIndividual):
            return IndividualThematicLayer._wrap(shadow)
        elif isinstance(shadow, ShadowThematicDensity):
            return DensityThematicLayer._wrap(shadow)
        print('Shadow is unknown or null:', shadow)
        return None


class ListThematic:
    """Список тематических слоев (тематик) карты."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow: ShadowThematicList):
        result = ListThematic.__new__(ListThematic)
        result._shadow = shadow
        return result

    def append(self, lay: ThematicLayer):
        """Добавить тематику.

        Args:
            lay: Добавляемый тематический слой.
        """
        self._shadow.add(lay._shadow)

    def add(self, lay: ThematicLayer):
        self._shadow.add(lay._shadow)

    def remove(self, idx: int):
        """Удалить тематику.

        Args:
            idx: Индекс удаляемого слоя.
        """
        self._shadow.remove(idx)

    def move(self, fromIdx: int, toIdx: int):
        """Поменять тематики местами.

        Args:
            fromIdx: Текущий индекс.
            toIdx: Новое положение.
        """
        self._shadow.move(fromIdx, toIdx)

    @property
    def count(self) -> int:
        """Количество тематик слоя."""
        return self.__len__()

    def at(self, idx: int) -> ThematicLayer:
        """Получение тематики по ее индексу.

        Args:
            idx: Индекс запрашиваемой тематики.
        """
        return ThematicLayer._wrap(self._shadow.at(idx))

#    def by_name(self, name: str) -> ThematicLayer:
#        """Получение тематики по ее наименованию.
#
#        Args:
#            name: Наименование запрашиваемой тематики.
#
#        Returns:
#            Если не найдено, возвращает None.
#        """
#        return ThematicLayer._wrap(self._shadow.by_name(name))

    def __getitem__(self, item: Union[int, str]):
        if isinstance(item, str):
            return self._shadow.by_name(item)
        return self.at(item)

    def __len__(self):
        return self._shadow.count()

    def __iter__(self):
        return (self.at(idx) for idx in range(self.__len__()))


class VectorLayer(Layer):
    """Слой, основанный на базе векторных данных.

    Note:
        Создание слоя производится посредством метода вызова :meth:`Layer.create`

    .. literalinclude:: /../../tests/doc_examples/render/test_example_layer.py
        :caption: Примеры работы со свойствами слоя.
        :pyobject: test_run_example_layer_props
        :lines: 3-
        :dedent: 4
    """

    @classmethod
    def _wrap_typed(cls, _shadow: ShadowVectorLayer):
        result = cls.__new__(cls)
        result._shadow = _shadow
        result._thematics = None
        result._label = None
        result._shadow.internalDelete.connect(result.__internalDeleteSlot)
        return result

    def __internalDeleteSlot(self):
        'If shadow obj has been deleted by signal from core'
        self._thematics = None
        self._label = None
        self._shadow = None

    @property
    def showCentroid(self) -> bool:
        """Показ центроидов на слое."""
        return self._shadow.get_show_centroid()

    @showCentroid.setter
    def showCentroid(self, n: bool):
        self._shadow.set_show_centroid(n)

    @property
    def nodesVisible(self) -> bool:
        """Показ узлов линий и полигонов."""
        return self._shadow.get_nodesVisible()

    @nodesVisible.setter
    def nodesVisible(self, n: bool):
        self._shadow.set_nodesVisible(n)

    @property
    def linesDirectionVisibile(self) -> bool:
        """Показ направлений линий."""
        return self._shadow.get_linesDirectionVisibile()

    @linesDirectionVisibile.setter
    def linesDirectionVisibile(self, n: bool):
        self._shadow.set_linesDirectionVisibile(n)

    @property
    def overrideStyle(self) -> Style:
        """Переопределяемый стиль слоя. Если задан как None (по умолчанию), объекты будут отображены на 
        основании оформления источника данных.
        """
        return Style._wrap(self._shadow.get_override_style())

    @overrideStyle.setter
    def overrideStyle(self, style: Style):
        self._shadow.set_override_style(style._shadow if style is not None else None)

    @property
    def thematic(self) -> ListThematic:
        """Перечень тематик для данного слоя. Работа с тематическими слоями похожа на работу со списком `list`.
        
        .. literalinclude:: /../../tests/doc_examples/render/test_example_layer.py
            :caption: Пример.
            :pyobject: test_run_example_layer_thematic
            :lines: 3-11
            :dedent: 4
        
        """
        if self._thematics is None:
            self._thematics = ListThematic._wrap(self._shadow.get_thematics())
        return self._thematics

    @property
    def label(self) -> Label:
        """Метки слоя. В качестве формулы может использоваться или наименование поля таблицы или выражение."""
        if self._label is None:
            self._label = Label._wrap(self._shadow)
        return self._label
    
    @property
    def hotlink(self) -> str:
        """
        Наименование атрибута таблицы для хранения гиперссылки. 

        .. csv-table:: Возможны следующие варианты
            :header: Значение, Описание

            axioma://world.tab, Открывает файл или рабочее пространство в аксиоме
            addlayer://world, Добавляет слой world в текущую карту
            exec://gimp,  Запускает на выполнение программу gimp
            https://axioma-gis.ru/, Открывает ссылку в браузере

        Если префикс отсутствует, то производится попытка запустить по ассоциации.
        """
        return self._shadow.get_hotlink()

    @hotlink.setter
    def hotlink(self, expr: str):
        self._shadow.set_hotlink(expr)



#    def labelProperties(self) -> LabelProperties:
#        """Возвращает свойства подписей."""
#        return LabelProperties._fromShadow(self._shadow.labelProperties())

#    def setLabelProperties(self, properties: LabelProperties):
#        """Устанавливает свойства подписей.
        
#        Args:
#            properties: Свойства подписей.
#        """
#        self._shadow.setLabelProperties(properties._toShadow())


class CosmeticLayer(VectorLayer):
    """Косметический слой.
    """

    @classmethod
    def _wrap_typed(cls, shadow: ShadowCosmeticLayer):
        result = cls.__new__(cls)
        result._shadow = shadow
        result._thematics = None
        result._label = None
        return result


class RasterLayer(Layer):
    """Класс, который должен использоваться в качестве базового класса для тех слоев, в которых используются
    свойства отрисовки растрового изображения.

    Note:
        Создание слоя производится посредством метода вызова :meth:`Layer.create`

    .. literalinclude:: /../../tests/doc_examples/render/test_example_layer.py
        :caption: Примеры создания растрового слоя.
        :pyobject: test_run_example_layer_raster
        :lines: 3-
        :dedent: 4
    """

    @property
    def transparentColor(self) -> QColor:
        """Цвет растра, который обрабатывается как прозрачный."""
        return self._shadow.get_transparentColor()

    @transparentColor.setter
    def transparentColor(self, cl: QColor):
        self._shadow.set_transparentColor(cl)

    @property
    def brightness(self) -> int:
        """Яркость. Значение может быть в пределах от -100 до 100."""
        return self._shadow.get_brightness()

    @brightness.setter
    def brightness(self, v: int):
        self._shadow.set_brightness(v)

    @property
    def contrast(self) -> int:
        """Контраст. Значение может быть в пределах от -100 до 100."""
        return self._shadow.get_contrast()

    @contrast.setter
    def contrast(self, v: int):
        self._shadow.set_contrast(v)

    @property
    def grayscale(self) -> bool:
        """Является ли данное изображение черно-белым."""
        return self._shadow.get_grayscale()

    @grayscale.setter
    def grayscale(self, v: bool):
        self._shadow.set_grayscale(v)
