import os
from typing import Optional, Any

from PySide2.QtWidgets import QApplication
from axipy.cpp_app import ShadowApplication

from axipy._internal._utils import _NoInitReadOnlyMeta
from axipy.da.providers.provider_manager import provider_manager
from axipy.gui import gui_instance

__all__ = [
    "init_axioma",
    "AxiomaInitLanguage",
    "AxiomaInitLogLevel",

    "io",
    "data",
]

# backwards compatibility
io = provider_manager
data = provider_manager


class AxiomaInitLanguage(_NoInitReadOnlyMeta):
    """
    Язык Аксиомы при запуске из python.
    """
    ru = "ru"
    """Русский язык"""
    en = "en"
    """Английский язык"""


class AxiomaInitLogLevel(_NoInitReadOnlyMeta):
    """
    Уровень логирования Аксиомы при запуске из python.
    """
    maximum = "0"
    """Максимальный"""
    high = "1"
    """Высокий"""
    medium = "2"
    """Средний"""
    minimum = "3"
    """Минимальный"""
    disabled = "-1"
    """Логирование отключено"""


def init_axioma(
        *,
        log_level: Optional[AxiomaInitLogLevel] = None,
        language: Optional[AxiomaInitLanguage] = None,
) -> QApplication:
    """
    Инициализирует ядро ГИС Аксиома.

    Args:
        log_level: Уровень логирования.
        language: Язык Аксиомы.

    Returns:
        Приложение Qt5 c очередью событий (event-loop).

    Пример::

        from axipy import init_axioma, mainwindow, AxiomaInitLogLevel

        app = init_axioma(log_level=AxiomaInitLogLevel.disabled)
        mainwindow.show()
        app.exec_()  # запускает обработку очереди событий

    """
    global AXIOMA_CORE
    global APP

    def set_environ(arg: Any, environ_name: str, default_value: str, actual_value: Any = None):
        if actual_value is None:
            actual_value = arg

        if arg is None:
            if os.environ.get(environ_name) is None:
                os.environ[environ_name] = default_value
        else:
            os.environ[environ_name] = actual_value

    set_environ(log_level, "AXIOMA_LOG_LEVEL", AxiomaInitLogLevel.medium)
    # set_environ(load_python_plugins, "AXIOMA_MISS_LOAD_PYTHON_PLUGINS", "True", str(load_python_plugins))
    set_environ(language, "AXIOMA_LANGUAGE", AxiomaInitLanguage.ru)

    if gui_instance._is_initialized():
        AXIOMA_CORE = gui_instance
        APP = QApplication.instance()
    else:
        home_dir = os.environ["AXIOMA_HOME"]
        os.chdir(home_dir)
        APP = QApplication()
        ShadowApplication.initialize(APP)
        AXIOMA_CORE = gui_instance
        AXIOMA_CORE._initialize()
    return APP
