from typing import Optional

from PySide2.QtCore import Signal
from PySide2.QtGui import QResizeEvent, QPaintEvent, QPainter
from PySide2.QtWidgets import QDialog, QPushButton, QWidget
from axipy.cpp_gui import (ShadowChooseCoordSystemWidget, ShadowStyleButton, ShadowGui,
                           PasswordWidget, BoundingRectWidget, ShadowWidgetContainer)

from axipy._internal._decorator import _deprecated_by
from axipy.cs import CoordSystem
from axipy.da import Style
from axipy.utl import Rect

__all__ = [
    "ChooseCoordSystemDialog",
    "StyledButton",
    "StyleButton",
    "PasswordDialog",
    "BoundingRectDialog"
]


class BoundingRectDialog(QDialog):
    """
    Диалог задания параметров прямоугольника. Например, охвата для таблицы или системы координат.

    Пример::

        dlg = BoundingRectDialog()
        dlg.unit = 'км'
        dlg.is_square = True
        dlg.rect = Rect(-100, -120, 200, 200)
        if dlg.exec() == QDialog.Accepted:
            print('>>>>', dlg.rect)

    """

    def __init__(self):
        super().__init__(ShadowGui.global_parent_static())
        self._shadow = BoundingRectWidget(self)
        self._shadow_container = ShadowWidgetContainer(self._shadow, self)

    @property
    def rect(self) -> Rect:
        """
        Параметры прямоугольника
        """
        return Rect.from_qt(self._shadow.resultRect())

    @rect.setter
    def rect(self, r):
        self._shadow.setRect(Rect._rect_value_to_qt(r))

    @property
    def unit(self) -> str:
        """
        Наименование единиц измерения.
        """
        return self._shadow.unitName()

    @unit.setter
    def unit(self, v):
        self._shadow.setUnitName(v)

    @property
    def is_square(self) -> bool:
        """
        Контроль соблюдения равенства ширины и высоты. При этом после ввода какого-либо значения
        производится коррекция таким образом, чтобы в результате получился квадрат.
        """
        self._shadow.isSquare()

    @is_square.setter
    def is_square(self, v):
        self._shadow.setIsSquare(v)


class ChooseCoordSystemDialog(QDialog):
    """
    Диалог выбора координатной системы.

        Args:
            coordsystem: Система координат по умолчанию.
                Если не указана, то задается как значение по умолчанию :meth:`axipy.cs.CoordSystem.current`

        Пример::

            csMercator = CoordSystem.from_prj('10, 104, 7, 0')
            dialog = ChooseCoordSystemDialog(csMercator)
            if dialog.exec() == QDialog.Accepted:
                result_cs = dialog.chosenCoordSystem()
                print("Выбрано:", result_cs.title)   
    """

    def __init__(self, coordsystem: CoordSystem = None):
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        cs = coordsystem if coordsystem is not None else CoordSystem.current()
        self._shadow = ShadowChooseCoordSystemWidget(cs._shadow, self)

    def chosenCoordSystem(self) -> CoordSystem:
        """Возвращает выбранную в диалоге систему координат."""
        return CoordSystem._wrap(self._shadow.chosenCoordSystem())


class StyleButton(QPushButton):
    """
    Кнопка, отображающая стиль и позволяющая его менять.

    Пример добавления кнопки на диалог::

        from PySide2.QtWidgets import QDialog
        from axipy import Style, StyleButton

        style = Style.from_mapinfo("Pen (1, 2, 8421504)  Brush (2, 255, 0)")


        class Dialog(QDialog):
            def __init__(self, parent=None):
                super().__init__(parent)
                self.pb = StyleButton(style, parent=self)
                self.pb.style_changed.connect(self.style_result)
                self.pb.setGeometry(100, 100, 100, 50)

            def style_result(self):
                print("Стиль изменен", self.pb.style)


        dialog = Dialog()
        dialog.open()
    """

    __styleChanged = Signal()

    def __init__(self, style: Optional[Style] = None, parent: Optional[QWidget] = None):
        """

        Args:
            style: Стиль по умолчанию.
            parent: Родительский виджет.

        """
        super().__init__(parent if parent is not None else ShadowGui.global_parent_static())
        self.__shadow = ShadowStyleButton(self, style._shadow if style is not None else None)
        self.__shadow.styleChanged.connect(self.__styleChanged)

    @property
    def style(self) -> Style:
        """Устанавливает или возвращает стиль кнопки."""
        return Style._wrap(ShadowStyleButton.get_style(self.__shadow))

    @style.setter
    def style(self, v: Style):
        if v is not None:
            self.__shadow.set_style(v._shadow)

    @property
    def style_changed(self) -> Signal:
        """Сигнал испускается при смене стиля."""
        return self.__styleChanged

    def paintEvent(self, event: QPaintEvent) -> None:
        super().paintEvent(event)
        painter = QPainter(self)
        painter.drawPixmap(2, 2, self.__shadow.pixmap())

    def resizeEvent(self, event: QResizeEvent) -> None:
        super().resizeEvent(event)
        self.__shadow.reinitPixmap()


class StyledButton(QPushButton):
    """
    Кнопка, отображающая стиль и позволяющая его менять.

    Note:
        Сигнал `styleChanged` испускается при смене стиля.

    Args:
        style: Стиль по умолчанию.

    Пример добавления кнопки на диалог::

        style = Style.from_mapinfo("Pen (1, 2, 8421504)  Brush (2, 255, 0)")

        class Dialog(QDialog):
            def __init__(self, parent = None):
                QDialog.__init__(self, parent)
                self.pb = StyledButton( style, self)
                self.pb.styleChanged.connect(self.styleResult)
                self.pb.setGeometry(100, 100, 100, 50)

            def styleResult(self):
                print('Стиль изменен', self.pb.style())

        dialog = Dialog()
        dialog.exec()
    """
    styleChanged = Signal()

    def __init__(self, style: Optional[Style] = None, parent: Optional[QWidget] = None):
        super().__init__(parent if parent is not None else ShadowGui.global_parent_static())
        self._shadow = ShadowStyleButton(self, style._shadow if style is not None else None)
        self._shadow.styleChanged.connect(self.styleChanged)

    @_deprecated_by("axipy.StyleButton.value property")
    def style(self) -> Style:
        """Результирующий стиль."""
        return Style._wrap(ShadowStyleButton.get_style(self._shadow))

    def paintEvent(self, event: QPaintEvent) -> None:
        super().paintEvent(event)
        painter = QPainter(self)
        painter.drawPixmap(2, 2, self._shadow.pixmap())

    def resizeEvent(self, event: QResizeEvent) -> None:
        super().resizeEvent(event)
        self._shadow.reinitPixmap()


class PasswordDialog(QDialog):
    """
    Диалог задания или корректировки данных аутентификации пользователя.

    Пример::

        dialog = PasswordDialog()
        dialog.user_name = 'user'
        dialog.password = 'pass'
        if dialog.exec() == QDialog.Accepted:
            print(dialog.user_name, dialog.password)
    """

    def __init__(self):
        super().__init__(ShadowGui.global_parent_static())
        if self.layout() is not None:
            self.layout().setContentsMargins(0, 0, 0, 0)
        self._shadow = PasswordWidget(self)

    @property
    def user_name(self) -> str:
        """Имя пользователя."""
        return self._shadow.userName()

    @user_name.setter
    def user_name(self, v):
        self._shadow.setUserName(v)

    @property
    def password(self) -> str:
        """Пароль."""
        return self._shadow.password()

    @password.setter
    def password(self, v):
        self._shadow.setPassword(v)
