from enum import Enum
from typing import Optional

from PySide2.QtCore import QObject, Qt, QUrl
from PySide2.QtGui import QIcon
from PySide2.QtWidgets import QMainWindow, QMdiSubWindow, QDockWidget
from axipy.cpp_app import ShadowMainWindow

from axipy._internal._shadow_instance_factory import _ShadowManager
from axipy.da import DataManager
from axipy.gui import View, gui_instance, MapView
from axipy.render import Layer

__all__ = [
    'MainWindow',
    'mainwindow',
    'DockWidgetArea'
]


class DockWidgetArea(Enum):
    """Расположение панели"""

    Left = Qt.LeftDockWidgetArea
    """Слева"""
    Right = Qt.RightDockWidgetArea
    """Справа"""
    Top = Qt.TopDockWidgetArea
    """Сверху"""
    Bottom = Qt.BottomDockWidgetArea
    """Снизу"""


class MainWindow(QObject):
    """Главное окно ГИС Аксиома.

    Note:
        Используйте готовый объект :attr:`axipy.mainwindow`.
    """

    def __init__(self):
        super().__init__()
        self._shadow = None
        self._catalog = None

    @staticmethod
    def show() -> 'MainWindow':
        """Создает и показывает главное окно программы."""
        ShadowMainWindow.create(gui_instance._shadow)
        mainwindow.qt_object().show()
        return mainwindow

    def service(self):
        if self._shadow is None:
            try:
                self._shadow = _ShadowManager.main_window
            except RuntimeError:
                pass
        return self._shadow

    @property
    def is_valid(self) -> bool:
        """Корректность состояния главного окна."""
        return self.service() is not None

    def qt_object(self) -> QMainWindow:
        """Возвращает Qt5 объект окна."""
        return self.service().qt_object() if self.service() is not None else None

    @property
    def catalog(self) -> DataManager:
        """Хранилище объектов приложения.

        Это то же хранилище, которое отображается в панели "Открытые данные".

        Note:
            При открытии объектов данных :meth:`axipy.da.ProviderManager.openfile`
            они автоматически попадают в каталог.
        """
        if self._catalog is None:
            self._catalog = DataManager._wrap(gui_instance._shadow.catalog())
        return self._catalog

    def add(self, view: View) -> QMdiSubWindow:
        """
        Добавляет окно просмотра данных.

        Args:
            view: окно просмотра данных.

        Note:
            При создании окон просмотра данных :meth:`axipy.gui.ViewManager.create_mapview`
            или :meth:`axipy.gui.ViewManager.create_tableview`
            они автоматически добавляются в главное окно программы.
        """
        return self.service().add(view._shadow)

    def load_workspace(self, fileName: str):
        """Читает рабочее пространство из файла.
        
        Args:
            fileName: Наименование входного файла.
        """
        self.service().open_workspace(fileName)

    def save_workspace(self, fileName: str):
        """Сохраняет рабочее пространство в файл.

        Args:
            fileName: Наименование выходного файла.
        """
        self.service().save_workspace(fileName)

    def add_layer_interactive(self, layer: Layer) -> MapView:
        """Добавляет слой с запросом на помещение на текущую карту или в новую."""
        return View._wrap(self.service().addLayerInteractive(layer._shadow))

    def add_layer_current_map(self, layer: Layer) -> MapView:
        """Добавляет слой в текущей карте."""
        return View._wrap(self.service().addLayerCurrentMap(layer._shadow))

    def add_layer_new_map(self, layer: Layer) -> MapView:
        """Открывает слой в новой карте."""
        return View._wrap(self.service().addLayerNewMap(layer._shadow))

    def add_dock_widget(
            self,
            dock_widget: QDockWidget,
            area: DockWidgetArea,
            icon: QIcon = None
    ) -> bool:
        """
        Добавляет панель в главное окно приложения. При успешном добавлении возвращает True.
        Если же данная панель уже присутствует, то команда игнорируется и возвращается False.
        Элементы управления, которые требуется разместить на панели, создаются в дополнительном окне,
        а уже это окно, в свою очередь, устанавливается для панели (см. пример ниже).
        
        Args:
            dock_widget: Пользовательская созданная панель.
            area: Расположение.
            icon: Иконка для отображения в списке всех доступных панелей.

        Пример::

            from PySide2.QtWidgets import QDockWidget, QWidget, QPushButton
            from PySide2.QtCore import Qt

            dock = QDockWidget('Заголовок')
            widget = QWidget()
            layout = QVBoxLayout()
            button = QPushButton("Кнопка")
            button.clicked.connect(lambda: print('Реакция на кнопку'))
            layout.addWidget(button)
            layout.addStretch()
            widget.setLayout(layout)
            dock.setWidget(widget)
            app.mainwindow.add_dock_widget(dock, Qt.RightDockWidgetArea, QIcon('filename.png'))
        """

        if isinstance(area, DockWidgetArea):
            area = area.value

        return self.service().addDockWidget(dock_widget, area, icon if icon is not None else QIcon())

    def remove_dock_widget(self, dock: QDockWidget):
        """Удаляет существующую панель у главного окна приложения."""
        return self.service().removeDockWidget(dock)

    def show_html_url(self, url: QUrl, caption: Optional[str]):
        """
        Показывает окно для локального файла html или если это web страница, запускает браузер по ассоциации.

        Args:
            url: Ссылка на файл html или адрес страницы.
            caption: Заголовок окна.
        """
        self.service().showHtmlUrl(caption, url)


mainwindow = MainWindow()
