"""
Окружность по трём точкам.
"""
from typing import List

from PySide2.QtCore import Qt, QPoint
from PySide2.QtGui import QPainter, QPen

from axipy import Notifications
from axipy import Plugin
from axipy import tr
from axipy.da import Feature, Style
from axipy.mi import Ellipse
from axipy.utl import Pnt, Rect
from .. import circle_utils
from ..circle_utils import CircleMapTool
from ..helper import print_exc_


class CircleByPoints(CircleMapTool):

    def __init__(self, iface: Plugin, title: str) -> None:
        super().__init__(iface, title)

    def prepare_to_draw(self, painter: QPainter, points: List[QPoint]):
        draw_qt_ellipse(painter, points)

    def prepare_to_insert(self) -> None:
        try:
            points = circle_utils.ensure_util_geom_visual(self.view, self.points)
            result = circle_utils.center_and_radius_of_circle(points)
            if result is None:
                raise Exception("None")
            center, radius = result
            self.insert_ellipse(center, radius)
        except (Exception,):
            print_exc_()
            Notifications.push(
                self.title,
                tr("Не удалось построить окружность по 3-м точкам. Попробуйте ещё раз, изменив координаты."),
                Notifications.Critical
            )
        finally:
            self.clear_and_redraw()

    def insert_ellipse(self, c: Pnt, r: float):
        rect = Rect(c.x - r, c.y - r, c.x + r, c.y + r)
        rect = circle_utils.ensure_util_geom_visual(self.view, rect, inverse=True)
        ellipse = Ellipse(rect=rect, cs=self.view.coordsystem)
        ellipse_style = Style.from_mapinfo("Pen (1, 2, 0) Brush (1, 16777215, 16777215)")
        f = Feature(geometry=ellipse, style=ellipse_style)
        circle_utils.insert_feature(f)
        Notifications.push(self.title, tr("Окружность успешно создана."), Notifications.Success)


def draw_qt_ellipse(painter: QPainter, points: List[Pnt]):
    result = circle_utils.center_and_radius_of_circle(points)
    if result is None:
        return None
    center, radius = result
    if radius == 0:
        painter.drawLine(*points[::2])
        return None

    # Запомнить старый стиль
    pen = painter.pen()
    old_pen = QPen(pen)
    # Задать новый стиль
    pen.setStyle(Qt.DashLine)
    painter.setPen(pen)
    # Нарисовать
    painter.drawEllipse(center.to_qt(), radius, radius)
    # Возвратить старый стиль
    painter.setPen(old_pen)
