from pathlib import Path
from typing import Union, Any, Callable

import axipy.app.main_window
from PySide2.QtCore import QCoreApplication
from PySide2.QtCore import QSettings
from PySide2.QtGui import QIcon
from axipy._internal._shadow_instance_factory import _ShadowManager
from axipy.da.observer_manager import Observer
from axipy.gui import gui_instance, MapTool, view_manager
from axipy.menubar import Button, ActionButton, ToolButton, title_to_id

__all__ = [
    "Plugin",
]


class Plugin:
    """
    Вспомогательный класс для создания плагинов.
    """

    @classmethod
    def __create(cls, plugin_dir: Path, *args, **kwargs) -> 'Plugin':
        inst = super().__new__(cls, *args, **kwargs)
        inst.__plugin_dir = plugin_dir
        inst.__plugin_id = plugin_dir.name
        inst.__settings = QSettings("axipy_plugin", inst.__plugin_id)
        inst.__init__()
        if axipy.app.main_window.mainwindow.is_valid:
            inst.load()
        else:
            view_manager.mainwindow_activated.connect(inst.__load)
        return inst

    def tr(self, text: str) -> str:
        """
        Ищет перевод строки. Производит поиск строки в загруженных файлах перевода.

        Args:
            text:
                Строка для перевода.

        Returns:
            Перевод стоки, если строка найдена. Иначе - сама переданная строка.

        Пример::

            button_name = self.tr("My button")
        """
        return QCoreApplication.translate(self.__plugin_id, text)

    @property
    def language(self) -> str:
        """Значение языка, с которым запущено приложение. Например ``'ru'``."""
        return _ShadowManager.core.translationLanguage()

    @property
    def settings(self) -> QSettings:
        """
        Настройки плагина.

        Позволяет сохранять и загружать параметры.

        See also:
            Подробнее в документации на класс :class:`PySide2.QtCore.QSettings`.
        """
        return self.__settings

    def __unload(self) -> None:
        self.unload()

    def __load(self) -> None:
        view_manager.mainwindow_activated.disconnect(self.__load)
        self.load()

    def load(self) -> None:
        """
        Переопределите этот метод для задания логики загрузки плагина.
        """
        pass

    def unload(self) -> None:
        """
        Переопределите этот метод для очистки ресурсов при выгрузке плагина.
        """
        pass

    def create_tool(
            self,
            title: str,
            on_click: Union[Callable[[], MapTool], MapTool],
            icon: Union[str, QIcon] = "",
            enable_on: Union[str, Observer] = None,
            tooltip: str = None,
            doc_file: str = None
    ) -> ToolButton:
        """
        Создает кнопку с инструментом.

        Args:
            title: Текст.
            on_click: Класс инструмента.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Относительная ссылка на файл документации. Расположение рассматривается по отношению к
              каталогу `documentation`.
        Return:
            Кнопка с инструментом.

        Note:
            То же, что и :class:`ToolButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного плагина.
        """
        result = ToolButton(title, on_click, icon, enable_on, tooltip)
        if doc_file is not None:
            result.action.setWhatsThis(self.__full_doc_file(doc_file))
        self.__disambiguate(result)
        return result

    def __full_doc_file(self, filename) -> str:
        doc_path = str(self.plugin_dir / "documentation" / filename)
        return "file:///{}".format(doc_path.replace("\\", "/"))

    def create_action(
            self,
            title: str,
            on_click: Callable[[], Any],
            icon: Union[str, QIcon] = "",
            enable_on: Observer = None,
            tooltip: str = None,
            doc_file: str = None
    ) -> ActionButton:
        """
        Создает кнопку с действием.

        Args:
            title: Текст.
            on_click: Действие на нажатие.
            icon: Иконка. Может быть путем к файлу или адресом ресурса.
            enable_on: Идентификатор наблюдателя для определения доступности кнопки.
            tooltip: Строка с дополнительной короткой информацией по данному действию.
            doc_file: Относительная ссылка на файл документации. Расположение рассматривается по отношению к
              каталогу `documentation`.
        Return:
            Кнопка с действием.

        Note:
            То же, что и :class:`ActionButton`, но дополнительно
            делает идентификатор кнопки уникальным для данного плагина.
        """
        result = ActionButton(title, on_click, icon, enable_on, tooltip=tooltip)
        if doc_file is not None:
            result.action.setWhatsThis(self.__full_doc_file(doc_file))
        self.__disambiguate(result)
        return result

    def __disambiguate(self, button: Button):

        def make_action_id(title: str) -> str:
            return self.__plugin_id + "_" + title_to_id(title)

        button.action.setObjectName(make_action_id(button.action.text()))

    @property
    def plugin_dir(self) -> Path:
        """
        Возвращает путь к папке плагина.
        """
        return self.__plugin_dir

    def get_plugin_data_dir(self) -> Path:
        """
        Возвращает каталог, в котором находятся изменяемые данные плагина.
        """
        path = Path(gui_instance._shadow.installedPluginsPath()) / "installed_modules" / "data" / self.__plugin_id
        path.mkdir(parents=True, exist_ok=True)
        return path
