from typing import Optional

from PySide2.QtCore import QRect, Signal
from PySide2.QtGui import QTransform
from PySide2.QtPrintSupport import QPrinter
from PySide2.QtWidgets import QWidget, QTableView
from axipy.cpp_gui import ShadowView, ShadowMapView, ShadowTableView, ShadowReportView, ShadowLegendView

from axipy.cs import CoordSystem, LinearUnit
from axipy.da import DataObject
from axipy.render import Map, Layer, VectorLayer, Report, Legend
from axipy.utl import Rect, Pnt

__all__ = [
    "View",
    "TableView",
    "DrawableView",
    "ReportView",
    "MapView",
    "ListLegend",
    "LegendView",
]


class View:
    """Базовый класс для отображения данных в окне."""

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result._shadow = shadow
        return result

    @classmethod
    def _wrap(cls, obj: ShadowView):
        if isinstance(obj, ShadowTableView):
            return TableView._wrap_typed(obj)
        elif isinstance(obj, ShadowMapView):
            return MapView._wrap_typed(obj)
        elif isinstance(obj, ShadowReportView):
            return ReportView._wrap_typed(obj)
        elif isinstance(obj, ShadowLegendView):
            return LegendView._wrap_typed(obj)
        return None

    @property
    def widget(self) -> QWidget:
        """Виджет, соответствующий содержимому окна.

        Return:
            Qt5 виджет содержимого.
        """
        return self._shadow.widget()

    @property
    def title(self) -> str:
        """Заголовок окна просмотра."""
        return self._shadow.widget().windowTitle()

    @title.setter
    def title(self, v: str):
        self._shadow.widget().setWindowTitle(v)

    def __base_widget(self):
        w = self.widget
        while w is not None:
            if w.__class__.__name__ == 'QMdiSubWindow':
                return w
            p = w.parent()
            if p is not None:
                w = p
            else:
                break
        return self.widget

    @property
    def rect(self) -> QRect:
        """Размер и положение окна.
        
        Warning:

            .. deprecated:: 4.0
                Используйте :attr:`position`.
        """
        return self.position

    @rect.setter
    def rect(self, v: QRect):
        self.position = v

    @property
    def position(self) -> QRect:
        """Размер и положение окна."""
        w = self.__base_widget()
        if w is not None:
            return w.geometry()
        return QRect()

    @position.setter
    def position(self, v: QRect):
        w = self.__base_widget()
        if w is not None:
            rect = Rect._rect_value_to_qt(v).toRect()
            w.setGeometry(rect)
            """
            move нужен, чтобы переместилась вся рамка виджета, включая заголовок,
            функция setGeometry не учитывает размер заголовка, при размещении виджета на экране,
            в результате, если запускать виджет без аксиомы, под Windows,
            при position = Rect(10, 10, 500, 500), виджет уходит за видимую область экрана
            """
            w.move(rect.x(), rect.y())

    SHOW_NORMAL = 1
    SHOW_MINIMIZED = 2
    SHOW_MAXIMIZED = 3

    def show(self, type: int = SHOW_NORMAL):
        """Показывает окно в соответствие с приведенным типом.

        .. csv-table:: Допустимые значения:
            :header: "Константа", "Значение", "Описание"
            :align: left
            :widths: 10, 5, 40

            SHOW_NORMAL, 1, "Обычный показ окна (по умолчанию)."
            SHOW_MINIMIZED, 2, "Показ окна в режиме минимизации."
            SHOW_MAXIMIZED, 3, "Показ окна в режиме распахивания."
        """
        w = self.__base_widget()
        if w is not None:
            if type == self.SHOW_MAXIMIZED:
                w.showMaximized()
            elif type == self.SHOW_MINIMIZED:
                w.showMinimized()
            else:
                w.showNormal()

    def close(self):
        """Закрывает окно."""
        w = self.__base_widget()
        if w is not None:
            w.close()

    @property
    def show_type(self) -> int:
        """Возвращает тип состояния окна. Подробнее см. :meth:`show`"""
        w = self.__base_widget()
        if w is not None:
            if w.isMaximized():
                return self.SHOW_MAXIMIZED
            elif w.isMinimized():
                return self.SHOW_MINIMIZED
        return self.SHOW_NORMAL


    def __str__(self):
        return str(type(self))


class TableView(View):
    """Таблица просмотра атрибутивной информации.
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_tableview` через экземпляр `view_manager`
    """

    @property
    def data_object(self) -> DataObject:
        """Таблица, на основании которой создается данное окно просмотра.

        Return:
            Таблица.
        """
        return DataObject._wrap(self._shadow.dataobject())

    @property
    def table_view(self) -> QTableView:
        """Ссылка на объект таблицы просмотра.
        
        Пример установки сортировки для таблицы в текущем окне по второй колонке по возрастанию::

            if isinstance(view_manager.active, TableView):
                view_manager.active.table_view.sortByColumn(2, Qt.AscendingOrder)
        
        """
        return self._shadow.table_view()


class DrawableView(View):
    """Базовый класс для визуализации геометрических данных."""

    def __init__(self):
        raise NotImplementedError

    @property
    def snap_mode(self) -> bool:
        """Включает режим привязки координат при редактировании геометрии в окне карты или отчета."""
        return self._shadow.isSnapMode()
    
    @snap_mode.setter
    def snap_mode(self, v):
        self._shadow.setSnapMode(v)

    def scale_with_center(self, scale: float, center: Pnt):
        """Установка нового центра с заданным масштабированием.
        
        Args:
            scale: Коэффициент масштабирования по отношению к текущему.
            center: Устанавливаемый центр.
        """
        self._shadow.scaleWithCenter(scale, Pnt._point_value_to_qt(center))

    @property
    def scene_changed(self) -> Signal:
        """
        Сигнал об изменении контента окна.

        :rtype: Signal[]
        """
        return self._shadow.sceneChanged


    @property
    def is_modified(self) -> bool:
        """Есть ли изменения в окне."""
        return self._shadow.hasModified()

    @property
    def can_undo(self) -> bool:
        """Возможен ли откат на один шаг назад."""
        return self._shadow.canRollBack()
    
    def undo(self):
        """Производит откат на один шаг назад."""
        if self.can_undo:
            self._shadow.rollBack()

    @property
    def can_redo(self) -> bool:
        """Возможен ли откат на один шаг вперед."""
        return self._shadow.canRollForward()
    
    def redo(self):
        """Производит откат на один шаг вперед. При этом возвращается состояние до последней отмены."""
        if self.can_redo:
            self._shadow.rollForward()

    def offset(self, dx: float, dy: float):
        """Производит сдвиг окна карты или отчета. Особенностью является то, что при этом сохраняется
        прежний центр (актуально для карты).
        
        Args:
            dx: Смещение по горизонтали в координатах экрана (пикселях)
            dy: Смещение по вертикали в координатах экрана (пикселях)
        """
        self._shadow.offset(dx, dy)


class Guidelines:

    def __init__(self):
        raise NotImplementedError

    @classmethod
    def _wrap(cls, shadow):
        result = cls.__new__(cls)
        result._shadow = shadow
        return result


class XGuidelines(Guidelines):

    def append(self, v):
        self._shadow.addXGuideline(v)

    def __getitem__(self, idx):
        return self._shadow.xGuidelines()[idx]

    def __setitem__(self, idx, v):
        self._shadow.setXGuideline(idx, v)
    
    def __len__(self):
        return len(self._shadow.xGuidelines())


class YGuidelines(Guidelines):

    def append(self, v):
        self._shadow.addYGuideline(v)

    def __getitem__(self, idx):
        return self._shadow.yGuidelines()[idx]

    def __setitem__(self, idx, v):
        self._shadow.setYGuideline(idx, v)
    
    def __len__(self):
        return len(self._shadow.yGuidelines())


class ReportView(DrawableView):
    """Окно с планом отчета. 
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_reportview` через экземпляр `view_manager`.
    До параметров самого отчета :class:`axipy.render.Report` можно доступиться через свойство :meth:`ReportView.report`

    Пример создания отчета::

        reportview = view_manager.create_reportview()

        # Добавим полигон
        geomItem = GeometryReportItem()
        geomItem.geometry = Polygon((10,10), (10, 100), (100, 100), (10, 10))
        geomItem.style = PolygonStyle(45, Qt.red)
        reportview.report.items.add(geomItem)

        # Установим текущий масштаб
        reportview.view_scale = 33
    """

    def __init__(self):
        super().__init__()
        self.__xguidelines = None
        self.__yguidelines = None

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result._shadow = shadow
        result.mouse_moved = result._shadow.mouseMoved
        result.__xguidelines = None
        result.__yguidelines = None
        return result

    @property
    def report(self) -> Report:
        """Объект отчета.

        Return:
            Отчет.
        """
        return Report._wrap(ShadowReportView.report(self._shadow))

    @property
    def show_mesh(self) -> bool:
        """Показывать сетку привязки."""
        return self._shadow.showMesh()
    
    @show_mesh.setter
    def show_mesh(self, v):
        self._shadow.setShowMesh(v)

    @property
    def snap_to_mesh(self) -> bool:
        """Включение режима притяжения элементов отчета к узлам сетки."""
        return self._shadow.snapToMesh()
    
    @snap_to_mesh.setter
    def snap_to_mesh(self, v):
        self._shadow.setSnapToMesh(v)

    @property
    def snap_to_guidelines(self) -> bool:
        """Включение режима притяжения элементов отчета к направляющим."""
        return self._shadow.snapToGuidelines()
    
    @snap_to_guidelines.setter
    def snap_to_guidelines(self, v):
        self._shadow.setSnapToGuidelines(v)

    @property
    def x_guidelines(self):
        """Вертикальные направляющие. Значения содержатся в единицах измерения отчета. 
        
        Рассмотрим на примере::

            # Добавление вертикальной направляющей
            reportview.x_guidelines.append(20)
            # Изменение значения направляющей по индексу
            reportview.x_guidelines[0] = 80
            # Удаление всех направляющих.            
            reportview.clear_guidelines()
        """
        if self.__xguidelines is None:
            self.__xguidelines = XGuidelines._wrap(self._shadow)
        return self.__xguidelines

    @property
    def y_guidelines(self):
        """Горизонтальные направляющие. Работа с ними производится по аналогии с вертикальными направляющими."""
        if self.__yguidelines is None:
            self.__yguidelines = YGuidelines._wrap(self._shadow)
        return self.__yguidelines

    def clear_guidelines(self):
        """Удаляет все направляющие."""
        self._shadow.clearGuidelines()

    def clear_selected_guidelines(self):
        """Очищает выбранные направляющие."""
        self._shadow.removeSelectedGuideline()

    def fill_on_pages(self):
        """Наиболее эффективно заполняет пространство отчета масштабированием его элементов."""
        self._shadow.fillOnPages()

    @property
    def view_scale(self) -> float:
        """Текущий масштаб."""
        return self._shadow.viewScale()

    @view_scale.setter
    def view_scale(self, v):
        self._shadow.setViewScale(v)

    @property
    def show_borders(self) -> float:
        """Показывать границы страниц."""
        return self._shadow.showLayoutBorders()

    @show_borders.setter
    def show_borders(self, v):
        self._shadow.setShowLayoutBorders(v)

    @property
    def show_ruler(self) -> float:
        """Показывать линейку по краям."""
        return self._shadow.showLayoutRuler()

    @show_ruler.setter
    def show_ruler(self, v):
        self._shadow.setShowLayoutRuler(v)

    @property
    def mesh_size(self) -> float:
        """Размер ячейки сетки."""
        return self._shadow.meshSize()

    @mesh_size.setter
    def mesh_size(self, v):
        self._shadow.setMeshSize(v)

    def mouse_moved(self, x: float, y: float)-> Signal:
        """Сигнал при смещении курсора мыши. Возвращает значения в координатах отчета.
        
        Args:
            x: X координата
            y: Y координата

        Пример::

            reportview.mouse_moved.connect(lambda x,y: print('Coords: {} {}'.format(x, y)))
        """
        return None

    def get_printer(self) -> QPrinter:
        """Ссылка на используемый текущий принтер. Для того, чтобы изменить настройки, нужно запросить существующий
        объект, поменять необходимые значения и снова назначить посредством :meth:`ReportView.set_printer`. Или же установить другой объект :class:`PySide2.QtPrintSupport.QPrinter`.

        .. literalinclude:: /../../tests/doc_examples/render/test_example_report.py
            :caption: Пример смены свойств у текущего окна отчета
            :pyobject: test_run_example_report_printer
            :lines: 3-
            :dedent: 4
        """
        return self._shadow.get_printer()

    def set_printer(self, printer: QPrinter):
        """Устанавливает для отчета объект :class:`PySide2.QtPrintSupport.QPrinter`
        
        Args:
            printer: Новое значение принтера или измененное запрошенное ранее через :meth:`ReportView.get_printer`
        """
        self._shadow.set_printer(printer)


class MapView(DrawableView):
    """Окно просмотра карты. Используется для проведения различных манипуляций с картой.
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_mapview` через экземпляр `view_manager` (пример см. ниже).

    Note:
        При создании 'MapView' посредством :meth:`axipy.gui.ViewManager.create_mapview` производится клонирование экземпляра карты :class:`axipy.render.Map` 
        и для последующей работы при доступе к данному объекту необходимо использовать свойство :attr:`map`.

    Свойство :attr:`device_rect` определяет размер самого окна карты, а свойство :attr:`scene_rect`
    - прямоугольную область, которая умещается в этом окне в СК карты.

    Преобразование между этими двумя прямоугольниками производится с помощью матриц трансформации 
    :attr:`scene_to_device_transform` и :attr:`device_to_scene_transform`. 

    К параметрам самой карты можно получить доступ через свойство :attr:`map`.
    Единицы измерения координат (:attr:`unit`) также берутся из наиболее подходящей СК, но при желании они могут быть изменены.
    К примеру, вместо метров могут быть установлены километры.

    Рассмотрим пример создания карты с последующим помещением ее в окно ее просмотра. Далее, попробуем преобразовать
    объект типа полигон из координат окна экрана в координаты СК слоя посредством :meth:`axipy.da.Geometry.affine_transform`.

    ::

        # Откроем таблицу, создадим на ее базе слой и добавим в карту
        table_world = provider_manager.openfile('world.tab')
        world = Layer.create(table_world)
        map = Map([ world ])
        # Для полученной карты создадим окно просмотра
        mapview = view_manager.create_mapview(map)
        # Выведем полученные параметры отображения
        print('Прямоугольник экрана:', mapview.device_rect)
        print('Прямоугольник карты:', mapview.scene_rect)
        # Установим ширину карты и ее центр
        mapview.center = (1000000, 1000000)
        mapview.set_zoom(10e6 )

        >>> Прямоугольник экрана: (0.0 0.0) (300.0 200.0)
        >>> Прямоугольник карты: (-16194966.287183324 -8621185.324024437) (16789976.633236416 8326222.646170927)

    ::

        #Создадим геометрический объект полигон в координатах экрана и преобразуем его в СК карты
        poly_device = Polygon([(100,100), (100,150), (150, 150), (150,100)])
        # Используя матрицу трансформации, преобразуем его в координаты карты.
        poly_scene = poly_device.affine_transform(mapview.device_to_scene_transform)
        # Для контроля выведем полученный полигон в виде WKT
        print('WKT:', poly_scene.to_wkt())

        >>> WKT: POLYGON ((-5199985.31371008 -147481.338926755, -5199985.31371008 -4384333.3314756, 297505.173026545 -4384333.3314756, 297505.173026545 -147481.338926755, -5199985.31371008 -147481.338926755))
    """

    @property
    def map(self) -> Map:
        """Объект карты.

        Return:
            Карта.
        """
        return Map._wrap(ShadowMapView.map(self._shadow))

    @property
    def coordsystem(self) -> CoordSystem:
        """Система координат карты."""
        return CoordSystem._wrap(self._shadow.coordsystem())

    @coordsystem.setter
    def coordsystem(self, new_coordsystem):
        self._shadow.setCoordSystem(new_coordsystem._shadow)

    @property
    def editable_layer(self) -> Optional[VectorLayer]:
        """Редактируемый слой на карте.

        Return:
            Редактируемый слой. Если не определен, возвращает None.
        """
        return Layer._wrap(ShadowMapView.editableLayer(self._shadow))

    @editable_layer.setter
    def editable_layer(self, layer):
        self.map.editable_layer = layer

    @property
    def selected_layer(self) -> Optional[VectorLayer]:
        """Выделенный слой на карте.

        Return:
            Выделенный слой. Если выделение отсутствует, возвращает None.
        """
        return Layer._wrap(ShadowMapView.selectionLayer(self._shadow))

    @property
    def unit(self) -> LinearUnit:
        """Единицы измерения координат карты."""
        return LinearUnit._wrap(self._shadow.get_unit())

    @unit.setter
    def unit(self, unit: LinearUnit):
        self._shadow.set_unit(unit._shadow)

    @property
    def scene_rect(self) -> Rect:
        """Видимая область в координатах карты (в единицах измерения СК).

        Return:
            Прямоугольник в координатах карты.
        """
        return Rect.from_qt(self._shadow.sceneRect())

    @scene_rect.setter
    def scene_rect(self, rect: Rect):
        self._shadow.setSceneRect(rect.to_qt())

    @property
    def device_rect(self) -> Rect:
        """Видимая область в координатах окна (пиксели).

        Return:
            Прямоугольник в координатах окна.
        """
        return Rect.from_qt(self._shadow.deviceRect())

    @property
    def scene_to_device_transform(self) -> QTransform:
        """Объект трансформации из координат карты в координаты окна.

        Return:
            Объект трансформации.
        """
        return self._shadow.sceneToDeviceTransform()

    @property
    def device_to_scene_transform(self) -> QTransform:
        """Объект трансформации из координат окна в координаты карты.

        Return:
            Объект трансформации.
        """
        return self._shadow.deviceToSceneTransform()

    @property
    def editable_layer_changed(self) -> Signal:
        """Сигнал о том, что редактируемый слой сменился."""
        return self._shadow.editableLayerChanged

    def zoom(self, unit: LinearUnit = None) -> float:
        """Ширина окна карты.
        
        Args:
            unit: Единицы измерения. Если не заданы, берутся текущие для карты.
        """
        return self._shadow.zoom(unit._shadow if unit is not None else None)

    def set_zoom(self, zoom: float, unit: LinearUnit = None):
        """"Задание ширины окна карты.
        
        Args:
            zoom: Значение ширины карты.
            unit: Единицы измерения. Если не заданы, берутся текущие для карты.
        """
        self._shadow.setZoom(zoom, unit._shadow if unit is not None else None)

    def set_zoom_and_center(self, zoom: float, center: Pnt, unit: LinearUnit = None):
        """"Задает новый центр и ширину окна карты.
        
        Args:
            zoom: Значение ширины карты.
            center: Центр карты.
            unit: Единицы измерения. Если не заданы, берутся текущие для карты.
        """
        self._shadow.setZoomAndCenter(zoom, Pnt._point_value_to_qt(center), unit._shadow if unit is not None else None)

    @property
    def center(self) -> Pnt:
        """Центр окна карты."""
        return Pnt.from_qt(self._shadow.center())
    
    @center.setter
    def center(self, p: Pnt):
        self._shadow.setCenter(Pnt._point_value_to_qt(p))

    @property
    def scale(self) -> float:
        """Масштаб карты."""
        return self._shadow.scale()
    
    @scale.setter
    def scale(self, v: float):
        self._shadow.setScale(v)

    def show_all(self):
        """Полностью показывает все слои карты."""
        self._shadow.showAll()

    def show_selection(self):
        """Перемещает карту к группе выделенных объектов, максимально увеличивая масштаб,
         но так, чтобы все объекты попадали."""
        self._shadow.showSelection()

    @property
    def coordsystem_changed(self) -> Signal:
        """Сигнал о том, что система координат изменилась.
        
        Пример::

            layer_world = Layer.create(table_world)
            map = Map([ layer_world ])
            mapview = view_manager.create_mapview(map)
            mapview.coordsystem_changed.connect(lambda: print('СК была изменена'))
            csLL = CoordSystem.from_prj("1, 104")
            mapview.coordsystem = csLL

            >>> СК была изменена
        """
        return self._shadow.coordSystemChanged

    @classmethod
    def _wrap_typed(cls, shadow):
        result = cls.__new__(cls)
        result._shadow = shadow
        result.mouse_moved = result._shadow.mouseMoved
        return result

    def mouse_moved(self, x: float, y: float)-> Signal:
        """Сигнал при смещении курсора мыши. Возвращает значения в СК карты.
        
        Args:
            x: X координата
            y: Y координата

        Пример::

            mapview.mouse_moved.connect(lambda x,y: print('Coords: {} {}'.format(x, y)))

            >> Coords: -5732500.0 958500.0
            >> Coords: -5621900.0 847900.0
        """
        return None

    @property
    def coordsystem_visual(self) -> CoordSystem:
        """Система координат карты с учетом поправки цены градуса по широте.
        Отличается от :attr:`coordsystem` только лишь в случае, когда основная система 
        координат - это широта/долгота и широта имеет ненулевое значение.
        При этом в диапазоне широты (-70...70) градусов вводится поправочный коэффициент,
        растягивающий изображение по широте и равный 1/cos(y).
        """
        return CoordSystem._wrap(self._shadow.coordSystemVisual())


# class ListLegend(_NoInit, MutableSequence):
class ListLegend:
    """Список добавленных в окно легенд."""

    # def __setitem__(self, index, value):
    #     if isinstance(value, Legend):
    #         self._shadow.replace(index, value._shadow)
    #     else:
    #         raise TypeError
    #
    # def __delitem__(self, index):
    #     self._shadow.remove(index)
    #
    # def insert(self, index, value):
    #     if isinstance(value, Legend):
    #         # TODO: implement insert
    #         raise NotImplementedError
    #     else:
    #         raise TypeError
    #
    # def __getitem__(self, index):
    #     return Legend._wrap_typed(ShadowLegendView.at(self._shadow, index))
    #
    # def __len__(self):
    #     return self._shadow.count()

    @classmethod
    def _wrap(cls, shadow: ShadowLegendView):
        result = ListLegend.__new__(ListLegend)
        result._shadow = shadow
        return result

    def append(self, legend: Legend):
        self._shadow.append(legend._shadow)

    def __getitem__(self, idx: int):
        return Legend._wrap_typed(ShadowLegendView.at(self._shadow, idx))

    def __len__(self):
       return self._shadow.count()

    def remove(self, idx: int):
        self._shadow.remove(idx)

    def __iter__(self):
       return (self.__getitem__(idx) for idx in range(self.__len__()))


class LegendView(View):
    """Легенда для карты.
    Для создания экземпляра необходимо использовать :meth:`axipy.gui.ViewManager.create_legendview` через экземпляр `view_manager`.
    В качестве параметра передается открытое ранее окно с картой::

        legendView = view_manager.create_legendview(map_view)

    Список легенд доступен через свойство :attr:`legends`::

        for legend in legendView.legends:
           print(legend.caption)

    Состав может меняться посредством вызова соответствующих методов свойства :attr:`legends`.

    Добавление легенды для слоя карты::

        legend = Legend(map_view.map.layers[0])
        legend.caption = 'Легенда слоя'
        legendView.legends.append(legend)
        legendView.arrange()

    Доступ к элементу по индексу. Поменяем описание четвертого оформления у первой легенды :class:`axipy.render.Legend` окна::

        legend = legendView.legends[1]
        item = legend.items[3]
        item.title = 'Описание'
        legend.items[3] = item

    Удаление первой легенды из окна::

        legendView.legends.remove(0)


    """

    @classmethod
    def _wrap_typed(cls, _shadow):
        result = cls.__new__(cls)
        result._shadow = _shadow
        result._legends = None
        return result

    @property
    def legends(self) -> ListLegend:
        """Перечень добавленных в окно легенд.
        """
        if self._legends is None:
            self._legends = ListLegend._wrap(self._shadow)
        return self._legends

    def arrange(self):
        """Упорядочивает легенды с целью устранения наложений легенд друг на друга."""
        self._shadow.arrange()
