from enum import Enum
from typing import Optional, Union

from PySide2.QtCore import QEvent, QPointF, QPoint, QRectF, QRect, QObject, Qt, QTimer
from PySide2.QtGui import QMouseEvent, QKeyEvent, QWheelEvent, QPaintEvent, QPainter, QPolygonF, QCursor
from axipy.cpp_gui import (
    InteractionType,
    ShadowGuiUtils,
    ShadowMapTool as ShadowTool,
    ShadowToolFactory as CppToolFactory,
    ShadowViewPointsHelper
)

from axipy._internal._decorator import InitOnce
from axipy.da import Observer
from axipy.utl import Rect, Pnt
from .view import View, MapView

__all__ = [
    "DeactivationReason",
    "MapTool",
]


class DeactivationReason(Enum):
    """
    Причина выключения инструмента.
    """
    Unknown = 0
    """Не определено"""
    ObjectClose = 1
    """Закрытие объекта данных"""
    WindowClose = 2
    """Закрытие окна"""
    ActionClick = 3
    """Нажатие на действие"""
    ActionShortcut = 4
    """Вызов действия комбинацией клавиш"""
    LayerClick = 5
    """Нажатие на свойства слоя"""


class MapTool(ShadowTool):
    """
    Инструмент окна карты.
    При создании своего инструмента новый инструмент наследуется от этого класса,
    и переопределяет необходимые обработчики событий.

    See also:
        :class:`axipy.ObserverManager`.

    Пример::

        MyTool(MapTool):

            def mousePressEvent(self, event):
                print('mouse pressed')
                return self.PassEvent

    """

    PassEvent: bool = False
    """
    Передать событие дальше. Значение :data:`False`.
    """
    BlockEvent: bool = True
    """
    Прекратить обработку события. Значение :data:`True`.
    """
    enable_on: Union[str, Observer] = ''
    """
    Идентификатор наблюдателя для определения доступности инструмента. По умолчанию отсутствует.
    """

    @property
    def view(self) -> MapView:
        """Отображение данных в окне."""
        return View._wrap(self.shadow_view())

    def to_scene(self, device: Union[QPoint, QRect]) -> Union[Pnt, Rect]:
        """
        Переводит точки из координат окна(пикселей) в координаты на карте.

        Args:
            device: Точки в координатах окна.

        Return:
            Точки в координатах карты.
        """
        if type(device) is QPoint:
            device = QPointF(device.x(), device.y())
            return Pnt._point_value_from_qt(self.view.device_to_scene_transform.map(device))
        if type(device) is QRect:
            device = QRectF(device.x(), device.y(), device.width(), device.height())
            return Rect._rect_value_from_qt(self.view.device_to_scene_transform.mapRect(device))
        raise TypeError('Unsuported type.')

    def to_device(self, scene: Union[Pnt, Rect]) -> Union[QPoint, QRect]:
        """
        Переводит точки из координат на карте в координаты окна(пиксели).

        Args:
            scene: Точки в координатах карты.

        Return:
            Точки в координатах окна.
        """
        if type(scene) is Pnt:
            mapped = self.view.scene_to_device_transform.map(Pnt._point_value_to_qt(scene))
            return mapped.toPoint()
        if type(scene) is Rect:
            mapped = self.view.scene_to_device_transform.mapRect(Rect._rect_value_to_qt(scene))
            return mapped.toRect()
        raise TypeError('Unsuported type.')

    def get_select_rect(self, device: QPoint, size: int = 3) -> Rect:
        """
        Возвращает прямоугольник в координатах карты для точки на экране.
        Удобно для использования при поиске объектов.
        
        Args:
            device: Точка в координатах окна.
            size: Размер квадрата в пикселях.
        
        Return:
            Прямоугольник в координатах карты.
        
        Пример::
        
            device_point = event.pos()
            bbox = self.get_select_rect(device_point, 30)
            features = table.items(bbox=bbox)
        """
        device_box = QRect(0, 0, size, size)
        device_box.moveCenter(device)
        return self.to_scene(device_box)

    def redraw(self):
        """
        Перерисовывает окно карты.
        
        Создает событие :class:`PySide2.QtGui.QPaintEvent` и помещает его в
        очередь обработки событий. Аналогично
        :meth:`PySide2.QtWidgets.QWidget.update`.
        """
        self.view.widget.update()

    def is_snapped(self) -> bool:
        """
        Проверяет, сработала ли привязка к элементам карты или отчета
        для текущего положения указателя мыши.

        See also:
            :meth:`snap`, :meth:`snap_device`.
        """
        return self.isSnapped()

    def snap(self, default_value: Pnt = None) -> Optional[Pnt]:
        """
        Возвращает исправленные координаты, если сработала привязка к
        элементам в единицах измерения карты или отчета.
        
        Args:
            default_value: Значение по умолчанию.
        
        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.
        
        Пример::
        
            point = self.to_scene(event.pos())
            current_point = self.snap(point)
        
        See also:
            :meth:`is_snapped`, :meth:`snap_device`, :meth:`to_scene`.
        """
        if not self.is_snapped():
            return default_value
        return Pnt._point_value_from_qt(self.snapPoint())

    def snap_device(self, default_value: QPoint = None) -> Optional[QPoint]:
        """
        Возвращает исправленные координаты, если сработала привязка к
        элементам в единицах измерения окна карты (виджета).
        
        Args:
            default_value: Значение по умолчанию.
        
        Возвращает значение по умолчанию, если не сработала привязка к элементам
        карты или отчета.
        
        Пример::
        
            device_point = event.pos()
            current_device_point = self.snap_device(device_point)

        See also:
            :meth:`is_snapped`, :meth:`snap`.
        """
        if not self.is_snapped():
            return default_value
        return self.to_device(self.snap())

    def handleEvent_impl(self, event: QEvent) -> bool:
        result = self.handleEvent(event)
        if result == self.BlockEvent:
            return self.BlockEvent
        return super().handleEvent_impl(event)

    def handleEvent(self, event: QEvent) -> Optional[bool]:
        """
        Первичный обработчик всех событий инструмента.
        
        Если событие не блокируется этим обработчиком, то оно будет передано
        дальше в соответствующий специализированный обработчик
        :meth:`mousePressEvent`, :meth:`keyReleaseEvent` и прочие в зависимости
        от типа.
        
        Args:
            event: Событие.
        
        Return:
            :attr:`BlockEvent`, чтобы блокировать дальнейшую обработку события.
        """
        pass

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        """Обрабатывает событие отрисовки.

        Args:
            event: Событие отрисовки.
            painter: QPainter для рисования поверх виджета
        """
        pass

    def mousePressEvent_impl(self, event: QEvent) -> bool:
        result = self.mousePressEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mousePressEvent_impl(event)

    def mouseReleaseEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseReleaseEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mouseReleaseEvent_impl(event)

    def mouseDoubleClickEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseDoubleClickEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().mouseDoubleClickEvent_impl(event)

    def mouseMoveEvent_impl(self, event: QEvent) -> bool:
        result = self.mouseMoveEvent(event)
        if result == self.BlockEvent:
            return self.BlockEvent
        return super().mouseMoveEvent_impl(event)

    def keyPressEvent_impl(self, event: QEvent) -> bool:
        result = self.keyPressEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().keyPressEvent_impl(event)

    def keyReleaseEvent_impl(self, event: QEvent) -> bool:
        result = self.keyReleaseEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().keyReleaseEvent_impl(event)

    def wheelEvent_impl(self, event: QEvent) -> bool:
        result = self.wheelEvent(event)
        if result != self.PassEvent:
            return self.BlockEvent
        return super().wheelEvent_impl(event)

    def mousePressEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие нажатия клавиши мыши.

        Args:
            event: Событие нажатия клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseReleaseEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие отпускания клавиши мыши.

        Args:
            event: Событие отпускания клавиши мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseDoubleClickEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие двойного клика мыши.

        Args:
            event: Событие двойного клика мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def mouseMoveEvent(self, event: QMouseEvent) -> Optional[bool]:
        """Обрабатывает событие перемещения мыши.

        Args:
            event: Событие перемещения мыши.

        Return:
            :attr:`PassEvent` или :data:`None`, чтобы пропустить событие дальше по цепочке
            обработчиков. :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def keyPressEvent(self, event: QKeyEvent) -> Optional[bool]:
        """Обрабатывает событие нажатия клавиши клавиатуры.

        Args:
            event: Событие нажатия клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def keyReleaseEvent(self, event: QKeyEvent) -> Optional[bool]:
        """Обрабатывает событие отпускания клавиши клавиатуры.

        Args:
            event: Событие отпускания клавиши клавиатуры.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def wheelEvent(self, event: QWheelEvent) -> Optional[bool]:
        """Обрабатывает событие колеса мыши.

        Args:
            event: Событие колеса мыши.

        Return:
            :attr:`PassEvent`, чтобы пропустить событие дальше по цепочке
            обработчиков. :data:`None` или :attr:`BlockEvent`, чтобы блокировать дальнейшую
            обработку события.
        """
        return self.PassEvent

    def unload(self):
        """Выполняет действия непосредственно перед выключением инструмента и
        перед его удалением.

        Переопределите этот метод, чтобы задать свои действия.

        See also:
            :meth:`load`.
        """
        self.deactivate()

    def deactivate(self):
        """Выполняет действия непосредственно перед выключением инструмента и
        перед его удалением.

        Warning:
            .. deprecated:: 3.6
                Используйте :meth:`unload`.
        """
        pass

    def deactivate_impl(self):
        self.unload()

    def load(self):
        """Выполняет действия непосредственно перед включением инструмента.

        Переопределите этот метод, чтобы задать свои действия.

        See also:
            :meth:`unload`.
        """
        pass

    def onActivate_impl(self):
        self.load()

    def canUnload(self, reason: DeactivationReason) -> bool:
        """Обрабатывает причину выключения инструмента.
        
        Переопределите этот метод, чтобы задать свой обработчик.
        
        Args:
            reason: причина выключения.
        
        Return:
            False чтобы прервать выключение, иначе True.
        """
        return self.canDeactivate(reason)

    def canDeactivate(self, reason: DeactivationReason) -> bool:
        """Обрабатывает причину выключения инструмента.
        
        Переопределите этот метод, чтобы задать свой обработчик.
        
        Args:
            reason: причина выключения.
        
        Return:
            False чтобы прервать выключение, иначе True.

        Warning:
            .. deprecated:: 3.6
                Используйте :meth:`canUnload`.
        """
        return True

    def canDeactivate_impl(self, reason: InteractionType) -> bool:
        result = self.canUnload(DeactivationReason(reason))
        return result if result is not None else True

    @staticmethod
    def reset():
        """Переключает текущий инструмент на инструмент по умолчанию.
        
        Обычно инструментом по умолчанию является Выбор.
        """
        CppToolFactory.requestReset()

    @property
    def cursor(self) -> QCursor:
        """Текущий курсор для данного инструмента.
        
        Первоначально курсор для инструмента можно установить, переопределив метод :meth:`load`::

            class MyTool(MapTool):
                def load(self):
                    self.cursor = QCursor(Qt.SizeAllCursor)

        Если же требуется устанавливать различный типы курсора в зависимости от статуса нажатия ПКМ,
        следует переопределить методы
        :meth:`mousePressEvent` и :meth:`mouseReleaseEvent` и установить нужное значение там::

            class MyTool(MapTool):
                def mousePressEvent(self, event) -> bool:
                    if event.button() == Qt.LeftButton:
                        self.cursor = QCursor(Qt.SizeAllCursor)

        """
        return self.view.widget.cursor()

    @cursor.setter
    def cursor(self, v: QCursor):
        self.view.widget.setCursor(v)


class ViewPointsHelper:
    """
    Дополнительные методы для получения координат точек на карте/отчете.
    """

    def __init__(self, view) -> None:
        self.__view = view
        self.__shadow = ShadowViewPointsHelper(view._shadow)

    def normalize_point(self, point: Union[Pnt, QPoint, QPointF]) -> QPointF:
        """ 
        Нормализует координаты переданной точки в соответствии с установленным ItemView.
        Нормализует точность координат точки с учётом текущего размера окна.  Если 
        точки находятся за границей КС карты то они подтягиваются на границу КС.
        """
        p = point
        supported_types = (QPointF, QPoint, Pnt)
        if not isinstance(point, supported_types):
            text = QObject().tr(f"Передан объект типа {type(point)}, но поддерживаются только: {supported_types}")
            raise TypeError(text)
        if isinstance(point, Pnt):
            p = point.to_qt()
        poly = QPolygonF()
        return self.__shadow.normalize(p, poly)


class SelectToolBehavior(QObject):
    """
    Класс позволяющий выделять объекты на карте аналогично инструменту Выбор.
    """

    def __init__(self, tool: MapTool) -> None:
        super().__init__()
        self.__tool = tool
        self.__start_point = None  # type: Optional[Pnt]
        self.__current_point = None  # type: Optional[Pnt]
        self.__press_timer = QTimer()
        time_in_ms = 250
        self.__press_timer.setInterval(time_in_ms)
        self.__press_timer.timeout.connect(self.__mouseClickProcessing)
        self.__mouse_press_event = None  # type: Optional[QMouseEvent]

    def mousePressEvent(self, event: QMouseEvent) -> bool:
        if event.button() != Qt.LeftButton:
            return False
        self.__invalidate_selection()
        self.__mouse_press_event = QMouseEvent(
            event.type(),
            event.localPos(),
            event.screenPos(),
            event.button(),
            event.buttons(),
            event.modifiers()
        )
        self.__press_timer.start()
        return True

    def mouseMoveEvent(self, event: QMouseEvent) -> bool:
        self.__current_point = self.__tool.to_scene(event.pos())
        self.__redraw()
        return True

    def paintEvent(self, event: QPaintEvent, painter: QPainter):
        if self.__start_point and self.__current_point:
            rect = self.__to_rect()  # type: QRect
            old_pen = painter.pen()
            pen = painter.pen()
            pen.setStyle(Qt.DotLine)
            painter.setPen(pen)
            painter.drawRect(rect)
            painter.setPen(old_pen)

    def mouseReleaseEvent(self, event: QMouseEvent) -> bool:
        if event.button() != Qt.LeftButton:
            return False

        if self.__start_point and self.__current_point:
            # Выделяем всё что в выбранной области
            rect = QRectF(self.__start_point.to_qt(), self.__current_point.to_qt())
            rect = rect.normalized()
            rect = Rect.from_qt(rect)
            SelectToolHelpers.select_by_rect(self.__tool.view, rect, event.modifiers())
        elif self.__mouse_press_event:
            # Выделяем по клику
            SelectToolHelpers.select_by_mouse(self.__tool.view, self.__mouse_press_event)

        self.__invalidate_selection()
        self.__redraw()
        return True

    def keyPressEvent(self, event: QKeyEvent) -> bool:
        if event.key() == Qt.Key_Escape and self.__is_rect_selection():
            self.__invalidate_selection()
            self.__redraw()
            return True
        return False

    def __to_rect(self) -> Optional[QRect]:
        if not (self.__start_point and self.__current_point):
            return None
        p1 = self.__tool.to_device(self.__start_point)
        p2 = self.__tool.to_device(self.__current_point)
        return QRect(p1, p2).normalized()

    def __invalidate_press_timer(self):
        self.__press_timer.stop()

    def __mouseClickProcessing(self):
        event = self.__mouse_press_event
        if event is None:
            return
        self.__start_point = self.__tool.to_scene(event.pos())

    def __redraw(self):
        self.__tool.redraw()

    def __is_rect_selection(self) -> bool:
        return self.__start_point is not None

    def __invalidate_selection(self):
        self.__start_point = None
        self.__current_point = None
        self.__mouse_press_event = None
        self.__invalidate_press_timer()


class SelectToolHelpers:
    """
    Набор вспомогательных функций из инструмента Выбор. 

    Можно использовать при реализации своего инструмента :class:`axipy.gui.MapTool`.
    """

    class _ShadowDesc:

        def __init__(self):
            self._s = None

        def __get__(self, obj, objtype=None):
            if self._s is None:
                self._s = ShadowGuiUtils()
            return self._s

    _shadow: ShadowGuiUtils = _ShadowDesc()

    @classmethod
    def select_by_mouse(cls, view: View, event: QMouseEvent):
        """
        Выделяет геометрию в месте нажатия мышкой.

        .. literalinclude:: /../../tests/doc_examples/test_example_tool_helpers.py
            :caption: Пример использования.
            :pyobject: test_select_by_mouse_tool
            :lines: 2-
            :dedent: 4

        """
        cls._shadow.selectByMouseClick(view._shadow, event)

    @classmethod
    def select_by_rect(cls, view: View, rect: Union[QRectF, Rect], modifiers: Qt.KeyboardModifiers):
        """
        Выделяет все геометрии c одного слоя попавшие в :attr:`rect`.
        """
        if isinstance(rect, Rect):
            rect = rect.to_qt()
        cls._shadow.selectByRect(view._shadow, rect, modifiers)
