from pathlib import Path
from typing import Optional

from PySide2.QtCore import Qt, QFileInfo
from PySide2.QtWidgets import QDialog, QDialogButtonBox, QFileDialog, QMessageBox
from axipy import tr, CurrentSettings, DefaultSettings
from axipy.gui import Notifications

from .converter import convert
from .ui_dialog import Ui_Dialog

title = tr("Конвертер стилей линий")


class Dialog(QDialog):

    def __init__(self, parent=None):
        super().__init__(parent)

        self._ui = Ui_Dialog()
        self._ui.setupUi(self)

        self._pen_file: Optional[Path] = None
        self._out_folder: Optional[Path] = None

        self._selected_filter: Optional[str] = None

        self._ok_button: Optional[QDialogButtonBox.Ok] = None
        self._cancel_button: Optional[QDialogButtonBox.Cancel] = None

        self.init_ui()

    def init_ui(self) -> None:
        self.setWindowTitle(title)
        self.setWindowFlags(self.windowFlags() & ~Qt.WindowContextHelpButtonHint)  # Выключение значка "?"

        def get_existing_last_open_path() -> Path:
            last_open_path_ = CurrentSettings.LastOpenPath
            if not Path(last_open_path_).exists():
                last_open_path_ = DefaultSettings.LastOpenPath
            return last_open_path_

        def on_open_files_clicked():
            try:
                last_open_path_ = get_existing_last_open_path()

                file_name, selected_filter = QFileDialog.getOpenFileName(
                    parent=self,
                    caption="Выбор файла pen",
                    dir=str(last_open_path_),
                    filter="Файлы PEN (*.pen);;Все файлы (*.*)",
                    selectedFilter=self._selected_filter
                )
                if not file_name:
                    return None

                self._pen_file = Path(file_name)
                self._ui.le_pen_path.setText(file_name)

                self._selected_filter = selected_filter
                CurrentSettings.LastOpenPath = QFileInfo(file_name).absolutePath()

                self.check_enabled_ok()
            except Exception as e:
                Notifications.push(title, f"Ошибка открытия файла: {e}", Notifications.Critical)

        self._ui.pb_choose_pen_path.clicked.connect(on_open_files_clicked)

        def get_existing_last_save_path():
            last_save_path_ = CurrentSettings.LastSavePath
            if not Path(last_save_path_).exists():
                last_save_path_ = DefaultSettings.LastSavePath
            return last_save_path_

        last_save_path = get_existing_last_save_path()
        self._ui.le_trtp_path.setText(str(last_save_path))
        self._out_folder = Path(last_save_path)

        def choose_out_folder():
            last_save_path_ = get_existing_last_save_path()
            out_folder_ = QFileDialog.getExistingDirectory(
                self,
                caption=tr("Выбор выходной папки"),
                dir=str(last_save_path_),
                options=QFileDialog.ShowDirsOnly,
            )

            if out_folder_:
                self._ui.le_trtp_path.setText(out_folder_)
                self._out_folder = Path(out_folder_)
                CurrentSettings.LastSavePath = out_folder_

                self.check_enabled_ok()

        self._ui.pb_choose_trtp_path.clicked.connect(choose_out_folder)

        self._ok_button = self._ui.buttonBox.button(QDialogButtonBox.Ok)
        self._ok_button.setText(tr("Конвертировать"))
        self._ok_button.setEnabled(False)
        self._cancel_button = self._ui.buttonBox.button(QDialogButtonBox.Cancel)
        self._cancel_button.setText(tr("Закрыть"))

    def check_enabled_ok(self):
        if self._pen_file is not None and self._pen_file.exists() and self._out_folder.exists():
            enabled = True
        else:
            enabled = False

        self._ok_button.setEnabled(enabled)

    def accept(self):
        self.convert()

    def convert(self):
        out_folder = (self._out_folder / self._pen_file.stem)
        try:
            out_folder.mkdir(exist_ok=False)
        except FileExistsError:
            mbox = QMessageBox(
                QMessageBox.Critical,  # icon
                tr("Ошибка"),  # title
                tr(f"Папка {out_folder} уже существует, пожалуйста, выберите другую выходную папку"),  # text
                parent=self,
            )
            mbox.open()
            return None

        try:
            convert(self._pen_file, out_folder)
        except Exception as e:
            mbox = QMessageBox(
                QMessageBox.Critical,  # icon
                tr("Ошибка"),  # title
                tr(f"В процессе конвертации произошла ошибка: {str(e)}"),  # text
                parent=self,
            )
            mbox.open()
            return None
        else:
            Notifications.push(
                title,
                tr(f"Конвертация завершена. Папка с файлами TRTP: {Path(*out_folder.parts[-2:])}"),
                Notifications.Success
            )
